#set( $symbol_pound = '#' )
#set( $symbol_dollar = '$' )
#set( $symbol_escape = '\' )
package ${package}.upms.service;

import cn.hutool.http.HttpUtil;
import cn.hutool.json.JSONObject;
import cn.hutool.json.JSONUtil;
import ${package}.config.AppProperties;
import ${package}.exception.BizException;
import ${package}.security.ArkUser;
import ${package}.security.SecurityUtils;
import ${package}.service.dto.*;
import ${package}.upms.entity.UpmsAdminRoleBind;
import ${package}.upms.entity.UpmsUser;
import ${package}.upms.mapper.UpmsAdminRoleBindMapper;
import ${package}.upms.mapper.UpmsUserMapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import ${package}.web.rest.AdminUserApiDelegate;
import ${package}.web.rest.AuthApiDelegate;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.config.annotation.authentication.builders.AuthenticationManagerBuilder;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.security.oauth2.jwt.JwsHeader;
import org.springframework.security.oauth2.jwt.JwtClaimsSet;
import org.springframework.security.oauth2.jwt.JwtEncoder;
import org.springframework.security.oauth2.jwt.JwtEncoderParameters;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.context.request.NativeWebRequest;

import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import static ${package}.security.SecurityUtils.*;
import static ${package}.security.SecurityUtils.JWT_ALGORITHM;

/**
 * 用户服务
 */
@Slf4j
@Service
@RequiredArgsConstructor
public class UserService implements AuthApiDelegate, AdminUserApiDelegate {
    private final UpmsUserMapper userMapper;
    private final PasswordEncoder passwordEncoder;
    private final UpmsAdminRoleBindMapper adminRoleBindMapper;
    private final JwtEncoder jwtEncoder;
    private final AuthenticationManagerBuilder authenticationManagerBuilder;
    private final AppProperties appProperties;

    @Override
    public Optional<NativeWebRequest> getRequest() {
        return AuthApiDelegate.super.getRequest();
    }

    @Override
    public LoginData authorize(LoginRequest loginRequest) {
        if (appProperties.isEnableCfToken()) {
            verifyCloudFlareToken(loginRequest.getToken());
        }
        UsernamePasswordAuthenticationToken authenticationToken = new UsernamePasswordAuthenticationToken(
                loginRequest.getUsername(),
                loginRequest.getPassword()
        );

        Authentication authentication = authenticationManagerBuilder.getObject().authenticate(authenticationToken);
        SecurityContextHolder.getContext().setAuthentication(authentication);
        return createLoginData(authentication);
    }

    private void verifyCloudFlareToken(String token) {
        String body = HttpUtil.post("https://api.hcaptcha.com/siteverify",
                "secret=ES_b0ae86dddbe34639a802bd0589083271&response=" + token);
        JSONObject response = JSONUtil.parseObj(body);
        if (!response.getBool("success")) {
            throw new BizException("人机验证失败");
        }
    }

    private LoginData createLoginData(Authentication authentication) {
        Instant expiresTime = getExpiresTime(false);
        String accessToken = this.createToken(authentication, expiresTime);
        Instant refreshExpiresTime = getExpiresTime(true);
        String refreshToken = this.createToken(authentication, refreshExpiresTime);
        return new LoginData(accessToken, refreshToken, LocalDateTime.ofInstant(expiresTime, ZoneId.systemDefault()));
    }

    private Instant getExpiresTime(boolean rememberMe) {
        Instant now = Instant.now();
        Instant validity;
        if (rememberMe) {
            validity = now.plus(appProperties.getSecurity().getAuthentication()
                    .getJwt().getTokenValidityInSecondsForRememberMe(), ChronoUnit.SECONDS);
        } else {
            validity = now.plus(appProperties.getSecurity()
                    .getAuthentication().getJwt().getTokenValidityInSeconds(), ChronoUnit.SECONDS);
        }
        return validity;
    }

    private String createToken(Authentication authentication, Instant validity) {
        String authorities = authentication.getAuthorities()
                .stream()
                .map(GrantedAuthority::getAuthority)
                .collect(Collectors.joining(" "));

        Instant now = Instant.now();

        // @formatter:off
        JwtClaimsSet claims = JwtClaimsSet.builder()
                .issuedAt(now)
                .expiresAt(validity)
                .subject(authentication.getName())
                .claim(AUTHORITIES_KEY, authorities)
                .claim(USERID_KEY, ((ArkUser)authentication.getPrincipal()).getUserId())
                .claim(USER_TYPE_KEY, ((ArkUser)authentication.getPrincipal()).getUserType())
                .build();

        JwsHeader jwsHeader = JwsHeader.with(JWT_ALGORITHM).build();
        return this.jwtEncoder.encode(JwtEncoderParameters.from(jwsHeader, claims)).getTokenValue();
    }
    @Override
    public LoginData refreshToken(String authorization) {
        Authentication authentication = SecurityUtils.getCurrentAuthentication()
                .orElseThrow(() -> new BizException("未登录"));
        return createLoginData(authentication);
    }

    @Override
    public UserInfoData userInfo() {
        Authentication authentication = SecurityContextHolder.getContext().getAuthentication();
        String name = authentication.getName();
        return getUserByUsername(name)
                .map(upmsUser -> {
                    UserInfoData user = new UserInfoData(upmsUser.getAvatar(), upmsUser.getUsername(), upmsUser.getType());
                    user.setNickname(upmsUser.getNickname());
                    user.setPermissions(authentication.getAuthorities().stream().map(GrantedAuthority::getAuthority).toList());
                    user.setExpireTime(upmsUser.getExpireTime());
                    return user;
                })
                .orElseThrow(() -> new BizException("用户不存在"));
    }

    public Optional<UpmsUser> getUserByUsername(String username) {
        return userMapper.selectByUserName(username);
    }

    @Transactional
    public CreateAdminUserData createAdminUser(CreateAdminUserRequest createAdminUserRequest) {
        if (getUserByUsername(createAdminUserRequest.getUsername()).isPresent()) {
            throw new BizException("该用户名已存在");
        }
        UpmsUser upmsUser = new UpmsUser();
        upmsUser.setUsername(createAdminUserRequest.getUsername());
        String encryptedPassword = passwordEncoder.encode(StringUtils.defaultIfBlank(createAdminUserRequest.getPassword(), "123456"));
        upmsUser.setPassword(encryptedPassword);
        upmsUser.setNickname(createAdminUserRequest.getNickname());
        upmsUser.setAvatar(createAdminUserRequest.getAvatar());
        if (createAdminUserRequest.getIsActive()) {
            upmsUser.active();
        } else {
            upmsUser.deactivate();
        }
        if (createAdminUserRequest.getIsSuperAdmin()) {
            upmsUser.setSuperAdmin();
        } else {
            upmsUser.setAdmin();
        }
        upmsUser.setCreateTime(LocalDateTime.now());
        upmsUser.setCreator(SecurityUtils.getCurrentUserLogin().orElse("system"));
        userMapper.insert(upmsUser);
        if (!createAdminUserRequest.getIsSuperAdmin()) {
            bindRole(createAdminUserRequest.getRoles(), upmsUser);
        }
        return new CreateAdminUserData(upmsUser.getId());
    }

    private void bindRole(List<Long> roleIds, UpmsUser upmsUser) {
        if (roleIds != null) {
            roleIds.stream().map(r ->{
                        var rela = new UpmsAdminRoleBind();
                        rela.setAdminId(upmsUser.getId());
                        rela.setRoleId(r);
                        return rela;
                    })
                    .forEach(adminRoleBindMapper::insert);
        }
    }

    public ReadAdminUserData readAdminUser(ReadAdminUserRequest readAdminUserRequest) {
        UpmsUser upmsUser = userMapper.selectById(readAdminUserRequest.getId());
        if (upmsUser == null) {
            throw new BizException("该用户不存在");
        }
        ReadAdminUserData readAdminUserData = new ReadAdminUserData();
        readAdminUserData.setIsActive(upmsUser.isActivated());
        readAdminUserData.setIsSuperAdmin(upmsUser.isSuperAdmin());
        readAdminUserData.setId(upmsUser.getId());
        readAdminUserData.setUsername(upmsUser.getUsername());
        readAdminUserData.setNickname(upmsUser.getNickname());
        readAdminUserData.setAvatar(upmsUser.getAvatar());
        readAdminUserData.setCreateTime(upmsUser.getCreateTime());
        readAdminUserData.setCreator(upmsUser.getCreator());
        readAdminUserData.setUpdateTime(upmsUser.getUpdateTime());
        readAdminUserData.setUpdater(upmsUser.getUpdater());
        readAdminUserData.setRoles(adminRoleBindMapper.selectRoleIdsByAdminId(upmsUser.getId()));
        return readAdminUserData;
    }

    public QueryAdminUserData queryAdminUser(QueryAdminUserRequest req) {
        Page<QueryAdminUserDataElement> page = Page.of(req.getCurrentPage(), req.getSize());
        List<QueryAdminUserDataElement> records = userMapper.queryAdminUser(page, req);
        return new QueryAdminUserData(
                page.getTotal(),
                page.getPages(),
                page.getSize(),
                page.getCurrent(),
                records
        );
    }

    @Transactional
    public UpdateAdminUserData updateAdminUser(UpdateAdminUserRequest updateAdminUserRequest) {
        UpmsUser upmsUser = userMapper.selectById(updateAdminUserRequest.getId());
        if (upmsUser == null) {
            throw new BizException("该用户不存在");
        }
        if (updateAdminUserRequest.getIsActive()) {
            upmsUser.active();
        } else {
            upmsUser.deactivate();
        }
        if (updateAdminUserRequest.getIsSuperAdmin()) {
            upmsUser.setSuperAdmin();
        } else {
            upmsUser.setAdmin();
        }
        if (StringUtils.isNotBlank(updateAdminUserRequest.getPassword())) {
            String encryptedPassword = passwordEncoder.encode(updateAdminUserRequest.getPassword());
            upmsUser.setPassword(encryptedPassword);
        }
        upmsUser.setNickname(updateAdminUserRequest.getNickname());
        upmsUser.setAvatar(updateAdminUserRequest.getAvatar());
        if (updateAdminUserRequest.getRoles() != null) {
            adminRoleBindMapper.deleteByAdminId(upmsUser.getId());
            bindRole(updateAdminUserRequest.getRoles(), upmsUser);
        }
        upmsUser.setUpdateTime(LocalDateTime.now());
        upmsUser.setUpdater(SecurityUtils.getCurrentUserLogin().orElse("system"));
        userMapper.updateById(upmsUser);
        return new UpdateAdminUserData(upmsUser.getId());
    }

    @Transactional
    public void deleteAdminUser(DeleteAdminUserRequest deleteAdminUserRequest) {
        adminRoleBindMapper.deleteByAdminId(deleteAdminUserRequest.getId());
        userMapper.deleteById(deleteAdminUserRequest.getId());
    }

}
