import { defineStore } from "pinia";
import {
  type userType,
  store,
  router,
  resetRouter,
  routerArrays,
  storageLocal
} from "../utils";
import { useMultiTagsStoreHook } from "./multiTags";
import { type DataInfo, setToken, removeToken, userKey, userInfoKey } from "@/utils/auth";
import {shApi} from "@/api/Api";
import { LoginRequest, LoginResponse, UserInfoData } from "@/api/generated/api";
import dayjs from "dayjs";

export const useUserStore = defineStore({
  id: "pure-user",
  state: (): userType => ({
    // 头像
    avatar: storageLocal().getItem<DataInfo<number>>(userKey)?.avatar ?? "",
    // 用户名
    username: storageLocal().getItem<DataInfo<number>>(userKey)?.username ?? "",
    // 昵称
    nickname: storageLocal().getItem<DataInfo<number>>(userKey)?.nickname ?? "",
    // 页面级别权限
    permissions: storageLocal().getItem<DataInfo<number>>(userKey)?.permissions ?? [],
    // 用户类型
    type: storageLocal().getItem<DataInfo<number>>(userKey)?.type ?? 2,
    // 用户到期时间
    expireTime: storageLocal().getItem<UserInfoData>(userInfoKey)?.expireTime,
    // 是否勾选了登录页的免登录
    isRemembered: false,
    // 登录页的免登录存储几天，默认7天
    loginDay: 7
  }),
  actions: {
    setExpireTime(expireTime: string) {
      this.expireTime = expireTime;
    },
    /** 存储头像 */
    SET_AVATAR(avatar: string) {
      this.avatar = avatar;
    },
    /** 存储用户名 */
    SET_USERNAME(username: string) {
      this.username = username;
    },
    /** 存储昵称 */
    SET_NICKNAME(nickname: string) {
      this.nickname = nickname;
    },
    /** 存储用户类型 */
    SET_TYPE(userType: number) {
      this.type = userType;
    },
    /** 存储权限 */
    SET_PERMISSIONS(permissions: Array<string>) {
      this.permissions = permissions;
    },
    /** 存储是否勾选了登录页的免登录 */
    SET_ISREMEMBERED(bool: boolean) {
      this.isRemembered = bool;
    },
    /** 设置登录页的免登录存储几天 */
    SET_LOGINDAY(value: number) {
      this.loginDay = Number(value);
    },
    /** 登入 */
    async loginByUsername(data: LoginRequest) {
      return new Promise<LoginResponse>((resolve, reject) => {
        shApi.authenticate.authorize(data)
          .then(data => {
            if (data?.code == 0){
              setToken({
                accessToken: data.data.accessToken,
                refreshToken: data.data.refreshToken,
                expires: dayjs(data.data.expires, "YYYY-MM-DD HH:mm:ss").toDate()
              })
            }
            resolve(data);
          })
          .catch(error => {
            reject(error);
          });
      });
    },
    /** 前端登出（不调用接口） */
    logOut() {
      this.username = "";
      this.permissions = [];
      removeToken();
      useMultiTagsStoreHook().handleTags("equal", [...routerArrays]);
      resetRouter();
      router.push("/login");
    },
    /** 刷新`token` */
    async handRefreshToken(data) {
      return new Promise<LoginResponse>((resolve, reject) => {
        shApi.authenticate.refreshToken(data.refreshToken)
          .then(data => {
            if (data) {
              setToken({
                accessToken: data.data.accessToken,
                refreshToken: data.data.refreshToken,
                expires: dayjs(data.data.expires, "YYYY-MM-DD HH:mm:ss").toDate()
              });
              resolve(data);
            }
          })
          .catch(error => {
            reject(error);
          });
      });
    }
  }
});

export function useUserStoreHook() {
  return useUserStore(store);
}
