#set( $symbol_pound = '#' )
#set( $symbol_dollar = '$' )
#set( $symbol_escape = '\' )
package ${package}.queue;

import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

/**
 * 延迟队列，队列中的元素关联一个延迟时间，在延迟时间后，该
 * 元素会自动弹出并调动监听函数。
 *
 * @author yujiaxin
 */
public interface DelayedQueue<T> {

    /**
     * 向队列中添加一个元素，该元素在指定延迟时间后自动弹出，
     * 并调用已注册的所有回调函数。这些回调函数是顺序执行的。
     *
     * @param ele      元素
     * @param delay    延迟时间
     * @param timeUnit 时间单位
     * @see ${symbol_pound}register(Consumer)
     * @see ${symbol_pound}offer(Object, long)
     */
    void offer(T ele, long delay, TimeUnit timeUnit);

    /**
     * 向队列中添加一个元素，该元素在指定延迟指定秒后自动弹出，
     * 并调用已注册的所有回调函数。这些回调函数是顺序执行的。
     *
     * @param ele          元素
     * @param delaySeconds 延迟秒数
     * @see ${symbol_pound}register(Consumer)
     * @see ${symbol_pound}offer(Object, long, TimeUnit)
     */
    void offer(T ele, long delaySeconds);

    /**
     * 向队列中追加一个回调函数，当队列中有元素被弹出时，
     * 会自动调用该函数。
     *
     * @param action 回调函数
     */
    void register(Consumer<T> action);

    /**
     * 主动触发符合条件的元素消费。
     * 这个方法通常在应用运行中断，应用重启之后调用，
     * 用于保证应用重启的这段时间内，延迟
     * 队列中弹出的元素在应用重启之后
     * 被消费。
     */
    void consume();

    /**
     * 提前移除队列中的元素
     *
     * @param ele 要移除的元素
     * @return 如果元素之前存在在队列中，调用此方法后被删除，返回true
     */
    boolean remove(T ele);
}
