import dayjs from "dayjs";
import editForm from "../form.vue";
import { message } from "@/utils/message";
import { ElMessageBox } from "element-plus";
import { usePublicHooks } from "../../hooks";
import { addDialog } from "@/components/ReDialog";
import type { FormItemProps } from "../utils/types";
import type { PaginationProps } from "@pureadmin/table";
import { getKeyList, deviceDetection } from "@pureadmin/utils";

import { shApi } from "@/api/Api";

import { type Ref, reactive, ref, onMounted, h, toRaw, watch } from "vue";

export function useUser() {
  const form = reactive({
    username: "",
    isSuperAdmin: false,
  });
  const curRow = ref();
  const formRef = ref();
  const dataList = ref([]);
  const loading = ref(true);
  const isLinkage = ref(true);
  const switchLoadMap = ref({});
  const { switchStyle } = usePublicHooks();
  const pagination = reactive<PaginationProps>({
    total: 0,
    pageSize: 10,
    currentPage: 1,
    background: true
  });
  const columns: TableColumnList = [
    {
      label: "用户编号",
      prop: "id"
    },
    {
      label: "用户名",
      prop: "username"
    },
    {
      label: "用户昵称",
      prop: "nickname"
    },
    {
      label: "角色",
      prop: "roles"
    },
    {
      label: "是否超管",
      prop: "isSuperAdmin",
      cellRenderer: ({ row }) => {
        return (
          <el-tag type={row.isSuperAdmin ? "success" : "info"}>{row.isSuperAdmin ? "是" : "否"}</el-tag>
        )
      },
    },
    {
      label: "是否启用",
      prop: "isActive",
      cellRenderer: ({ row }) => {
        return (
          <el-tag type={row.isActive ? "success" : "info"}>{row.isActive ? "是" : "否"}</el-tag>
        )
      },
    },
    {
      label: "创建时间",
      prop: "createTime",
      minWidth: 160,
      formatter: ({ createTime }) =>
        dayjs(createTime).format("YYYY-MM-DD HH:mm:ss")
    },
    {
      label: "更新时间",
      prop: "updateTime",
      minWidth: 160,
      formatter: ({ updateTime }) => {
        if(updateTime){
          return dayjs(updateTime).format("YYYY-MM-DD HH:mm:ss")
        }
        return "";
      }
    },
    {
      label: "创建人",
      prop: "creator"
    },
    {
      label: "更新人",
      prop: "updater"
    },
    {
      label: "操作",
      fixed: "right",
      width: 210,
      slot: "operation"
    }
  ];

  function handleDelete(row) {
    shApi.adminUser.deleteAdminUser({
      id: row.id
    }).then(() => {
      message(`您删除了管理员: ${row.username}`, { type: "success" });
      onSearch();
    })
  }

  function handleSizeChange(val: number) {
    pagination.currentPage = 1
    onSearch()
  }

  function handleCurrentChange(val: number) {
    onSearch()
  }

  function handleSelectionChange(val) {
    console.log("handleSelectionChange", val);
  }

  async function onSearch() {
    loading.value = true;
    const formData = toRaw(form);
    const { data } = await shApi.adminUser.queryAdminUser({
      currentPage: pagination.currentPage,
      size: pagination.pageSize,
      ...formData
    });
    dataList.value = data.elements;
    pagination.total = data.totalElements;
    pagination.currentPage = data.currentPage;

    setTimeout(() => {
      loading.value = false;
    }, 500);
  }

  const resetForm = formEl => {
    if (!formEl) return;
    formEl.resetFields();
    onSearch();
  };

  const roleOptions = ref([])

  const fetchRoleOptions = () => {
    shApi.role.queryRole({
      currentPage: 1,
      size: 1000
    }).then(res => {
      roleOptions.value = res.data.elements;
    })
  }

  function openDialog(title = "新增", row?: FormItemProps) {
    addDialog({
      title: `${title}管理员`,
      props: {
        formInline: {
          username: row?.username ?? "",
          nickname: row?.nickname ?? "",
          avatar: row?.avatar ?? "",
          isSuperAdmin: row?.isSuperAdmin ?? false,
          isActive: row?.isActive ?? false,
          roles: row?.roles ?? [],
        },
        roleOptions: roleOptions.value,
        isUpdate: title !== "新增"
      },
      width: "40%",
      draggable: true,
      fullscreen: deviceDetection(),
      fullscreenIcon: true,
      closeOnClickModal: false,
      contentRenderer: () => h(editForm, { ref: formRef }),
      beforeSure: (done, { options }) => {
        const FormRef = formRef.value.getRef();
        const curData = options.props.formInline as FormItemProps;
        function chores() {
          message(`您${title}了管理员: ${curData.username}`, {
            type: "success"
          });
          done(); // 关闭弹框
          onSearch(); // 刷新表格数据
        }
        FormRef.validate(valid => {
          if (valid) {
            if (title === "新增") {
              shApi.adminUser.createAdminUser({
                username: curData.username,
                password: curData.password,
                nickname: curData.nickname,
                roles: curData.roles,
                avatar: curData.avatar,
                isSuperAdmin: curData.isSuperAdmin,
                isActive: curData.isActive
              }).then(res => {
                chores();
              })
            } else {
              shApi.adminUser.updateAdminUser({
                id: row.id,
                password: curData.password,
                nickname: curData.nickname,
                roles: curData.roles,
                avatar: curData.avatar,
                isSuperAdmin: curData.isSuperAdmin,
                isActive: curData.isActive
              }).then(res => {
                chores();
              })
            }
          }
        });
      }
    });
  }

  /** 高亮当前权限选中行 */
  function rowStyle({ row: { id } }) {
    return {
      cursor: "pointer",
      background: id === curRow.value?.id ? "var(--el-fill-color-light)" : ""
    };
  }

  onMounted(async () => {
    onSearch();
    fetchRoleOptions();
  });

  return {
    form,
    curRow,
    loading,
    columns,
    rowStyle,
    dataList,
    isLinkage,
    pagination,
    // buttonClass,
    onSearch,
    resetForm,
    openDialog,
    handleDelete,
    // handleDatabase,
    handleSizeChange,
    handleCurrentChange,
    handleSelectionChange
  };
}
