package cn.miw.gen.utils;

import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.zip.ZipInputStream;

import javax.sql.DataSource;

import com.jfinal.kit.StrKit;
import com.jfinal.template.Engine;
import com.jfinal.template.Template;
import com.jfinal.template.ext.directive.NowDirective;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.io.FileUtil;
import cn.hutool.core.io.resource.NoResourceException;
import cn.hutool.core.io.resource.ResourceUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.core.util.ZipUtil;
import cn.hutool.db.meta.Column;
import cn.hutool.db.meta.MetaUtil;
import cn.hutool.db.meta.Table;
import cn.hutool.json.JSONArray;
import cn.hutool.json.JSONObject;
import cn.hutool.json.JSONUtil;

public class GenUtil {
	private DataSource ds;

	public GenUtil(DataSource ds) {
		super();
		this.ds = ds;
	}

	/**
	 * 生成种类文件
	 * 
	 * @param pkg      包名
	 * @param business 业务模块名(可以为空，如果有尽量使用一个单词，不要有空格和特殊符号)
	 */
	public void gen(String pkg, String business) {
		List<String> tables = ds != null ? MetaUtil.getTables(ds) : null;
		gen(pkg, business, tables);
	}

	/**
	 * 生成种类文件
	 * 
	 * @param pkg      包名
	 * @param business 业务模块名(可以为空，如果有尽量使用一个单词，不要有空格和特殊符号)
	 * @param tables   数据库中的表名列表(针对使用程序选择表的使用方式，但必须要有数据源)
	 */
	public void gen(String pkg, String business, List<String> tables) {
		gen(pkg, business, tables, "/src/main/java", "/src/main/resources/templates", true);
	}

	/**
	 * 生成种类文件
	 * 
	 * @param pkg      包名
	 * @param business 业务模块名(可以为空，如果有尽量使用一个单词，不要有空格和特殊符号)
	 * @param tables   数据库中的表名列表(针对使用程序选择表的使用方式，但必须要有数据源)
	 * @param javaPath 指定java文件的保存目录(值样例：/src/main/java)
	 * @param tempPath 指定页面文件保存目录(值样例：/src/main/resources/templates)
	 * @param override 是否覆盖生成
	 */
	private void gen(String pkg, String business, List<String> tables, String javaPath, String tempPath, boolean override) {
		JSONObject data = new JSONObject().set("package", pkg);
		business = (StrUtil.isNotBlank(business) && !business.startsWith(".")) ? "." + business : business;
		String staticDir = processStatic(tempPath);
		Set<JSONObject> menus = new HashSet<>();
		if (tables != null && ds != null)
			for (String tableName : tables) {
				Table tableMeta = MetaUtil.getTableMeta(ds, tableName);
				String comment = tableMeta.getComment();
				comment = StrKit.isBlank(comment)?tableName:comment;
				String fix=StrKit.notBlank(business)?business.substring(1)+"/":"";
				String entityName= SimpleUtil.genName(tableName,0);
				menus.add(new JSONObject().set("title",comment).set("url", "/admin/"+fix+entityName));
				data.set("tableName", tableName).set("business", business).set("tableComment", comment);
				Set<String> pkNames = tableMeta.getPkNames();
				String pk = CollUtil.isNotEmpty(pkNames) ? pkNames.stream().findFirst().get() : null;
				Column pkCol = null;
				Collection<Column> fields = tableMeta.getColumns();
				if (pk != null) {
					pkCol = fields.stream().filter(x -> x.getName().equals(pk)).findFirst().get();
				} else
					pkCol = fields.stream().findFirst().get();
				data.set("fields", fields).set("pk", pkCol);

				String path, target;
				path = buildDir(javaPath, pkg, business, "model");
				target = path + File.separator + SimpleUtil.genName(tableName, 0) + ".java";
				File file = new File(target);
				if( !file.exists() || (file.exists() && override) ) genFile("/tpl/model.tpl", data, file);

				path = buildDir(javaPath, pkg, business, "mapper");
				target = path + File.separator + SimpleUtil.genName(tableName, 0) + "Mapper.java";
				file = new File(target);
				if( !file.exists() || (file.exists() && override) ) genFile("/tpl/Mapper.tpl", data, file);

				path = buildDir(javaPath, pkg, business, "service");
				target = path + File.separator + SimpleUtil.genName(tableName, 0) + "Service.java";
				file = new File(target);
				if( !file.exists() || (file.exists() && override) ) genFile("/tpl/Service.tpl", data, file);

				path = buildDir(javaPath, pkg, business, "service.impl");
				target = path + File.separator + SimpleUtil.genName(tableName, 0) + "ServiceImpl.java";
				file = new File(target);
				if( !file.exists() || (file.exists() && override) ) genFile("/tpl/Impl.tpl", data, file);

				path = buildDir(javaPath, pkg, business, "controller.admin");
				target = path + File.separator + "Admin" + SimpleUtil.genName(tableName, 0) + ".java";
				file = new File(target);
				if( !file.exists() || (file.exists() && override) ) genFile("/tpl/controller.tpl", data, file);

				path = buildDir(tempPath, "/admin", business, SimpleUtil.genName(tableName, 0));
				target = path + File.separator + "list.html";
				file = new File(target);
				if( !file.exists() || (file.exists() && override) ) genFile("/tpl/list.tpl", data, file);

				path = buildDir(tempPath, "/admin", business, SimpleUtil.genName(tableName, 0));
				target = path + File.separator + "edit.html";
				file = new File(target);
				if( !file.exists() || (file.exists() && override) ) genFile("/tpl/edit.tpl", data, file);
			}
		String bussMenuName = StrKit.isBlank(business)?"系统业务":business.substring(1);
		processMenu(staticDir,bussMenuName,menus);
		
		processConfig(pkg, javaPath);
		processDefault(tempPath);
		processPOM(javaPath);

	}

	private void processMenu(String staticDir, String buess, Set<JSONObject> menus) {
		String path = staticDir + File.separator + "json/menu.json";
		File menuFile = new File(path);
		JSONObject oldMenu;
		if (menuFile.exists()) {
			oldMenu = JSONUtil.readJSONObject(menuFile, Charset.forName("utf-8"));
		} else
			oldMenu = new JSONObject();

		JSONArray array = oldMenu.getJSONArray(buess);
		if (array == null) {
			array = new JSONArray();
		}
		array.addAll(menus);
		oldMenu.set(buess, array);
		FileUtil.writeUtf8String(oldMenu.toJSONString(0), menuFile);
	}

	private void processPOM(String javaPath) {
//		File parent = new File(new File("").getAbsolutePath() + javaPath).getParentFile();
//		String pomxml = parent.getParentFile().getParent()+File.separator+"pom.xml";
//		System.out.println(pomxml);
//		Document pom = XmlUtil.readXML(new File(pomxml));
//		NodeList byXPath = (NodeList) XmlUtil.getByXPath("//project/dependencies", pom,  XPathConstants.NODESET);
//		for(int i=0;i<byXPath.getLength();i++) {
//			System.out.println(byXPath.item(i));
//		}
	}

	private String processStatic(String tempPath) {
		File parent = new File(new File("").getAbsolutePath() + tempPath).getParentFile();

		try (ZipInputStream zipInputStream = new ZipInputStream(ResourceUtil.getStream("static.zip"))) {
			File unzip = ZipUtil.unzip(zipInputStream, parent);
			return unzip.getAbsolutePath() + File.separator + "static";
		} catch (NoResourceException | IOException e) {
			e.printStackTrace();
		}
		return "";
	}

	private void processDefault(String tempPath) {
		JSONObject data = new JSONObject();
		String path, target;
		path = buildDir(tempPath, "/admin", "", "");
		target = path + File.separator + "_temp.html";
		File file = new File(target);
		if (!file.exists())
			genFile("/tpl/default/_temp.html.tpl", data, file);
		target = path + File.separator + "index.html";
		file = new File(target);
		if (!file.exists())
			genFile("/tpl/default/index.html.tpl", data, file);
		target = path + File.separator + "welcome.html";
		file = new File(target);
		if (!file.exists())
			genFile("/tpl/default/welcome.html.tpl", data, file);
	}

	private void processConfig(String pkg, String javaPath) {
		JSONObject data = new JSONObject().set("package", pkg);
		String path, target;
		File file;
		path = buildDir(javaPath, pkg, "", "controller.admin");
		target = path + File.separator + "Index.java";
		file = new File(target);
		if (!file.exists())
			genFile("/tpl/default/Index.java.tpl", data, file);

		path = buildDir(javaPath, pkg, "", "config");
		target = path + File.separator + "MybatisPlusConfig.java";
		file = new File(target);
		if (!file.exists())
			genFile("/tpl/config/MybatisPlusConfig.java.tpl", data, file);
//
//		target = path + File.separator + "JSONAop.java";
//		file = new File(target);
//		if (!file.exists())
//			genFile("/tpl/config/JSONAop.java.tpl", data, file);

		target = path + File.separator + "ViewConfig.java";
		file = new File(target);
		if (!file.exists())
			genFile("/tpl/config/ViewConfig.java.tpl", data, file);

		target = path + File.separator + "AppConfig.java";
		file = new File(target);
		if (!file.exists())
			genFile("/tpl/config/AppConfig.java.tpl", data, file);
	}

	private String buildDir(String where, String pkg, String business, String t) {
		String x = where + File.separator + (StrUtil.isEmptyIfStr(pkg) ? "" : pkg + File.separator)
				+ (StrUtil.isEmptyIfStr(business) ? "" : business + File.separator) + t;
		String path = new File("").getAbsolutePath() + x.replace(".", File.separator);
		
		File file = new File(path);
		if (!file.exists() || file.isFile()) {
			file.mkdirs();
		}
		return path;
	}

	private void genFile(String tpl, Map<?, ?> data, File target) {
		Engine engine = Engine.use("genUtil");
		if (engine == null) {
			engine = Engine.create("genUtil");
			engine.setDevMode(true).setToClassPathSourceFactory();
			engine.addSharedMethod(new StrUtil());
			engine.addSharedMethod(new StrKit());
			engine.addSharedMethod(new CollUtil());
			engine.addSharedMethod(new SimpleUtil());
			engine.addDirective("now", NowDirective.class);
		}
		Template temp = engine.getTemplate(tpl);
		System.out.println(target.getAbsolutePath());
		temp.render(data, target);
	}

}
