package cn.net.vidyo.framework.data.jdbc.dao;

import cn.net.vidyo.framework.data.jdbc.utils.EntityUtils;
import org.springframework.jdbc.core.BeanPropertyRowMapper;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 *
 * 添加
 * 单个添加
 * 多个添加
 *
 * 修改
 * 根据条件修改某些字段
 * 根据条件修改实体
 *
 * 删除
 * 根据条件删除
 *
 * 查询
 *
 * 根据条件查询查询实体列表
 * 根据条件查询字段map列表
 * 根据条件查询一个实体
 * 根据条件查询字段map
 *
 *
 * 执行SQL
 *
 *
 *
 */
public class EntityDao<T> extends JdbcDao {
    Class<T> entityClass;

    JdbcDao jdbcDao;

    public EntityDao(Class<T> entityClass) {
        this.entityClass = entityClass;
    }

    public int insert(T entity) {
        Map<String, Object> fieldMap = EntityUtils.getFieldMap(entity);
        return insert(fieldMap,entity.getClass());
    }
    public int insert(Map<String, Object> fieldMap,Class<?> entityClass) {
        String tableName = EntityUtils.getTableName(entityClass);
        StringBuilder sqlBuilder = new StringBuilder("INSERT INTO ").
                append(tableName).append(" (").
                append(String.join(", ", fieldMap.keySet())).
                append(") VALUES (").
                append(fieldMap.keySet().stream().map(k -> "?").collect(Collectors.joining(", "))).
                append(")");
        return jdbcTemplate.update(sqlBuilder.toString(), fieldMap.values().toArray());
    }

    public int update(T entity) {
        Map<String, Object> fieldMap = EntityUtils.getFieldMap(entity);
        return update(fieldMap,entity.getClass());
    }
    public int update(Map<String, Object> fieldMap,Class<?> entityClass) {
        String tableName = EntityUtils.getTableName(entityClass);
        StringBuilder sqlBuilder = new StringBuilder("UPDATE ").
                append(tableName).append(" SET ").
                append(fieldMap.keySet().stream().map(k -> k + " = ?").collect(Collectors.joining(", "))).
                append(" WHERE id = ?");
        return jdbcTemplate.update(sqlBuilder.toString(), fieldMap.values().toArray());
    }

    public int deleteById(T entity) {
        String tableName = EntityUtils.getTableName(entity.getClass());
        String sql = "DELETE FROM " + tableName + " WHERE id = ?";
        return jdbcTemplate.update(sql, EntityUtils.getIdValue(entity));
    }

    public T getById(Object id) {
        String tableName = EntityUtils.getTableName(entityClass);
        String sql = "SELECT * FROM " + tableName + " WHERE id = ?";
        return jdbcTemplate.queryForObject(sql, new BeanPropertyRowMapper<>(entityClass), id);
    }

    public List<T> findByWhere(String whereClause, Object... params) {
        String tableName = EntityUtils.getTableName(entityClass);
        StringBuilder sqlBuilder = new StringBuilder("SELECT * FROM " + tableName);
        if(whereClause!=null && !whereClause.trim().isEmpty()) {
            sqlBuilder.append(" WHERE ").append(whereClause);
            return jdbcTemplate.query(sqlBuilder.toString(), new BeanPropertyRowMapper<>(entityClass), params);
        }else{
            return jdbcTemplate.query(sqlBuilder.toString(), new BeanPropertyRowMapper<>(entityClass));
        }
    }

    public T findOneByWhere(String whereClause, Object... params) {
        List<T> list = findByWhere(whereClause, params);
        if(!list.isEmpty()) {
            return list.getFirst();
        }
        return null;
    }

    public int updateByWhere(Map<String, Object> fieldMap, String whereClause, Object... params) {
        String tableName = EntityUtils.getTableName(entityClass);
        StringBuilder sqlBuilder = new StringBuilder("UPDATE ").
                append(tableName).append(" SET ").
                append(fieldMap.keySet().stream().map(k -> k + " = ?").collect(Collectors.joining(", ")));
        if(whereClause!=null && !whereClause.trim().isEmpty()) {
            sqlBuilder.append(" WHERE ").append(whereClause);
        }
        Object[] allParams = Stream.concat(fieldMap.values().stream(), Stream.of(params)).toArray();
        return jdbcTemplate.update(sqlBuilder.toString(), allParams);
    }

    public int deleteByWhere(String whereClause, Object... params) {
        String tableName = EntityUtils.getTableName(entityClass);
        StringBuilder sqlBuilder = new StringBuilder("DELETE FROM ").append(tableName);
        if(whereClause!=null && !whereClause.trim().isEmpty()) {
            sqlBuilder.append(" WHERE ").append(whereClause);
        }
        return jdbcTemplate.update(sqlBuilder.toString(), params);
    }

    public List<Map<String, Object>> selectFieldsByWhere(List<String> fields, String whereClause, Object... params) {
        return jdbcDao.selectFieldsByWhere(getTableName(), fields, whereClause, params);
    }
    public List<Map<String, Object>> selectFieldsByWhere(String select, String whereClause, Object... params) {
        return jdbcDao.selectFieldsByWhere(getTableName(), select, whereClause, params);
    }

    public Map<String, Object> getFieldsByWhere(List<String> fields, String whereClause, Object... params) {
        return jdbcDao.getFieldsByWhere(getTableName(), fields, whereClause, params);
    }
    public Map<String, Object> getFieldsByWhere(String select, String whereClause, Object... params) {
        return jdbcDao.getFieldsByWhere(getTableName(), select, whereClause, params);
    }

    public String getTableName() {
        return getTableName(entityClass);
    }
    public String getTableName(Class<?> entityClass) {
        return EntityUtils.getTableName(entityClass);
    }
    public Class<T> getEntityClass() {
        return entityClass;
    }

    public JdbcDao getJdbcDao() {
        return jdbcDao;
    }

    public void setJdbcDao(JdbcDao jdbcDao) {
        this.jdbcDao = jdbcDao;
    }
}
