/* __________              _____                                                *\
** \______   \____   _____/ ____\____   ____    Copyright (c) 2017-2023 Ponfee  **
**  |     ___/  _ \ /    \   __\/ __ \_/ __ \   http://www.ponfee.cn            **
**  |    |  (  <_> )   |  \  | \  ___/\  ___/   Apache License Version 2.0      **
**  |____|   \____/|___|  /__|  \___  >\___  >  http://www.apache.org/licenses/ **
**                      \/          \/     \/                                   **
\*                                                                              */

package cn.ponfee.commons.util;

import cn.ponfee.commons.date.Dates;
import com.google.common.collect.ImmutableMap;

import java.util.*;
import java.util.concurrent.ThreadLocalRandom;
import java.util.regex.Pattern;

import static java.util.Calendar.YEAR;

/**
 * 身份证解析及生成
 * 
 * http://www.mca.gov.cn/article/sj/xzqh/2018/
 * 
 * @author Ponfee
 */
public class IdcardResolver {

    private static final long ORIGINAL = Dates.toDate("1950-01-01 00:00:00").getTime();

    /**
     * 证件类型
     */
    public enum CertType {
        FIRST, SECOND, HONGKONG, MACAO, TAIWAN, PASSPORT
    }

    /**
     * 性别：M男；F女；N未知；
     */
    public enum Sex {
        M, F, N
    }

    private String idcard; // idCard
    private boolean isValid = false; // 是否有效
    private CertType type; // 类型
    private String province; // 省份
    private String district; // 市县
    private Date birthday; // 生日
    private Integer age; // 年龄
    private Sex sex; // 性别

    /**
     * 随机身份证号码生成
     * @return
     */
    public static String generate() {
        Random random = ThreadLocalRandom.current();
        StringBuilder builder = new StringBuilder(18);
        builder.append(AREA_CODE_LIST.get(random.nextInt(AREA_CODE_LIST.size()))); // 行政区号：6位
        builder.append(Dates.format(Dates.random(ORIGINAL, System.currentTimeMillis()), "yyyyMMdd")); // 生日：8位
        // 当地派出所在该日期的出生顺序号：3位，其中17位（倒数第二位）男为单数，女为双数
        builder.append(String.format("%03d", random.nextInt(1000)));
        builder.append(genPowerSum(builder.toString().toCharArray())); // 校验码：1位
        return builder.toString();
    }

    public IdcardResolver(String idcard) {
        if (idcard == null) {
            return;
        }

        this.idcard = idcard.trim().toUpperCase();
        this.isValid = this.resolve(this.idcard);
    }

    /**
     * 解析
     * @return
     */
    private boolean resolve(String idcard) {
        return isSecond(idcard) || isFirst(idcard) || isPassport(idcard) || isOther(idcard);
    }

    /**
     * 验证15位身份编码是否合法
     * @param idCard 身份编码
     * @return 是否合法
     */
    private static final Pattern FIRST = Pattern.compile("^[0-9]{15}$");
    private boolean isFirst(String idcard) {
        if (!FIRST.matcher(idcard).matches()) {
            return false;
        }

        // 转成18位
        idcard = idcard.substring(0, 6) + "19" + idcard.substring(6);
        idcard += genPowerSum(idcard.toCharArray());
        if (isSecond(idcard)) {
            this.type = CertType.FIRST;
            return true;
        } else {
            return false;
        }
    }

    /**
     * 验证18位身份编码是否合法
     * @param idCard 身份编码
     * @return 是否合法
     */
    private static final Pattern SECOND = Pattern.compile("^[0-9]{17}[0-9X]$");
    private boolean isSecond(String idcard) {
        if (!SECOND.matcher(idcard).matches()) {
            return false;
        }

        // 城市验证
        this.province = CITY_CODES.get(idcard.substring(0, 2));
        if (this.province == null) {
            return false;
        }

        this.district = DISTRICT_CODE_MAPPING.get(Integer.parseInt(idcard.substring(0, 6)));
        //if (this.district == null) return false; // 有些县升为地级市，code改变了

        // 校验码验证
        String prefix17 = idcard.substring(0, 17);
        String checkCode = genPowerSum(prefix17.toCharArray());
        if (!checkCode.equals(idcard.substring(17))) {
            return false;
        }

        // 生日验证
        if (!verifyBirthday(idcard.substring(6, 14))) {
            return false;
        }

        // 性别提取
        if ((Character.getNumericValue(idcard.charAt(16)) & 0x01) == 1) {
            this.sex = Sex.M;
        } else {
            this.sex = Sex.F;
        }

        this.type = CertType.SECOND;
        return true;
    }

    /**
     * 护照验证
     * @param idcard
     * @return
     */
    private static final Pattern PASSPORT_REGEX = Pattern.compile(
        "^1[45][0-9]{7}|G[0-9]{8}|P[0-9]{7}|S[0-9]{7,8}|D[0-9]+$"
    );
    private boolean isPassport(String idcard) {
        if (PASSPORT_REGEX.matcher(idcard).matches()) {
            this.type = CertType.PASSPORT;
            return true;
        }

        return false;
    }

    /**
     * 验证10位身份编码是否合法
     * @param idCard 身份编码
     * @return 身份证信息数组
     */
    private static final Pattern HONGKONG = Pattern.compile("^[A-Z]{1,2}[0-9]{6}\\(?[0-9A]\\)?$");
    private static final Pattern MACO = Pattern.compile("^([157])[0-9]{6}\\(?[0-9A-Z]\\)?$");
    private static final Pattern TAIWAN = Pattern.compile("^[a-zA-Z][0-9]{9}$");
    private boolean isOther(String idcard) {
        idcard = idcard.replaceAll("[(|)]", "");

        if (HONGKONG.matcher(idcard).matches()) { // 香港
            this.type = CertType.HONGKONG;
            this.sex = Sex.N;
            return validateHKCard(idcard);
        } else if (MACO.matcher(idcard).matches()) { // 澳门
            this.type = CertType.MACAO;
            this.sex = Sex.N;
            return true;
        } else if (TAIWAN.matcher(idcard).matches()) { // 台湾
            this.type = CertType.TAIWAN;
            String sex = idcard.substring(1, 2);
            if ("1".equals(sex)) {
                this.sex = Sex.M;
            } else if ("2".equals(sex)) {
                this.sex = Sex.F;
            } else {
                this.sex = Sex.N;
            }
            return validateTWCard(idcard);
        } else {
            return false;
        }
    }

    /**
     * 验证台湾身份证号码
     * @param idCard 身份证号码
     * @return 验证码是否符合
     */
    private boolean validateTWCard(String idCard) {
        String start = idCard.substring(0, 1);
        String mid = idCard.substring(1, 9);
        String end = idCard.substring(9, 10);
        int iStart = TW_FIRST_CODE.get(start);
        int sum = iStart / 10 + (iStart % 10) * 9;
        int iteration = 8;
        for (char c : mid.toCharArray()) {
            sum = sum + (int) c * iteration--;
        }
        return (10 - sum % 10) % 10 == Integer.parseInt(end);
    }

    /**
     * <pre>
     *   验证香港身份证号码(存在bug，部份特殊身份证无法校验)
     *   身份证前2位为英文字符，如果只出现一个英文字符则表示第一位是空格，对应数字58 
     *   前2位英文字符A-Z分别对应数字10-35 最后一位校验码为0-9的数字加上字符"A"，"A"代表10
     *   将身份证号码全部转换为数字，分别对应乘9-1相加的总和，整除11则证件号码有效
     * </pre>
     * 
     * @param idCard 身份证号码
     * @return 验证码是否符合
     */
    private boolean validateHKCard(String idCard) {
        String card = idCard.replaceAll("[\\(|\\)]", "");
        int sum;
        if (card.length() == 9) {
            sum = ((int) card.substring(0, 1).toUpperCase().toCharArray()[0] - 55) * 9
                + ((int) card.substring(1, 2).toUpperCase().toCharArray()[0] - 55) * 8;
            card = card.substring(1, 9);
        } else {
            sum = 522 + ((int) card.substring(0, 1).toUpperCase().toCharArray()[0] - 55) * 8;
        }
        String mid = card.substring(1, 7);
        String end = card.substring(7, 8);
        int iteration = 7;
        for (char c : mid.toCharArray()) {
            sum = sum + (int) c * iteration--;
        }

        if ("A".equalsIgnoreCase(end)) {
            sum = sum + 10;
        } else {
            sum = sum + Integer.parseInt(end);
        }

        return sum % 11 == 0;
    }

    /**
     * <pre>
     * 18位身份证验证
     * 根据〖中华人民共和国国家标准 GB 11643-1999〗中有关公民身份号码的规定，公民身份号码是特征组合码，
     * 由十七位数字本体码和一位数字校验码组成。
     *
     * 排列顺序从左至右依次为：六位数字地址码，八位数字出生日期码，三位数字顺序码和一位数字校验码。
     * 第十八位数字(校验码)的计算方法为：
     * 1.将前面的身份证号码17位数分别乘以不同的系数。从第一位到第十七位的系数分别为：7 9 10 5 8 4 2 1 6 3 7 9 10 5 8 4 2
     * 2.将这17位数字和系数相乘的结果相加。
     * 3.用加出来和除以11，看余数是多少？
     * 4.余数只可能有0 1 2 3 4 5 6 7 8 9 10这11个数字。其分别对应的最后一位身份证的号码为1 0 X 9 8 7 6 5 4 3 2。
     * 5.通过上面得知如果余数是2，就会在身份证的第18位数字上出现罗马数字的Ⅹ。如果余数是10，身份证的最后一位号码就是2。
     * </pre>
     */
    private static String genPowerSum(char[] chars) {
        int[] n = new int[chars.length];
        int result = 0;
        for (int i = 0; i < n.length; i++) {
            n[i] = Character.getNumericValue(chars[i]);
        }
        for (int i = 0; i < n.length; i++) {
            result += POWER[i] * n[i];
        }
        return JUXTAPOSE[result % 11];
    }

    /**
     * 验证生日是否有效
     * @param date
     * @return
     */
    private boolean verifyBirthday(String date) {
        try {
            this.birthday = Dates.toDate(date, "yyyyMMdd");
            if (this.birthday.after(new Date())) {
                return false;
            }
            Calendar calendar = Calendar.getInstance();
            calendar.setTime(this.birthday);
            this.age = Calendar.getInstance().get(YEAR) - calendar.get(YEAR);
            return true;
        } catch (Exception e) {
            return false;
        }
    }

    public String getIdcard() {
        return idcard;
    }

    public boolean isValid() {
        return isValid;
    }

    public CertType getType() {
        return type;
    }

    public String getProvince() {
        return province;
    }

    public String getDistrict() {
        return district;
    }

    public Date getBirthday() {
        return birthday;
    }

    public Sex getSex() {
        return sex;
    }

    public int getAge() {
        return age;
    }

    @Override
    public String toString() {
        return new StringBuilder("IdcardResolver {idcard=")
            .append(idcard)
            .append(", isValid=").append(isValid)
            .append(", type=").append(type)
            .append(", province=").append(province)
            .append(", district=").append(district)
            .append(", birthday=").append(birthday)
            .append(", age=").append(age)
            .append(", sex=").append(sex)
            .append("}").toString();
    }

    /** 每位加权因子 */
    private static final int[] POWER = {
        7, 9, 10, 5, 8, 4, 2, 1, 6, 3, 7, 9, 10, 5, 8, 4, 2 
    };

    /** 校验和对照表 */
    private static final String[] JUXTAPOSE = { 
        "1", "0", "X", "9", "8", "7", "6", "5", "4", "3", "2" 
    };

    /** 大陆城市 */
    private static final Map<String, String> CITY_CODES = 
      new ImmutableMap.Builder<String, String>()
        .put("11", "北京")
        .put("12", "天津")
        .put("13", "河北")
        .put("14", "山西")
        .put("15", "内蒙古")
        .put("21", "辽宁")
        .put("22", "吉林")
        .put("23", "黑龙江")
        .put("31", "上海")
        .put("32", "江苏")
        .put("33", "浙江")
        .put("34", "安徽")
        .put("35", "福建")
        .put("36", "江西")
        .put("37", "山东")
        .put("41", "河南")
        .put("42", "湖北")
        .put("43", "湖南")
        .put("44", "广东")
        .put("45", "广西")
        .put("46", "海南")
        .put("50", "重庆")
        .put("51", "四川")
        .put("52", "贵州")
        .put("53", "云南")
        .put("54", "西藏")
        .put("61", "陕西")
        .put("62", "甘肃")
        .put("63", "青海")
        .put("64", "宁夏")
        .put("65", "新疆")
        .put("71", "台湾")
        .put("81", "香港")
        .put("82", "澳门")
        .put("91", "国外")
        .build();

    /** 台湾身份首字母对应数字 */
    private static final Map<String, Integer> TW_FIRST_CODE = 
        new ImmutableMap.Builder<String, Integer>()
        .put("A", 10)
        .put("B", 11)
        .put("C", 12)
        .put("D", 13)
        .put("E", 14)
        .put("F", 15)
        .put("G", 16)
        .put("H", 17)
        .put("J", 18)
        .put("K", 19)
        .put("L", 20)
        .put("M", 21)
        .put("N", 22)
        .put("P", 23)
        .put("Q", 24)
        .put("R", 25)
        .put("S", 26)
        .put("T", 27)
        .put("U", 28)
        .put("V", 29)
        .put("X", 30)
        .put("Y", 31)
        .put("W", 32)
        .put("Z", 33)
        .put("I", 34)
        .put("O", 35)
        .build();

    public static final Map<Integer, String> DISTRICT_CODE_MAPPING;
    private static final List<Integer> AREA_CODE_LIST;
    static {
        ImmutableMap.Builder<Integer, String> b = new ImmutableMap.Builder<>();
        b.put(110000, "北京市");
        b.put(110101, "东城区");
        b.put(110102, "西城区");
        b.put(110105, "朝阳区");
        b.put(110106, "丰台区");
        b.put(110107, "石景山区");
        b.put(110108, "海淀区");
        b.put(110109, "门头沟区");
        b.put(110111, "房山区");
        b.put(110112, "通州区");
        b.put(110113, "顺义区");
        b.put(110114, "昌平区");
        b.put(110115, "大兴区");
        b.put(110116, "怀柔区");
        b.put(110117, "平谷区");
        b.put(110118, "密云区");
        b.put(110119, "延庆区");
        b.put(120000, "天津市");
        b.put(120101, "和平区");
        b.put(120102, "河东区");
        b.put(120103, "河西区");
        b.put(120104, "南开区");
        b.put(120105, "河北区");
        b.put(120106, "红桥区");
        b.put(120110, "东丽区");
        b.put(120111, "西青区");
        b.put(120112, "津南区");
        b.put(120113, "北辰区");
        b.put(120114, "武清区");
        b.put(120115, "宝坻区");
        b.put(120116, "滨海新区");
        b.put(120117, "宁河区");
        b.put(120118, "静海区");
        b.put(120119, "蓟州区");
        b.put(130000, "河北省");
        b.put(130100, "石家庄市");
        b.put(130102, "长安区");
        b.put(130104, "桥西区");
        b.put(130105, "新华区");
        b.put(130107, "井陉矿区");
        b.put(130108, "裕华区");
        b.put(130109, "藁城区");
        b.put(130110, "鹿泉区");
        b.put(130111, "栾城区");
        b.put(130121, "井陉县");
        b.put(130123, "正定县");
        b.put(130125, "行唐县");
        b.put(130126, "灵寿县");
        b.put(130127, "高邑县");
        b.put(130128, "深泽县");
        b.put(130129, "赞皇县");
        b.put(130130, "无极县");
        b.put(130131, "平山县");
        b.put(130132, "元氏县");
        b.put(130133, "赵县");
        b.put(130181, "辛集市");
        b.put(130183, "晋州市");
        b.put(130184, "新乐市");
        b.put(130200, "唐山市");
        b.put(130202, "路南区");
        b.put(130203, "路北区");
        b.put(130204, "古冶区");
        b.put(130205, "开平区");
        b.put(130207, "丰南区");
        b.put(130208, "丰润区");
        b.put(130209, "曹妃甸区");
        b.put(130223, "滦县");
        b.put(130224, "滦南县");
        b.put(130225, "乐亭县");
        b.put(130227, "迁西县");
        b.put(130229, "玉田县");
        b.put(130281, "遵化市");
        b.put(130283, "迁安市");
        b.put(130300, "秦皇岛市");
        b.put(130302, "海港区");
        b.put(130303, "山海关区");
        b.put(130304, "北戴河区");
        b.put(130306, "抚宁区");
        b.put(130321, "青龙满族自治县");
        b.put(130322, "昌黎县");
        b.put(130324, "卢龙县");
        b.put(130400, "邯郸市");
        b.put(130402, "邯山区");
        b.put(130403, "丛台区");
        b.put(130404, "复兴区");
        b.put(130406, "峰峰矿区");
        b.put(130407, "肥乡区");
        b.put(130408, "永年区");
        b.put(130423, "临漳县");
        b.put(130424, "成安县");
        b.put(130425, "大名县");
        b.put(130426, "涉县");
        b.put(130427, "磁县");
        b.put(130430, "邱县");
        b.put(130431, "鸡泽县");
        b.put(130432, "广平县");
        b.put(130433, "馆陶县");
        b.put(130434, "魏县");
        b.put(130435, "曲周县");
        b.put(130481, "武安市");
        b.put(130500, "邢台市");
        b.put(130502, "桥东区");
        b.put(130503, "桥西区");
        b.put(130521, "邢台县");
        b.put(130522, "临城县");
        b.put(130523, "内丘县");
        b.put(130524, "柏乡县");
        b.put(130525, "隆尧县");
        b.put(130526, "任县");
        b.put(130527, "南和县");
        b.put(130528, "宁晋县");
        b.put(130529, "巨鹿县");
        b.put(130530, "新河县");
        b.put(130531, "广宗县");
        b.put(130532, "平乡县");
        b.put(130533, "威县");
        b.put(130534, "清河县");
        b.put(130535, "临西县");
        b.put(130581, "南宫市");
        b.put(130582, "沙河市");
        b.put(130600, "保定市");
        b.put(130602, "竞秀区");
        b.put(130606, "莲池区");
        b.put(130607, "满城区");
        b.put(130608, "清苑区");
        b.put(130609, "徐水区");
        b.put(130623, "涞水县");
        b.put(130624, "阜平县");
        b.put(130626, "定兴县");
        b.put(130627, "唐县");
        b.put(130628, "高阳县");
        b.put(130629, "容城县");
        b.put(130630, "涞源县");
        b.put(130631, "望都县");
        b.put(130632, "安新县");
        b.put(130633, "易县");
        b.put(130634, "曲阳县");
        b.put(130635, "蠡县");
        b.put(130636, "顺平县");
        b.put(130637, "博野县");
        b.put(130638, "雄县");
        b.put(130681, "涿州市");
        b.put(130682, "定州市");
        b.put(130683, "安国市");
        b.put(130684, "高碑店市");
        b.put(130700, "张家口市");
        b.put(130702, "桥东区");
        b.put(130703, "桥西区");
        b.put(130705, "宣化区");
        b.put(130706, "下花园区");
        b.put(130708, "万全区");
        b.put(130709, "崇礼区");
        b.put(130722, "张北县");
        b.put(130723, "康保县");
        b.put(130724, "沽源县");
        b.put(130725, "尚义县");
        b.put(130726, "蔚县");
        b.put(130727, "阳原县");
        b.put(130728, "怀安县");
        b.put(130730, "怀来县");
        b.put(130731, "涿鹿县");
        b.put(130732, "赤城县");
        b.put(130800, "承德市");
        b.put(130802, "双桥区");
        b.put(130803, "双滦区");
        b.put(130804, "鹰手营子矿区");
        b.put(130821, "承德县");
        b.put(130822, "兴隆县");
        b.put(130824, "滦平县");
        b.put(130825, "隆化县");
        b.put(130826, "丰宁满族自治县");
        b.put(130827, "宽城满族自治县");
        b.put(130828, "围场满族蒙古族自治县");
        b.put(130881, "平泉市");
        b.put(130900, "沧州市");
        b.put(130902, "新华区");
        b.put(130903, "运河区");
        b.put(130921, "沧县");
        b.put(130922, "青县");
        b.put(130923, "东光县");
        b.put(130924, "海兴县");
        b.put(130925, "盐山县");
        b.put(130926, "肃宁县");
        b.put(130927, "南皮县");
        b.put(130928, "吴桥县");
        b.put(130929, "献县");
        b.put(130930, "孟村回族自治县");
        b.put(130981, "泊头市");
        b.put(130982, "任丘市");
        b.put(130983, "黄骅市");
        b.put(130984, "河间市");
        b.put(131000, "廊坊市");
        b.put(131002, "安次区");
        b.put(131003, "广阳区");
        b.put(131022, "固安县");
        b.put(131023, "永清县");
        b.put(131024, "香河县");
        b.put(131025, "大城县");
        b.put(131026, "文安县");
        b.put(131028, "大厂回族自治县");
        b.put(131081, "霸州市");
        b.put(131082, "三河市");
        b.put(131100, "衡水市");
        b.put(131102, "桃城区");
        b.put(131103, "冀州区");
        b.put(131121, "枣强县");
        b.put(131122, "武邑县");
        b.put(131123, "武强县");
        b.put(131124, "饶阳县");
        b.put(131125, "安平县");
        b.put(131126, "故城县");
        b.put(131127, "景县");
        b.put(131128, "阜城县");
        b.put(131182, "深州市");
        b.put(140000, "山西省");
        b.put(140100, "太原市");
        b.put(140105, "小店区");
        b.put(140106, "迎泽区");
        b.put(140107, "杏花岭区");
        b.put(140108, "尖草坪区");
        b.put(140109, "万柏林区");
        b.put(140110, "晋源区");
        b.put(140121, "清徐县");
        b.put(140122, "阳曲县");
        b.put(140123, "娄烦县");
        b.put(140181, "古交市");
        b.put(140200, "大同市");
        b.put(140212, "新荣区");
        b.put(140213, "平城区");
        b.put(140214, "云冈区");
        b.put(140215, "云州区");
        b.put(140221, "阳高县");
        b.put(140222, "天镇县");
        b.put(140223, "广灵县");
        b.put(140224, "灵丘县");
        b.put(140225, "浑源县");
        b.put(140226, "左云县");
        b.put(140300, "阳泉市");
        b.put(140302, "城区");
        b.put(140303, "矿区");
        b.put(140311, "郊区");
        b.put(140321, "平定县");
        b.put(140322, "盂县");
        b.put(140400, "长治市");
        b.put(140402, "城区");
        b.put(140411, "郊区");
        b.put(140421, "长治县");
        b.put(140423, "襄垣县");
        b.put(140424, "屯留县");
        b.put(140425, "平顺县");
        b.put(140426, "黎城县");
        b.put(140427, "壶关县");
        b.put(140428, "长子县");
        b.put(140429, "武乡县");
        b.put(140430, "沁县");
        b.put(140431, "沁源县");
        b.put(140481, "潞城市");
        b.put(140500, "晋城市");
        b.put(140502, "城区");
        b.put(140521, "沁水县");
        b.put(140522, "阳城县");
        b.put(140524, "陵川县");
        b.put(140525, "泽州县");
        b.put(140581, "高平市");
        b.put(140600, "朔州市");
        b.put(140602, "朔城区");
        b.put(140603, "平鲁区");
        b.put(140621, "山阴县");
        b.put(140622, "应县");
        b.put(140623, "右玉县");
        b.put(140681, "怀仁市");
        b.put(140700, "晋中市");
        b.put(140702, "榆次区");
        b.put(140721, "榆社县");
        b.put(140722, "左权县");
        b.put(140723, "和顺县");
        b.put(140724, "昔阳县");
        b.put(140725, "寿阳县");
        b.put(140726, "太谷县");
        b.put(140727, "祁县");
        b.put(140728, "平遥县");
        b.put(140729, "灵石县");
        b.put(140781, "介休市");
        b.put(140800, "运城市");
        b.put(140802, "盐湖区");
        b.put(140821, "临猗县");
        b.put(140822, "万荣县");
        b.put(140823, "闻喜县");
        b.put(140824, "稷山县");
        b.put(140825, "新绛县");
        b.put(140826, "绛县");
        b.put(140827, "垣曲县");
        b.put(140828, "夏县");
        b.put(140829, "平陆县");
        b.put(140830, "芮城县");
        b.put(140881, "永济市");
        b.put(140882, "河津市");
        b.put(140900, "忻州市");
        b.put(140902, "忻府区");
        b.put(140921, "定襄县");
        b.put(140922, "五台县");
        b.put(140923, "代县");
        b.put(140924, "繁峙县");
        b.put(140925, "宁武县");
        b.put(140926, "静乐县");
        b.put(140927, "神池县");
        b.put(140928, "五寨县");
        b.put(140929, "岢岚县");
        b.put(140930, "河曲县");
        b.put(140931, "保德县");
        b.put(140932, "偏关县");
        b.put(140981, "原平市");
        b.put(141000, "临汾市");
        b.put(141002, "尧都区");
        b.put(141021, "曲沃县");
        b.put(141022, "翼城县");
        b.put(141023, "襄汾县");
        b.put(141024, "洪洞县");
        b.put(141025, "古县");
        b.put(141026, "安泽县");
        b.put(141027, "浮山县");
        b.put(141028, "吉县");
        b.put(141029, "乡宁县");
        b.put(141030, "大宁县");
        b.put(141031, "隰县");
        b.put(141032, "永和县");
        b.put(141033, "蒲县");
        b.put(141034, "汾西县");
        b.put(141081, "侯马市");
        b.put(141082, "霍州市");
        b.put(141100, "吕梁市");
        b.put(141102, "离石区");
        b.put(141121, "文水县");
        b.put(141122, "交城县");
        b.put(141123, "兴县");
        b.put(141124, "临县");
        b.put(141125, "柳林县");
        b.put(141126, "石楼县");
        b.put(141127, "岚县");
        b.put(141128, "方山县");
        b.put(141129, "中阳县");
        b.put(141130, "交口县");
        b.put(141181, "孝义市");
        b.put(141182, "汾阳市");
        b.put(150000, "内蒙古自治区");
        b.put(150100, "呼和浩特市");
        b.put(150102, "新城区");
        b.put(150103, "回民区");
        b.put(150104, "玉泉区");
        b.put(150105, "赛罕区");
        b.put(150121, "土默特左旗");
        b.put(150122, "托克托县");
        b.put(150123, "和林格尔县");
        b.put(150124, "清水河县");
        b.put(150125, "武川县");
        b.put(150200, "包头市");
        b.put(150202, "东河区");
        b.put(150203, "昆都仑区");
        b.put(150204, "青山区");
        b.put(150205, "石拐区");
        b.put(150206, "白云鄂博矿区");
        b.put(150207, "九原区");
        b.put(150221, "土默特右旗");
        b.put(150222, "固阳县");
        b.put(150223, "达尔罕茂明安联合旗");
        b.put(150300, "乌海市");
        b.put(150302, "海勃湾区");
        b.put(150303, "海南区");
        b.put(150304, "乌达区");
        b.put(150400, "赤峰市");
        b.put(150402, "红山区");
        b.put(150403, "元宝山区");
        b.put(150404, "松山区");
        b.put(150421, "阿鲁科尔沁旗");
        b.put(150422, "巴林左旗");
        b.put(150423, "巴林右旗");
        b.put(150424, "林西县");
        b.put(150425, "克什克腾旗");
        b.put(150426, "翁牛特旗");
        b.put(150428, "喀喇沁旗");
        b.put(150429, "宁城县");
        b.put(150430, "敖汉旗");
        b.put(150500, "通辽市");
        b.put(150502, "科尔沁区");
        b.put(150521, "科尔沁左翼中旗");
        b.put(150522, "科尔沁左翼后旗");
        b.put(150523, "开鲁县");
        b.put(150524, "库伦旗");
        b.put(150525, "奈曼旗");
        b.put(150526, "扎鲁特旗");
        b.put(150581, "霍林郭勒市");
        b.put(150600, "鄂尔多斯市");
        b.put(150602, "东胜区");
        b.put(150603, "康巴什区");
        b.put(150621, "达拉特旗");
        b.put(150622, "准格尔旗");
        b.put(150623, "鄂托克前旗");
        b.put(150624, "鄂托克旗");
        b.put(150625, "杭锦旗");
        b.put(150626, "乌审旗");
        b.put(150627, "伊金霍洛旗");
        b.put(150700, "呼伦贝尔市");
        b.put(150702, "海拉尔区");
        b.put(150703, "扎赉诺尔区");
        b.put(150721, "阿荣旗");
        b.put(150722, "莫力达瓦达斡尔族自治旗");
        b.put(150723, "鄂伦春自治旗");
        b.put(150724, "鄂温克族自治旗");
        b.put(150725, "陈巴尔虎旗");
        b.put(150726, "新巴尔虎左旗");
        b.put(150727, "新巴尔虎右旗");
        b.put(150781, "满洲里市");
        b.put(150782, "牙克石市");
        b.put(150783, "扎兰屯市");
        b.put(150784, "额尔古纳市");
        b.put(150785, "根河市");
        b.put(150800, "巴彦淖尔市");
        b.put(150802, "临河区");
        b.put(150821, "五原县");
        b.put(150822, "磴口县");
        b.put(150823, "乌拉特前旗");
        b.put(150824, "乌拉特中旗");
        b.put(150825, "乌拉特后旗");
        b.put(150826, "杭锦后旗");
        b.put(150900, "乌兰察布市");
        b.put(150902, "集宁区");
        b.put(150921, "卓资县");
        b.put(150922, "化德县");
        b.put(150923, "商都县");
        b.put(150924, "兴和县");
        b.put(150925, "凉城县");
        b.put(150926, "察哈尔右翼前旗");
        b.put(150927, "察哈尔右翼中旗");
        b.put(150928, "察哈尔右翼后旗");
        b.put(150929, "四子王旗");
        b.put(150981, "丰镇市");
        b.put(152200, "兴安盟");
        b.put(152201, "乌兰浩特市");
        b.put(152202, "阿尔山市");
        b.put(152221, "科尔沁右翼前旗");
        b.put(152222, "科尔沁右翼中旗");
        b.put(152223, "扎赉特旗");
        b.put(152224, "突泉县");
        b.put(152500, "锡林郭勒盟");
        b.put(152501, "二连浩特市");
        b.put(152502, "锡林浩特市");
        b.put(152522, "阿巴嘎旗");
        b.put(152523, "苏尼特左旗");
        b.put(152524, "苏尼特右旗");
        b.put(152525, "东乌珠穆沁旗");
        b.put(152526, "西乌珠穆沁旗");
        b.put(152527, "太仆寺旗");
        b.put(152528, "镶黄旗");
        b.put(152529, "正镶白旗");
        b.put(152530, "正蓝旗");
        b.put(152531, "多伦县");
        b.put(152900, "阿拉善盟");
        b.put(152921, "阿拉善左旗");
        b.put(152922, "阿拉善右旗");
        b.put(152923, "额济纳旗");
        b.put(210000, "辽宁省");
        b.put(210100, "沈阳市");
        b.put(210102, "和平区");
        b.put(210103, "沈河区");
        b.put(210104, "大东区");
        b.put(210105, "皇姑区");
        b.put(210106, "铁西区");
        b.put(210111, "苏家屯区");
        b.put(210112, "浑南区");
        b.put(210113, "沈北新区");
        b.put(210114, "于洪区");
        b.put(210115, "辽中区");
        b.put(210123, "康平县");
        b.put(210124, "法库县");
        b.put(210181, "新民市");
        b.put(210200, "大连市");
        b.put(210202, "中山区");
        b.put(210203, "西岗区");
        b.put(210204, "沙河口区");
        b.put(210211, "甘井子区");
        b.put(210212, "旅顺口区");
        b.put(210213, "金州区");
        b.put(210214, "普兰店区");
        b.put(210224, "长海县");
        b.put(210281, "瓦房店市");
        b.put(210283, "庄河市");
        b.put(210300, "鞍山市");
        b.put(210302, "铁东区");
        b.put(210303, "铁西区");
        b.put(210304, "立山区");
        b.put(210311, "千山区");
        b.put(210321, "台安县");
        b.put(210323, "岫岩满族自治县");
        b.put(210381, "海城市");
        b.put(210400, "抚顺市");
        b.put(210402, "新抚区");
        b.put(210403, "东洲区");
        b.put(210404, "望花区");
        b.put(210411, "顺城区");
        b.put(210421, "抚顺县");
        b.put(210422, "新宾满族自治县");
        b.put(210423, "清原满族自治县");
        b.put(210500, "本溪市");
        b.put(210502, "平山区");
        b.put(210503, "溪湖区");
        b.put(210504, "明山区");
        b.put(210505, "南芬区");
        b.put(210521, "本溪满族自治县");
        b.put(210522, "桓仁满族自治县");
        b.put(210600, "丹东市");
        b.put(210602, "元宝区");
        b.put(210603, "振兴区");
        b.put(210604, "振安区");
        b.put(210624, "宽甸满族自治县");
        b.put(210681, "东港市");
        b.put(210682, "凤城市");
        b.put(210700, "锦州市");
        b.put(210702, "古塔区");
        b.put(210703, "凌河区");
        b.put(210711, "太和区");
        b.put(210726, "黑山县");
        b.put(210727, "义县");
        b.put(210781, "凌海市");
        b.put(210782, "北镇市");
        b.put(210800, "营口市");
        b.put(210802, "站前区");
        b.put(210803, "西市区");
        b.put(210804, "鲅鱼圈区");
        b.put(210811, "老边区");
        b.put(210881, "盖州市");
        b.put(210882, "大石桥市");
        b.put(210900, "阜新市");
        b.put(210902, "海州区");
        b.put(210903, "新邱区");
        b.put(210904, "太平区");
        b.put(210905, "清河门区");
        b.put(210911, "细河区");
        b.put(210921, "阜新蒙古族自治县");
        b.put(210922, "彰武县");
        b.put(211000, "辽阳市");
        b.put(211002, "白塔区");
        b.put(211003, "文圣区");
        b.put(211004, "宏伟区");
        b.put(211005, "弓长岭区");
        b.put(211011, "太子河区");
        b.put(211021, "辽阳县");
        b.put(211081, "灯塔市");
        b.put(211100, "盘锦市");
        b.put(211102, "双台子区");
        b.put(211103, "兴隆台区");
        b.put(211104, "大洼区");
        b.put(211122, "盘山县");
        b.put(211200, "铁岭市");
        b.put(211202, "银州区");
        b.put(211204, "清河区");
        b.put(211221, "铁岭县");
        b.put(211223, "西丰县");
        b.put(211224, "昌图县");
        b.put(211281, "调兵山市");
        b.put(211282, "开原市");
        b.put(211300, "朝阳市");
        b.put(211302, "双塔区");
        b.put(211303, "龙城区");
        b.put(211321, "朝阳县");
        b.put(211322, "建平县");
        b.put(211324, "喀喇沁左翼蒙古族自治县");
        b.put(211381, "北票市");
        b.put(211382, "凌源市");
        b.put(211400, "葫芦岛市");
        b.put(211402, "连山区");
        b.put(211403, "龙港区");
        b.put(211404, "南票区");
        b.put(211421, "绥中县");
        b.put(211422, "建昌县");
        b.put(211481, "兴城市");
        b.put(220000, "吉林省");
        b.put(220100, "长春市");
        b.put(220102, "南关区");
        b.put(220103, "宽城区");
        b.put(220104, "朝阳区");
        b.put(220105, "二道区");
        b.put(220106, "绿园区");
        b.put(220112, "双阳区");
        b.put(220113, "九台区");
        b.put(220122, "农安县");
        b.put(220182, "榆树市");
        b.put(220183, "德惠市");
        b.put(220200, "吉林市");
        b.put(220202, "昌邑区");
        b.put(220203, "龙潭区");
        b.put(220204, "船营区");
        b.put(220211, "丰满区");
        b.put(220221, "永吉县");
        b.put(220281, "蛟河市");
        b.put(220282, "桦甸市");
        b.put(220283, "舒兰市");
        b.put(220284, "磐石市");
        b.put(220300, "四平市");
        b.put(220302, "铁西区");
        b.put(220303, "铁东区");
        b.put(220322, "梨树县");
        b.put(220323, "伊通满族自治县");
        b.put(220381, "公主岭市");
        b.put(220382, "双辽市");
        b.put(220400, "辽源市");
        b.put(220402, "龙山区");
        b.put(220403, "西安区");
        b.put(220421, "东丰县");
        b.put(220422, "东辽县");
        b.put(220500, "通化市");
        b.put(220502, "东昌区");
        b.put(220503, "二道江区");
        b.put(220521, "通化县");
        b.put(220523, "辉南县");
        b.put(220524, "柳河县");
        b.put(220581, "梅河口市");
        b.put(220582, "集安市");
        b.put(220600, "白山市");
        b.put(220602, "浑江区");
        b.put(220605, "江源区");
        b.put(220621, "抚松县");
        b.put(220622, "靖宇县");
        b.put(220623, "长白朝鲜族自治县");
        b.put(220681, "临江市");
        b.put(220700, "松原市");
        b.put(220702, "宁江区");
        b.put(220721, "前郭尔罗斯蒙古族自治县");
        b.put(220722, "长岭县");
        b.put(220723, "乾安县");
        b.put(220781, "扶余市");
        b.put(220800, "白城市");
        b.put(220802, "洮北区");
        b.put(220821, "镇赉县");
        b.put(220822, "通榆县");
        b.put(220881, "洮南市");
        b.put(220882, "大安市");
        b.put(222400, "延边朝鲜族自治州");
        b.put(222401, "延吉市");
        b.put(222402, "图们市");
        b.put(222403, "敦化市");
        b.put(222404, "珲春市");
        b.put(222405, "龙井市");
        b.put(222406, "和龙市");
        b.put(222424, "汪清县");
        b.put(222426, "安图县");
        b.put(230000, "黑龙江省");
        b.put(230100, "哈尔滨市");
        b.put(230102, "道里区");
        b.put(230103, "南岗区");
        b.put(230104, "道外区");
        b.put(230108, "平房区");
        b.put(230109, "松北区");
        b.put(230110, "香坊区");
        b.put(230111, "呼兰区");
        b.put(230112, "阿城区");
        b.put(230113, "双城区");
        b.put(230123, "依兰县");
        b.put(230124, "方正县");
        b.put(230125, "宾县");
        b.put(230126, "巴彦县");
        b.put(230127, "木兰县");
        b.put(230128, "通河县");
        b.put(230129, "延寿县");
        b.put(230183, "尚志市");
        b.put(230184, "五常市");
        b.put(230200, "齐齐哈尔市");
        b.put(230202, "龙沙区");
        b.put(230203, "建华区");
        b.put(230204, "铁锋区");
        b.put(230205, "昂昂溪区");
        b.put(230206, "富拉尔基区");
        b.put(230207, "碾子山区");
        b.put(230208, "梅里斯达斡尔族区");
        b.put(230221, "龙江县");
        b.put(230223, "依安县");
        b.put(230224, "泰来县");
        b.put(230225, "甘南县");
        b.put(230227, "富裕县");
        b.put(230229, "克山县");
        b.put(230230, "克东县");
        b.put(230231, "拜泉县");
        b.put(230281, "讷河市");
        b.put(230300, "鸡西市");
        b.put(230302, "鸡冠区");
        b.put(230303, "恒山区");
        b.put(230304, "滴道区");
        b.put(230305, "梨树区");
        b.put(230306, "城子河区");
        b.put(230307, "麻山区");
        b.put(230321, "鸡东县");
        b.put(230381, "虎林市");
        b.put(230382, "密山市");
        b.put(230400, "鹤岗市");
        b.put(230402, "向阳区");
        b.put(230403, "工农区");
        b.put(230404, "南山区");
        b.put(230405, "兴安区");
        b.put(230406, "东山区");
        b.put(230407, "兴山区");
        b.put(230421, "萝北县");
        b.put(230422, "绥滨县");
        b.put(230500, "双鸭山市");
        b.put(230502, "尖山区");
        b.put(230503, "岭东区");
        b.put(230505, "四方台区");
        b.put(230506, "宝山区");
        b.put(230521, "集贤县");
        b.put(230522, "友谊县");
        b.put(230523, "宝清县");
        b.put(230524, "饶河县");
        b.put(230600, "大庆市");
        b.put(230602, "萨尔图区");
        b.put(230603, "龙凤区");
        b.put(230604, "让胡路区");
        b.put(230605, "红岗区");
        b.put(230606, "大同区");
        b.put(230621, "肇州县");
        b.put(230622, "肇源县");
        b.put(230623, "林甸县");
        b.put(230624, "杜尔伯特蒙古族自治县");
        b.put(230700, "伊春市");
        b.put(230702, "伊春区");
        b.put(230703, "南岔区");
        b.put(230704, "友好区");
        b.put(230705, "西林区");
        b.put(230706, "翠峦区");
        b.put(230707, "新青区");
        b.put(230708, "美溪区");
        b.put(230709, "金山屯区");
        b.put(230710, "五营区");
        b.put(230711, "乌马河区");
        b.put(230712, "汤旺河区");
        b.put(230713, "带岭区");
        b.put(230714, "乌伊岭区");
        b.put(230715, "红星区");
        b.put(230716, "上甘岭区");
        b.put(230722, "嘉荫县");
        b.put(230781, "铁力市");
        b.put(230800, "佳木斯市");
        b.put(230803, "向阳区");
        b.put(230804, "前进区");
        b.put(230805, "东风区");
        b.put(230811, "郊区");
        b.put(230822, "桦南县");
        b.put(230826, "桦川县");
        b.put(230828, "汤原县");
        b.put(230881, "同江市");
        b.put(230882, "富锦市");
        b.put(230883, "抚远市");
        b.put(230900, "七台河市");
        b.put(230902, "新兴区");
        b.put(230903, "桃山区");
        b.put(230904, "茄子河区");
        b.put(230921, "勃利县");
        b.put(231000, "牡丹江市");
        b.put(231002, "东安区");
        b.put(231003, "阳明区");
        b.put(231004, "爱民区");
        b.put(231005, "西安区");
        b.put(231025, "林口县");
        b.put(231081, "绥芬河市");
        b.put(231083, "海林市");
        b.put(231084, "宁安市");
        b.put(231085, "穆棱市");
        b.put(231086, "东宁市");
        b.put(231100, "黑河市");
        b.put(231102, "爱辉区");
        b.put(231121, "嫩江县");
        b.put(231123, "逊克县");
        b.put(231124, "孙吴县");
        b.put(231181, "北安市");
        b.put(231182, "五大连池市");
        b.put(231200, "绥化市");
        b.put(231202, "北林区");
        b.put(231221, "望奎县");
        b.put(231222, "兰西县");
        b.put(231223, "青冈县");
        b.put(231224, "庆安县");
        b.put(231225, "明水县");
        b.put(231226, "绥棱县");
        b.put(231281, "安达市");
        b.put(231282, "肇东市");
        b.put(231283, "海伦市");
        b.put(232700, "大兴安岭地区");
        b.put(232701, "漠河市");
        b.put(232721, "呼玛县");
        b.put(232722, "塔河县");
        b.put(310000, "上海市");
        b.put(310101, "黄浦区");
        b.put(310104, "徐汇区");
        b.put(310105, "长宁区");
        b.put(310106, "静安区");
        b.put(310107, "普陀区");
        b.put(310109, "虹口区");
        b.put(310110, "杨浦区");
        b.put(310112, "闵行区");
        b.put(310113, "宝山区");
        b.put(310114, "嘉定区");
        b.put(310115, "浦东新区");
        b.put(310116, "金山区");
        b.put(310117, "松江区");
        b.put(310118, "青浦区");
        b.put(310120, "奉贤区");
        b.put(310151, "崇明区");
        b.put(320000, "江苏省");
        b.put(320100, "南京市");
        b.put(320102, "玄武区");
        b.put(320104, "秦淮区");
        b.put(320105, "建邺区");
        b.put(320106, "鼓楼区");
        b.put(320111, "浦口区");
        b.put(320113, "栖霞区");
        b.put(320114, "雨花台区");
        b.put(320115, "江宁区");
        b.put(320116, "六合区");
        b.put(320117, "溧水区");
        b.put(320118, "高淳区");
        b.put(320200, "无锡市");
        b.put(320205, "锡山区");
        b.put(320206, "惠山区");
        b.put(320211, "滨湖区");
        b.put(320213, "梁溪区");
        b.put(320214, "新吴区");
        b.put(320281, "江阴市");
        b.put(320282, "宜兴市");
        b.put(320300, "徐州市");
        b.put(320302, "鼓楼区");
        b.put(320303, "云龙区");
        b.put(320305, "贾汪区");
        b.put(320311, "泉山区");
        b.put(320312, "铜山区");
        b.put(320321, "丰县");
        b.put(320322, "沛县");
        b.put(320324, "睢宁县");
        b.put(320381, "新沂市");
        b.put(320382, "邳州市");
        b.put(320400, "常州市");
        b.put(320402, "天宁区");
        b.put(320404, "钟楼区");
        b.put(320411, "新北区");
        b.put(320412, "武进区");
        b.put(320413, "金坛区");
        b.put(320481, "溧阳市");
        b.put(320500, "苏州市");
        b.put(320505, "虎丘区");
        b.put(320506, "吴中区");
        b.put(320507, "相城区");
        b.put(320508, "姑苏区");
        b.put(320509, "吴江区");
        b.put(320581, "常熟市");
        b.put(320582, "张家港市");
        b.put(320583, "昆山市");
        b.put(320585, "太仓市");
        b.put(320600, "南通市");
        b.put(320602, "崇川区");
        b.put(320611, "港闸区");
        b.put(320612, "通州区");
        b.put(320623, "如东县");
        b.put(320681, "启东市");
        b.put(320682, "如皋市");
        b.put(320684, "海门市");
        b.put(320685, "海安市");
        b.put(320700, "连云港市");
        b.put(320703, "连云区");
        b.put(320706, "海州区");
        b.put(320707, "赣榆区");
        b.put(320722, "东海县");
        b.put(320723, "灌云县");
        b.put(320724, "灌南县");
        b.put(320800, "淮安市");
        b.put(320803, "淮安区");
        b.put(320804, "淮阴区");
        b.put(320812, "清江浦区");
        b.put(320813, "洪泽区");
        b.put(320826, "涟水县");
        b.put(320830, "盱眙县");
        b.put(320831, "金湖县");
        b.put(320900, "盐城市");
        b.put(320902, "亭湖区");
        b.put(320903, "盐都区");
        b.put(320904, "大丰区");
        b.put(320921, "响水县");
        b.put(320922, "滨海县");
        b.put(320923, "阜宁县");
        b.put(320924, "射阳县");
        b.put(320925, "建湖县");
        b.put(320981, "东台市");
        b.put(321000, "扬州市");
        b.put(321002, "广陵区");
        b.put(321003, "邗江区");
        b.put(321012, "江都区");
        b.put(321023, "宝应县");
        b.put(321081, "仪征市");
        b.put(321084, "高邮市");
        b.put(321100, "镇江市");
        b.put(321102, "京口区");
        b.put(321111, "润州区");
        b.put(321112, "丹徒区");
        b.put(321181, "丹阳市");
        b.put(321182, "扬中市");
        b.put(321183, "句容市");
        b.put(321200, "泰州市");
        b.put(321202, "海陵区");
        b.put(321203, "高港区");
        b.put(321204, "姜堰区");
        b.put(321281, "兴化市");
        b.put(321282, "靖江市");
        b.put(321283, "泰兴市");
        b.put(321300, "宿迁市");
        b.put(321302, "宿城区");
        b.put(321311, "宿豫区");
        b.put(321322, "沭阳县");
        b.put(321323, "泗阳县");
        b.put(321324, "泗洪县");
        b.put(330000, "浙江省");
        b.put(330100, "杭州市");
        b.put(330102, "上城区");
        b.put(330103, "下城区");
        b.put(330104, "江干区");
        b.put(330105, "拱墅区");
        b.put(330106, "西湖区");
        b.put(330108, "滨江区");
        b.put(330109, "萧山区");
        b.put(330110, "余杭区");
        b.put(330111, "富阳区");
        b.put(330112, "临安区");
        b.put(330122, "桐庐县");
        b.put(330127, "淳安县");
        b.put(330182, "建德市");
        b.put(330200, "宁波市");
        b.put(330203, "海曙区");
        b.put(330205, "江北区");
        b.put(330206, "北仑区");
        b.put(330211, "镇海区");
        b.put(330212, "鄞州区");
        b.put(330213, "奉化区");
        b.put(330225, "象山县");
        b.put(330226, "宁海县");
        b.put(330281, "余姚市");
        b.put(330282, "慈溪市");
        b.put(330300, "温州市");
        b.put(330302, "鹿城区");
        b.put(330303, "龙湾区");
        b.put(330304, "瓯海区");
        b.put(330305, "洞头区");
        b.put(330324, "永嘉县");
        b.put(330326, "平阳县");
        b.put(330327, "苍南县");
        b.put(330328, "文成县");
        b.put(330329, "泰顺县");
        b.put(330381, "瑞安市");
        b.put(330382, "乐清市");
        b.put(330400, "嘉兴市");
        b.put(330402, "南湖区");
        b.put(330411, "秀洲区");
        b.put(330421, "嘉善县");
        b.put(330424, "海盐县");
        b.put(330481, "海宁市");
        b.put(330482, "平湖市");
        b.put(330483, "桐乡市");
        b.put(330500, "湖州市");
        b.put(330502, "吴兴区");
        b.put(330503, "南浔区");
        b.put(330521, "德清县");
        b.put(330522, "长兴县");
        b.put(330523, "安吉县");
        b.put(330600, "绍兴市");
        b.put(330602, "越城区");
        b.put(330603, "柯桥区");
        b.put(330604, "上虞区");
        b.put(330624, "新昌县");
        b.put(330681, "诸暨市");
        b.put(330683, "嵊州市");
        b.put(330700, "金华市");
        b.put(330702, "婺城区");
        b.put(330703, "金东区");
        b.put(330723, "武义县");
        b.put(330726, "浦江县");
        b.put(330727, "磐安县");
        b.put(330781, "兰溪市");
        b.put(330782, "义乌市");
        b.put(330783, "东阳市");
        b.put(330784, "永康市");
        b.put(330800, "衢州市");
        b.put(330802, "柯城区");
        b.put(330803, "衢江区");
        b.put(330822, "常山县");
        b.put(330824, "开化县");
        b.put(330825, "龙游县");
        b.put(330881, "江山市");
        b.put(330900, "舟山市");
        b.put(330902, "定海区");
        b.put(330903, "普陀区");
        b.put(330921, "岱山县");
        b.put(330922, "嵊泗县");
        b.put(331000, "台州市");
        b.put(331002, "椒江区");
        b.put(331003, "黄岩区");
        b.put(331004, "路桥区");
        b.put(331022, "三门县");
        b.put(331023, "天台县");
        b.put(331024, "仙居县");
        b.put(331081, "温岭市");
        b.put(331082, "临海市");
        b.put(331083, "玉环市");
        b.put(331100, "丽水市");
        b.put(331102, "莲都区");
        b.put(331121, "青田县");
        b.put(331122, "缙云县");
        b.put(331123, "遂昌县");
        b.put(331124, "松阳县");
        b.put(331125, "云和县");
        b.put(331126, "庆元县");
        b.put(331127, "景宁畲族自治县");
        b.put(331181, "龙泉市");
        b.put(340000, "安徽省");
        b.put(340100, "合肥市");
        b.put(340102, "瑶海区");
        b.put(340103, "庐阳区");
        b.put(340104, "蜀山区");
        b.put(340111, "包河区");
        b.put(340121, "长丰县");
        b.put(340122, "肥东县");
        b.put(340123, "肥西县");
        b.put(340124, "庐江县");
        b.put(340181, "巢湖市");
        b.put(340200, "芜湖市");
        b.put(340202, "镜湖区");
        b.put(340203, "弋江区");
        b.put(340207, "鸠江区");
        b.put(340208, "三山区");
        b.put(340221, "芜湖县");
        b.put(340222, "繁昌县");
        b.put(340223, "南陵县");
        b.put(340225, "无为县");
        b.put(340300, "蚌埠市");
        b.put(340302, "龙子湖区");
        b.put(340303, "蚌山区");
        b.put(340304, "禹会区");
        b.put(340311, "淮上区");
        b.put(340321, "怀远县");
        b.put(340322, "五河县");
        b.put(340323, "固镇县");
        b.put(340400, "淮南市");
        b.put(340402, "大通区");
        b.put(340403, "田家庵区");
        b.put(340404, "谢家集区");
        b.put(340405, "八公山区");
        b.put(340406, "潘集区");
        b.put(340421, "凤台县");
        b.put(340422, "寿县");
        b.put(340500, "马鞍山市");
        b.put(340503, "花山区");
        b.put(340504, "雨山区");
        b.put(340506, "博望区");
        b.put(340521, "当涂县");
        b.put(340522, "含山县");
        b.put(340523, "和县");
        b.put(340600, "淮北市");
        b.put(340602, "杜集区");
        b.put(340603, "相山区");
        b.put(340604, "烈山区");
        b.put(340621, "濉溪县");
        b.put(340700, "铜陵市");
        b.put(340705, "铜官区");
        b.put(340706, "义安区");
        b.put(340711, "郊区");
        b.put(340722, "枞阳县");
        b.put(340800, "安庆市");
        b.put(340802, "迎江区");
        b.put(340803, "大观区");
        b.put(340811, "宜秀区");
        b.put(340822, "怀宁县");
        b.put(340824, "潜山县");
        b.put(340825, "太湖县");
        b.put(340826, "宿松县");
        b.put(340827, "望江县");
        b.put(340828, "岳西县");
        b.put(340881, "桐城市");
        b.put(341000, "黄山市");
        b.put(341002, "屯溪区");
        b.put(341003, "黄山区");
        b.put(341004, "徽州区");
        b.put(341021, "歙县");
        b.put(341022, "休宁县");
        b.put(341023, "黟县");
        b.put(341024, "祁门县");
        b.put(341100, "滁州市");
        b.put(341102, "琅琊区");
        b.put(341103, "南谯区");
        b.put(341122, "来安县");
        b.put(341124, "全椒县");
        b.put(341125, "定远县");
        b.put(341126, "凤阳县");
        b.put(341181, "天长市");
        b.put(341182, "明光市");
        b.put(341200, "阜阳市");
        b.put(341202, "颍州区");
        b.put(341203, "颍东区");
        b.put(341204, "颍泉区");
        b.put(341221, "临泉县");
        b.put(341222, "太和县");
        b.put(341225, "阜南县");
        b.put(341226, "颍上县");
        b.put(341282, "界首市");
        b.put(341300, "宿州市");
        b.put(341302, "埇桥区");
        b.put(341321, "砀山县");
        b.put(341322, "萧县");
        b.put(341323, "灵璧县");
        b.put(341324, "泗县");
        b.put(341500, "六安市");
        b.put(341502, "金安区");
        b.put(341503, "裕安区");
        b.put(341504, "叶集区");
        b.put(341522, "霍邱县");
        b.put(341523, "舒城县");
        b.put(341524, "金寨县");
        b.put(341525, "霍山县");
        b.put(341600, "亳州市");
        b.put(341602, "谯城区");
        b.put(341621, "涡阳县");
        b.put(341622, "蒙城县");
        b.put(341623, "利辛县");
        b.put(341700, "池州市");
        b.put(341702, "贵池区");
        b.put(341721, "东至县");
        b.put(341722, "石台县");
        b.put(341723, "青阳县");
        b.put(341800, "宣城市");
        b.put(341802, "宣州区");
        b.put(341821, "郎溪县");
        b.put(341822, "广德县");
        b.put(341823, "泾县");
        b.put(341824, "绩溪县");
        b.put(341825, "旌德县");
        b.put(341881, "宁国市");
        b.put(350000, "福建省");
        b.put(350100, "福州市");
        b.put(350102, "鼓楼区");
        b.put(350103, "台江区");
        b.put(350104, "仓山区");
        b.put(350105, "马尾区");
        b.put(350111, "晋安区");
        b.put(350112, "长乐区");
        b.put(350121, "闽侯县");
        b.put(350122, "连江县");
        b.put(350123, "罗源县");
        b.put(350124, "闽清县");
        b.put(350125, "永泰县");
        b.put(350128, "平潭县");
        b.put(350181, "福清市");
        b.put(350200, "厦门市");
        b.put(350203, "思明区");
        b.put(350205, "海沧区");
        b.put(350206, "湖里区");
        b.put(350211, "集美区");
        b.put(350212, "同安区");
        b.put(350213, "翔安区");
        b.put(350300, "莆田市");
        b.put(350302, "城厢区");
        b.put(350303, "涵江区");
        b.put(350304, "荔城区");
        b.put(350305, "秀屿区");
        b.put(350322, "仙游县");
        b.put(350400, "三明市");
        b.put(350402, "梅列区");
        b.put(350403, "三元区");
        b.put(350421, "明溪县");
        b.put(350423, "清流县");
        b.put(350424, "宁化县");
        b.put(350425, "大田县");
        b.put(350426, "尤溪县");
        b.put(350427, "沙县");
        b.put(350428, "将乐县");
        b.put(350429, "泰宁县");
        b.put(350430, "建宁县");
        b.put(350481, "永安市");
        b.put(350500, "泉州市");
        b.put(350502, "鲤城区");
        b.put(350503, "丰泽区");
        b.put(350504, "洛江区");
        b.put(350505, "泉港区");
        b.put(350521, "惠安县");
        b.put(350524, "安溪县");
        b.put(350525, "永春县");
        b.put(350526, "德化县");
        b.put(350527, "金门县");
        b.put(350581, "石狮市");
        b.put(350582, "晋江市");
        b.put(350583, "南安市");
        b.put(350600, "漳州市");
        b.put(350602, "芗城区");
        b.put(350603, "龙文区");
        b.put(350622, "云霄县");
        b.put(350623, "漳浦县");
        b.put(350624, "诏安县");
        b.put(350625, "长泰县");
        b.put(350626, "东山县");
        b.put(350627, "南靖县");
        b.put(350628, "平和县");
        b.put(350629, "华安县");
        b.put(350681, "龙海市");
        b.put(350700, "南平市");
        b.put(350702, "延平区");
        b.put(350703, "建阳区");
        b.put(350721, "顺昌县");
        b.put(350722, "浦城县");
        b.put(350723, "光泽县");
        b.put(350724, "松溪县");
        b.put(350725, "政和县");
        b.put(350781, "邵武市");
        b.put(350782, "武夷山市");
        b.put(350783, "建瓯市");
        b.put(350800, "龙岩市");
        b.put(350802, "新罗区");
        b.put(350803, "永定区");
        b.put(350821, "长汀县");
        b.put(350823, "上杭县");
        b.put(350824, "武平县");
        b.put(350825, "连城县");
        b.put(350881, "漳平市");
        b.put(350900, "宁德市");
        b.put(350902, "蕉城区");
        b.put(350921, "霞浦县");
        b.put(350922, "古田县");
        b.put(350923, "屏南县");
        b.put(350924, "寿宁县");
        b.put(350925, "周宁县");
        b.put(350926, "柘荣县");
        b.put(350981, "福安市");
        b.put(350982, "福鼎市");
        b.put(360000, "江西省");
        b.put(360100, "南昌市");
        b.put(360102, "东湖区");
        b.put(360103, "西湖区");
        b.put(360104, "青云谱区");
        b.put(360105, "湾里区");
        b.put(360111, "青山湖区");
        b.put(360112, "新建区");
        b.put(360121, "南昌县");
        b.put(360123, "安义县");
        b.put(360124, "进贤县");
        b.put(360200, "景德镇市");
        b.put(360202, "昌江区");
        b.put(360203, "珠山区");
        b.put(360222, "浮梁县");
        b.put(360281, "乐平市");
        b.put(360300, "萍乡市");
        b.put(360302, "安源区");
        b.put(360313, "湘东区");
        b.put(360321, "莲花县");
        b.put(360322, "上栗县");
        b.put(360323, "芦溪县");
        b.put(360400, "九江市");
        b.put(360402, "濂溪区");
        b.put(360403, "浔阳区");
        b.put(360404, "柴桑区");
        b.put(360423, "武宁县");
        b.put(360424, "修水县");
        b.put(360425, "永修县");
        b.put(360426, "德安县");
        b.put(360428, "都昌县");
        b.put(360429, "湖口县");
        b.put(360430, "彭泽县");
        b.put(360481, "瑞昌市");
        b.put(360482, "共青城市");
        b.put(360483, "庐山市");
        b.put(360500, "新余市");
        b.put(360502, "渝水区");
        b.put(360521, "分宜县");
        b.put(360600, "鹰潭市");
        b.put(360602, "月湖区");
        b.put(360603, "余江区");
        b.put(360681, "贵溪市");
        b.put(360700, "赣州市");
        b.put(360702, "章贡区");
        b.put(360703, "南康区");
        b.put(360704, "赣县区");
        b.put(360722, "信丰县");
        b.put(360723, "大余县");
        b.put(360724, "上犹县");
        b.put(360725, "崇义县");
        b.put(360726, "安远县");
        b.put(360727, "龙南县");
        b.put(360728, "定南县");
        b.put(360729, "全南县");
        b.put(360730, "宁都县");
        b.put(360731, "于都县");
        b.put(360732, "兴国县");
        b.put(360733, "会昌县");
        b.put(360734, "寻乌县");
        b.put(360735, "石城县");
        b.put(360781, "瑞金市");
        b.put(360800, "吉安市");
        b.put(360802, "吉州区");
        b.put(360803, "青原区");
        b.put(360821, "吉安县");
        b.put(360822, "吉水县");
        b.put(360823, "峡江县");
        b.put(360824, "新干县");
        b.put(360825, "永丰县");
        b.put(360826, "泰和县");
        b.put(360827, "遂川县");
        b.put(360828, "万安县");
        b.put(360829, "安福县");
        b.put(360830, "永新县");
        b.put(360881, "井冈山市");
        b.put(360900, "宜春市");
        b.put(360902, "袁州区");
        b.put(360921, "奉新县");
        b.put(360922, "万载县");
        b.put(360923, "上高县");
        b.put(360924, "宜丰县");
        b.put(360925, "靖安县");
        b.put(360926, "铜鼓县");
        b.put(360981, "丰城市");
        b.put(360982, "樟树市");
        b.put(360983, "高安市");
        b.put(361000, "抚州市");
        b.put(361002, "临川区");
        b.put(361003, "东乡区");
        b.put(361021, "南城县");
        b.put(361022, "黎川县");
        b.put(361023, "南丰县");
        b.put(361024, "崇仁县");
        b.put(361025, "乐安县");
        b.put(361026, "宜黄县");
        b.put(361027, "金溪县");
        b.put(361028, "资溪县");
        b.put(361030, "广昌县");
        b.put(361100, "上饶市");
        b.put(361102, "信州区");
        b.put(361103, "广丰区");
        b.put(361121, "上饶县");
        b.put(361123, "玉山县");
        b.put(361124, "铅山县");
        b.put(361125, "横峰县");
        b.put(361126, "弋阳县");
        b.put(361127, "余干县");
        b.put(361128, "鄱阳县");
        b.put(361129, "万年县");
        b.put(361130, "婺源县");
        b.put(361181, "德兴市");
        b.put(370000, "山东省");
        b.put(370100, "济南市");
        b.put(370102, "历下区");
        b.put(370103, "市中区");
        b.put(370104, "槐荫区");
        b.put(370105, "天桥区");
        b.put(370112, "历城区");
        b.put(370113, "长清区");
        b.put(370114, "章丘区");
        b.put(370124, "平阴县");
        b.put(370125, "济阳县");
        b.put(370126, "商河县");
        b.put(370200, "青岛市");
        b.put(370202, "市南区");
        b.put(370203, "市北区");
        b.put(370211, "黄岛区");
        b.put(370212, "崂山区");
        b.put(370213, "李沧区");
        b.put(370214, "城阳区");
        b.put(370215, "即墨区");
        b.put(370281, "胶州市");
        b.put(370283, "平度市");
        b.put(370285, "莱西市");
        b.put(370300, "淄博市");
        b.put(370302, "淄川区");
        b.put(370303, "张店区");
        b.put(370304, "博山区");
        b.put(370305, "临淄区");
        b.put(370306, "周村区");
        b.put(370321, "桓台县");
        b.put(370322, "高青县");
        b.put(370323, "沂源县");
        b.put(370400, "枣庄市");
        b.put(370402, "市中区");
        b.put(370403, "薛城区");
        b.put(370404, "峄城区");
        b.put(370405, "台儿庄区");
        b.put(370406, "山亭区");
        b.put(370481, "滕州市");
        b.put(370500, "东营市");
        b.put(370502, "东营区");
        b.put(370503, "河口区");
        b.put(370505, "垦利区");
        b.put(370522, "利津县");
        b.put(370523, "广饶县");
        b.put(370600, "烟台市");
        b.put(370602, "芝罘区");
        b.put(370611, "福山区");
        b.put(370612, "牟平区");
        b.put(370613, "莱山区");
        b.put(370634, "长岛县");
        b.put(370681, "龙口市");
        b.put(370682, "莱阳市");
        b.put(370683, "莱州市");
        b.put(370684, "蓬莱市");
        b.put(370685, "招远市");
        b.put(370686, "栖霞市");
        b.put(370687, "海阳市");
        b.put(370700, "潍坊市");
        b.put(370702, "潍城区");
        b.put(370703, "寒亭区");
        b.put(370704, "坊子区");
        b.put(370705, "奎文区");
        b.put(370724, "临朐县");
        b.put(370725, "昌乐县");
        b.put(370781, "青州市");
        b.put(370782, "诸城市");
        b.put(370783, "寿光市");
        b.put(370784, "安丘市");
        b.put(370785, "高密市");
        b.put(370786, "昌邑市");
        b.put(370800, "济宁市");
        b.put(370811, "任城区");
        b.put(370812, "兖州区");
        b.put(370826, "微山县");
        b.put(370827, "鱼台县");
        b.put(370828, "金乡县");
        b.put(370829, "嘉祥县");
        b.put(370830, "汶上县");
        b.put(370831, "泗水县");
        b.put(370832, "梁山县");
        b.put(370881, "曲阜市");
        b.put(370883, "邹城市");
        b.put(370900, "泰安市");
        b.put(370902, "泰山区");
        b.put(370911, "岱岳区");
        b.put(370921, "宁阳县");
        b.put(370923, "东平县");
        b.put(370982, "新泰市");
        b.put(370983, "肥城市");
        b.put(371000, "威海市");
        b.put(371002, "环翠区");
        b.put(371003, "文登区");
        b.put(371082, "荣成市");
        b.put(371083, "乳山市");
        b.put(371100, "日照市");
        b.put(371102, "东港区");
        b.put(371103, "岚山区");
        b.put(371121, "五莲县");
        b.put(371122, "莒县");
        b.put(371200, "莱芜市");
        b.put(371202, "莱城区");
        b.put(371203, "钢城区");
        b.put(371300, "临沂市");
        b.put(371302, "兰山区");
        b.put(371311, "罗庄区");
        b.put(371312, "河东区");
        b.put(371321, "沂南县");
        b.put(371322, "郯城县");
        b.put(371323, "沂水县");
        b.put(371324, "兰陵县");
        b.put(371325, "费县");
        b.put(371326, "平邑县");
        b.put(371327, "莒南县");
        b.put(371328, "蒙阴县");
        b.put(371329, "临沭县");
        b.put(371400, "德州市");
        b.put(371402, "德城区");
        b.put(371403, "陵城区");
        b.put(371422, "宁津县");
        b.put(371423, "庆云县");
        b.put(371424, "临邑县");
        b.put(371425, "齐河县");
        b.put(371426, "平原县");
        b.put(371427, "夏津县");
        b.put(371428, "武城县");
        b.put(371481, "乐陵市");
        b.put(371482, "禹城市");
        b.put(371500, "聊城市");
        b.put(371502, "东昌府区");
        b.put(371521, "阳谷县");
        b.put(371522, "莘县");
        b.put(371523, "茌平县");
        b.put(371524, "东阿县");
        b.put(371525, "冠县");
        b.put(371526, "高唐县");
        b.put(371581, "临清市");
        b.put(371600, "滨州市");
        b.put(371602, "滨城区");
        b.put(371603, "沾化区");
        b.put(371621, "惠民县");
        b.put(371622, "阳信县");
        b.put(371623, "无棣县");
        b.put(371625, "博兴县");
        b.put(371626, "邹平县");
        b.put(371700, "菏泽市");
        b.put(371702, "牡丹区");
        b.put(371703, "定陶区");
        b.put(371721, "曹县");
        b.put(371722, "单县");
        b.put(371723, "成武县");
        b.put(371724, "巨野县");
        b.put(371725, "郓城县");
        b.put(371726, "鄄城县");
        b.put(371728, "东明县");
        b.put(410000, "河南省");
        b.put(410100, "郑州市");
        b.put(410102, "中原区");
        b.put(410103, "二七区");
        b.put(410104, "管城回族区");
        b.put(410105, "金水区");
        b.put(410106, "上街区");
        b.put(410108, "惠济区");
        b.put(410122, "中牟县");
        b.put(410181, "巩义市");
        b.put(410182, "荥阳市");
        b.put(410183, "新密市");
        b.put(410184, "新郑市");
        b.put(410185, "登封市");
        b.put(410200, "开封市");
        b.put(410202, "龙亭区");
        b.put(410203, "顺河回族区");
        b.put(410204, "鼓楼区");
        b.put(410205, "禹王台区");
        b.put(410212, "祥符区");
        b.put(410221, "杞县");
        b.put(410222, "通许县");
        b.put(410223, "尉氏县");
        b.put(410225, "兰考县");
        b.put(410300, "洛阳市");
        b.put(410302, "老城区");
        b.put(410303, "西工区");
        b.put(410304, "瀍河回族区");
        b.put(410305, "涧西区");
        b.put(410306, "吉利区");
        b.put(410311, "洛龙区");
        b.put(410322, "孟津县");
        b.put(410323, "新安县");
        b.put(410324, "栾川县");
        b.put(410325, "嵩县");
        b.put(410326, "汝阳县");
        b.put(410327, "宜阳县");
        b.put(410328, "洛宁县");
        b.put(410329, "伊川县");
        b.put(410381, "偃师市");
        b.put(410400, "平顶山市");
        b.put(410402, "新华区");
        b.put(410403, "卫东区");
        b.put(410404, "石龙区");
        b.put(410411, "湛河区");
        b.put(410421, "宝丰县");
        b.put(410422, "叶县");
        b.put(410423, "鲁山县");
        b.put(410425, "郏县");
        b.put(410481, "舞钢市");
        b.put(410482, "汝州市");
        b.put(410500, "安阳市");
        b.put(410502, "文峰区");
        b.put(410503, "北关区");
        b.put(410505, "殷都区");
        b.put(410506, "龙安区");
        b.put(410522, "安阳县");
        b.put(410523, "汤阴县");
        b.put(410526, "滑县");
        b.put(410527, "内黄县");
        b.put(410581, "林州市");
        b.put(410600, "鹤壁市");
        b.put(410602, "鹤山区");
        b.put(410603, "山城区");
        b.put(410611, "淇滨区");
        b.put(410621, "浚县");
        b.put(410622, "淇县");
        b.put(410700, "新乡市");
        b.put(410702, "红旗区");
        b.put(410703, "卫滨区");
        b.put(410704, "凤泉区");
        b.put(410711, "牧野区");
        b.put(410721, "新乡县");
        b.put(410724, "获嘉县");
        b.put(410725, "原阳县");
        b.put(410726, "延津县");
        b.put(410727, "封丘县");
        b.put(410728, "长垣县");
        b.put(410781, "卫辉市");
        b.put(410782, "辉县市");
        b.put(410800, "焦作市");
        b.put(410802, "解放区");
        b.put(410803, "中站区");
        b.put(410804, "马村区");
        b.put(410811, "山阳区");
        b.put(410821, "修武县");
        b.put(410822, "博爱县");
        b.put(410823, "武陟县");
        b.put(410825, "温县");
        b.put(410882, "沁阳市");
        b.put(410883, "孟州市");
        b.put(410900, "濮阳市");
        b.put(410902, "华龙区");
        b.put(410922, "清丰县");
        b.put(410923, "南乐县");
        b.put(410926, "范县");
        b.put(410927, "台前县");
        b.put(410928, "濮阳县");
        b.put(411000, "许昌市");
        b.put(411002, "魏都区");
        b.put(411003, "建安区");
        b.put(411024, "鄢陵县");
        b.put(411025, "襄城县");
        b.put(411081, "禹州市");
        b.put(411082, "长葛市");
        b.put(411100, "漯河市");
        b.put(411102, "源汇区");
        b.put(411103, "郾城区");
        b.put(411104, "召陵区");
        b.put(411121, "舞阳县");
        b.put(411122, "临颍县");
        b.put(411200, "三门峡市");
        b.put(411202, "湖滨区");
        b.put(411203, "陕州区");
        b.put(411221, "渑池县");
        b.put(411224, "卢氏县");
        b.put(411281, "义马市");
        b.put(411282, "灵宝市");
        b.put(411300, "南阳市");
        b.put(411302, "宛城区");
        b.put(411303, "卧龙区");
        b.put(411321, "南召县");
        b.put(411322, "方城县");
        b.put(411323, "西峡县");
        b.put(411324, "镇平县");
        b.put(411325, "内乡县");
        b.put(411326, "淅川县");
        b.put(411327, "社旗县");
        b.put(411328, "唐河县");
        b.put(411329, "新野县");
        b.put(411330, "桐柏县");
        b.put(411381, "邓州市");
        b.put(411400, "商丘市");
        b.put(411402, "梁园区");
        b.put(411403, "睢阳区");
        b.put(411421, "民权县");
        b.put(411422, "睢县");
        b.put(411423, "宁陵县");
        b.put(411424, "柘城县");
        b.put(411425, "虞城县");
        b.put(411426, "夏邑县");
        b.put(411481, "永城市");
        b.put(411500, "信阳市");
        b.put(411502, "浉河区");
        b.put(411503, "平桥区");
        b.put(411521, "罗山县");
        b.put(411522, "光山县");
        b.put(411523, "新县");
        b.put(411524, "商城县");
        b.put(411525, "固始县");
        b.put(411526, "潢川县");
        b.put(411527, "淮滨县");
        b.put(411528, "息县");
        b.put(411600, "周口市");
        b.put(411602, "川汇区");
        b.put(411621, "扶沟县");
        b.put(411622, "西华县");
        b.put(411623, "商水县");
        b.put(411624, "沈丘县");
        b.put(411625, "郸城县");
        b.put(411626, "淮阳县");
        b.put(411627, "太康县");
        b.put(411628, "鹿邑县");
        b.put(411681, "项城市");
        b.put(411700, "驻马店市");
        b.put(411702, "驿城区");
        b.put(411721, "西平县");
        b.put(411722, "上蔡县");
        b.put(411723, "平舆县");
        b.put(411724, "正阳县");
        b.put(411725, "确山县");
        b.put(411726, "泌阳县");
        b.put(411727, "汝南县");
        b.put(411728, "遂平县");
        b.put(411729, "新蔡县");
        b.put(419001, "济源市");
        b.put(420000, "湖北省");
        b.put(420100, "武汉市");
        b.put(420102, "江岸区");
        b.put(420103, "江汉区");
        b.put(420104, "硚口区");
        b.put(420105, "汉阳区");
        b.put(420106, "武昌区");
        b.put(420107, "青山区");
        b.put(420111, "洪山区");
        b.put(420112, "东西湖区");
        b.put(420113, "汉南区");
        b.put(420114, "蔡甸区");
        b.put(420115, "江夏区");
        b.put(420116, "黄陂区");
        b.put(420117, "新洲区");
        b.put(420200, "黄石市");
        b.put(420202, "黄石港区");
        b.put(420203, "西塞山区");
        b.put(420204, "下陆区");
        b.put(420205, "铁山区");
        b.put(420222, "阳新县");
        b.put(420281, "大冶市");
        b.put(420300, "十堰市");
        b.put(420302, "茅箭区");
        b.put(420303, "张湾区");
        b.put(420304, "郧阳区");
        b.put(420322, "郧西县");
        b.put(420323, "竹山县");
        b.put(420324, "竹溪县");
        b.put(420325, "房县");
        b.put(420381, "丹江口市");
        b.put(420500, "宜昌市");
        b.put(420502, "西陵区");
        b.put(420503, "伍家岗区");
        b.put(420504, "点军区");
        b.put(420505, "猇亭区");
        b.put(420506, "夷陵区");
        b.put(420525, "远安县");
        b.put(420526, "兴山县");
        b.put(420527, "秭归县");
        b.put(420528, "长阳土家族自治县");
        b.put(420529, "五峰土家族自治县");
        b.put(420581, "宜都市");
        b.put(420582, "当阳市");
        b.put(420583, "枝江市");
        b.put(420600, "襄阳市");
        b.put(420602, "襄城区");
        b.put(420606, "樊城区");
        b.put(420607, "襄州区");
        b.put(420624, "南漳县");
        b.put(420625, "谷城县");
        b.put(420626, "保康县");
        b.put(420682, "老河口市");
        b.put(420683, "枣阳市");
        b.put(420684, "宜城市");
        b.put(420700, "鄂州市");
        b.put(420702, "梁子湖区");
        b.put(420703, "华容区");
        b.put(420704, "鄂城区");
        b.put(420800, "荆门市");
        b.put(420802, "东宝区");
        b.put(420804, "掇刀区");
        b.put(420822, "沙洋县");
        b.put(420881, "钟祥市");
        b.put(420882, "京山市");
        b.put(420900, "孝感市");
        b.put(420902, "孝南区");
        b.put(420921, "孝昌县");
        b.put(420922, "大悟县");
        b.put(420923, "云梦县");
        b.put(420981, "应城市");
        b.put(420982, "安陆市");
        b.put(420984, "汉川市");
        b.put(421000, "荆州市");
        b.put(421002, "沙市区");
        b.put(421003, "荆州区");
        b.put(421022, "公安县");
        b.put(421023, "监利县");
        b.put(421024, "江陵县");
        b.put(421081, "石首市");
        b.put(421083, "洪湖市");
        b.put(421087, "松滋市");
        b.put(421100, "黄冈市");
        b.put(421102, "黄州区");
        b.put(421121, "团风县");
        b.put(421122, "红安县");
        b.put(421123, "罗田县");
        b.put(421124, "英山县");
        b.put(421125, "浠水县");
        b.put(421126, "蕲春县");
        b.put(421127, "黄梅县");
        b.put(421181, "麻城市");
        b.put(421182, "武穴市");
        b.put(421200, "咸宁市");
        b.put(421202, "咸安区");
        b.put(421221, "嘉鱼县");
        b.put(421222, "通城县");
        b.put(421223, "崇阳县");
        b.put(421224, "通山县");
        b.put(421281, "赤壁市");
        b.put(421300, "随州市");
        b.put(421303, "曾都区");
        b.put(421321, "随县");
        b.put(421381, "广水市");
        b.put(422800, "恩施土家族苗族自治州");
        b.put(422801, "恩施市");
        b.put(422802, "利川市");
        b.put(422822, "建始县");
        b.put(422823, "巴东县");
        b.put(422825, "宣恩县");
        b.put(422826, "咸丰县");
        b.put(422827, "来凤县");
        b.put(422828, "鹤峰县");
        b.put(429004, "仙桃市");
        b.put(429005, "潜江市");
        b.put(429006, "天门市");
        b.put(429021, "神农架林区");
        b.put(430000, "湖南省");
        b.put(430100, "长沙市");
        b.put(430102, "芙蓉区");
        b.put(430103, "天心区");
        b.put(430104, "岳麓区");
        b.put(430105, "开福区");
        b.put(430111, "雨花区");
        b.put(430112, "望城区");
        b.put(430121, "长沙县");
        b.put(430181, "浏阳市");
        b.put(430182, "宁乡市");
        b.put(430200, "株洲市");
        b.put(430202, "荷塘区");
        b.put(430203, "芦淞区");
        b.put(430204, "石峰区");
        b.put(430211, "天元区");
        b.put(430212, "渌口区");
        b.put(430223, "攸县");
        b.put(430224, "茶陵县");
        b.put(430225, "炎陵县");
        b.put(430281, "醴陵市");
        b.put(430300, "湘潭市");
        b.put(430302, "雨湖区");
        b.put(430304, "岳塘区");
        b.put(430321, "湘潭县");
        b.put(430381, "湘乡市");
        b.put(430382, "韶山市");
        b.put(430400, "衡阳市");
        b.put(430405, "珠晖区");
        b.put(430406, "雁峰区");
        b.put(430407, "石鼓区");
        b.put(430408, "蒸湘区");
        b.put(430412, "南岳区");
        b.put(430421, "衡阳县");
        b.put(430422, "衡南县");
        b.put(430423, "衡山县");
        b.put(430424, "衡东县");
        b.put(430426, "祁东县");
        b.put(430481, "耒阳市");
        b.put(430482, "常宁市");
        b.put(430500, "邵阳市");
        b.put(430502, "双清区");
        b.put(430503, "大祥区");
        b.put(430511, "北塔区");
        b.put(430521, "邵东县");
        b.put(430522, "新邵县");
        b.put(430523, "邵阳县");
        b.put(430524, "隆回县");
        b.put(430525, "洞口县");
        b.put(430527, "绥宁县");
        b.put(430528, "新宁县");
        b.put(430529, "城步苗族自治县");
        b.put(430581, "武冈市");
        b.put(430600, "岳阳市");
        b.put(430602, "岳阳楼区");
        b.put(430603, "云溪区");
        b.put(430611, "君山区");
        b.put(430621, "岳阳县");
        b.put(430623, "华容县");
        b.put(430624, "湘阴县");
        b.put(430626, "平江县");
        b.put(430681, "汨罗市");
        b.put(430682, "临湘市");
        b.put(430700, "常德市");
        b.put(430702, "武陵区");
        b.put(430703, "鼎城区");
        b.put(430721, "安乡县");
        b.put(430722, "汉寿县");
        b.put(430723, "澧县");
        b.put(430724, "临澧县");
        b.put(430725, "桃源县");
        b.put(430726, "石门县");
        b.put(430781, "津市市");
        b.put(430800, "张家界市");
        b.put(430802, "永定区");
        b.put(430811, "武陵源区");
        b.put(430821, "慈利县");
        b.put(430822, "桑植县");
        b.put(430900, "益阳市");
        b.put(430902, "资阳区");
        b.put(430903, "赫山区");
        b.put(430921, "南县");
        b.put(430922, "桃江县");
        b.put(430923, "安化县");
        b.put(430981, "沅江市");
        b.put(431000, "郴州市");
        b.put(431002, "北湖区");
        b.put(431003, "苏仙区");
        b.put(431021, "桂阳县");
        b.put(431022, "宜章县");
        b.put(431023, "永兴县");
        b.put(431024, "嘉禾县");
        b.put(431025, "临武县");
        b.put(431026, "汝城县");
        b.put(431027, "桂东县");
        b.put(431028, "安仁县");
        b.put(431081, "资兴市");
        b.put(431100, "永州市");
        b.put(431102, "零陵区");
        b.put(431103, "冷水滩区");
        b.put(431121, "祁阳县");
        b.put(431122, "东安县");
        b.put(431123, "双牌县");
        b.put(431124, "道县");
        b.put(431125, "江永县");
        b.put(431126, "宁远县");
        b.put(431127, "蓝山县");
        b.put(431128, "新田县");
        b.put(431129, "江华瑶族自治县");
        b.put(431200, "怀化市");
        b.put(431202, "鹤城区");
        b.put(431221, "中方县");
        b.put(431222, "沅陵县");
        b.put(431223, "辰溪县");
        b.put(431224, "溆浦县");
        b.put(431225, "会同县");
        b.put(431226, "麻阳苗族自治县");
        b.put(431227, "新晃侗族自治县");
        b.put(431228, "芷江侗族自治县");
        b.put(431229, "靖州苗族侗族自治县");
        b.put(431230, "通道侗族自治县");
        b.put(431281, "洪江市");
        b.put(431300, "娄底市");
        b.put(431302, "娄星区");
        b.put(431321, "双峰县");
        b.put(431322, "新化县");
        b.put(431381, "冷水江市");
        b.put(431382, "涟源市");
        b.put(433100, "湘西土家族苗族自治州");
        b.put(433101, "吉首市");
        b.put(433122, "泸溪县");
        b.put(433123, "凤凰县");
        b.put(433124, "花垣县");
        b.put(433125, "保靖县");
        b.put(433126, "古丈县");
        b.put(433127, "永顺县");
        b.put(433130, "龙山县");
        b.put(440000, "广东省");
        b.put(440100, "广州市");
        b.put(440103, "荔湾区");
        b.put(440104, "越秀区");
        b.put(440105, "海珠区");
        b.put(440106, "天河区");
        b.put(440111, "白云区");
        b.put(440112, "黄埔区");
        b.put(440113, "番禺区");
        b.put(440114, "花都区");
        b.put(440115, "南沙区");
        b.put(440117, "从化区");
        b.put(440118, "增城区");
        b.put(440200, "韶关市");
        b.put(440203, "武江区");
        b.put(440204, "浈江区");
        b.put(440205, "曲江区");
        b.put(440222, "始兴县");
        b.put(440224, "仁化县");
        b.put(440229, "翁源县");
        b.put(440232, "乳源瑶族自治县");
        b.put(440233, "新丰县");
        b.put(440281, "乐昌市");
        b.put(440282, "南雄市");
        b.put(440300, "深圳市");
        b.put(440303, "罗湖区");
        b.put(440304, "福田区");
        b.put(440305, "南山区");
        b.put(440306, "宝安区");
        b.put(440307, "龙岗区");
        b.put(440308, "盐田区");
        b.put(440309, "龙华区");
        b.put(440310, "坪山区");
        b.put(440311, "光明区");
        b.put(440400, "珠海市");
        b.put(440402, "香洲区");
        b.put(440403, "斗门区");
        b.put(440404, "金湾区");
        b.put(440500, "汕头市");
        b.put(440507, "龙湖区");
        b.put(440511, "金平区");
        b.put(440512, "濠江区");
        b.put(440513, "潮阳区");
        b.put(440514, "潮南区");
        b.put(440515, "澄海区");
        b.put(440523, "南澳县");
        b.put(440600, "佛山市");
        b.put(440604, "禅城区");
        b.put(440605, "南海区");
        b.put(440606, "顺德区");
        b.put(440607, "三水区");
        b.put(440608, "高明区");
        b.put(440700, "江门市");
        b.put(440703, "蓬江区");
        b.put(440704, "江海区");
        b.put(440705, "新会区");
        b.put(440781, "台山市");
        b.put(440783, "开平市");
        b.put(440784, "鹤山市");
        b.put(440785, "恩平市");
        b.put(440800, "湛江市");
        b.put(440802, "赤坎区");
        b.put(440803, "霞山区");
        b.put(440804, "坡头区");
        b.put(440811, "麻章区");
        b.put(440823, "遂溪县");
        b.put(440825, "徐闻县");
        b.put(440881, "廉江市");
        b.put(440882, "雷州市");
        b.put(440883, "吴川市");
        b.put(440900, "茂名市");
        b.put(440902, "茂南区");
        b.put(440904, "电白区");
        b.put(440981, "高州市");
        b.put(440982, "化州市");
        b.put(440983, "信宜市");
        b.put(441200, "肇庆市");
        b.put(441202, "端州区");
        b.put(441203, "鼎湖区");
        b.put(441204, "高要区");
        b.put(441223, "广宁县");
        b.put(441224, "怀集县");
        b.put(441225, "封开县");
        b.put(441226, "德庆县");
        b.put(441284, "四会市");
        b.put(441300, "惠州市");
        b.put(441302, "惠城区");
        b.put(441303, "惠阳区");
        b.put(441322, "博罗县");
        b.put(441323, "惠东县");
        b.put(441324, "龙门县");
        b.put(441400, "梅州市");
        b.put(441402, "梅江区");
        b.put(441403, "梅县区");
        b.put(441422, "大埔县");
        b.put(441423, "丰顺县");
        b.put(441424, "五华县");
        b.put(441426, "平远县");
        b.put(441427, "蕉岭县");
        b.put(441481, "兴宁市");
        b.put(441500, "汕尾市");
        b.put(441502, "城区");
        b.put(441521, "海丰县");
        b.put(441523, "陆河县");
        b.put(441581, "陆丰市");
        b.put(441600, "河源市");
        b.put(441602, "源城区");
        b.put(441621, "紫金县");
        b.put(441622, "龙川县");
        b.put(441623, "连平县");
        b.put(441624, "和平县");
        b.put(441625, "东源县");
        b.put(441700, "阳江市");
        b.put(441702, "江城区");
        b.put(441704, "阳东区");
        b.put(441721, "阳西县");
        b.put(441781, "阳春市");
        b.put(441800, "清远市");
        b.put(441802, "清城区");
        b.put(441803, "清新区");
        b.put(441821, "佛冈县");
        b.put(441823, "阳山县");
        b.put(441825, "连山壮族瑶族自治县");
        b.put(441826, "连南瑶族自治县");
        b.put(441881, "英德市");
        b.put(441882, "连州市");
        b.put(441900, "东莞市");
        b.put(442000, "中山市");
        b.put(445100, "潮州市");
        b.put(445102, "湘桥区");
        b.put(445103, "潮安区");
        b.put(445122, "饶平县");
        b.put(445200, "揭阳市");
        b.put(445202, "榕城区");
        b.put(445203, "揭东区");
        b.put(445222, "揭西县");
        b.put(445224, "惠来县");
        b.put(445281, "普宁市");
        b.put(445300, "云浮市");
        b.put(445302, "云城区");
        b.put(445303, "云安区");
        b.put(445321, "新兴县");
        b.put(445322, "郁南县");
        b.put(445381, "罗定市");
        b.put(450000, "广西壮族自治区");
        b.put(450100, "南宁市");
        b.put(450102, "兴宁区");
        b.put(450103, "青秀区");
        b.put(450105, "江南区");
        b.put(450107, "西乡塘区");
        b.put(450108, "良庆区");
        b.put(450109, "邕宁区");
        b.put(450110, "武鸣区");
        b.put(450123, "隆安县");
        b.put(450124, "马山县");
        b.put(450125, "上林县");
        b.put(450126, "宾阳县");
        b.put(450127, "横县");
        b.put(450200, "柳州市");
        b.put(450202, "城中区");
        b.put(450203, "鱼峰区");
        b.put(450204, "柳南区");
        b.put(450205, "柳北区");
        b.put(450206, "柳江区");
        b.put(450222, "柳城县");
        b.put(450223, "鹿寨县");
        b.put(450224, "融安县");
        b.put(450225, "融水苗族自治县");
        b.put(450226, "三江侗族自治县");
        b.put(450300, "桂林市");
        b.put(450302, "秀峰区");
        b.put(450303, "叠彩区");
        b.put(450304, "象山区");
        b.put(450305, "七星区");
        b.put(450311, "雁山区");
        b.put(450312, "临桂区");
        b.put(450321, "阳朔县");
        b.put(450323, "灵川县");
        b.put(450324, "全州县");
        b.put(450325, "兴安县");
        b.put(450326, "永福县");
        b.put(450327, "灌阳县");
        b.put(450328, "龙胜各族自治县");
        b.put(450329, "资源县");
        b.put(450330, "平乐县");
        b.put(450381, "荔浦市");
        b.put(450332, "恭城瑶族自治县");
        b.put(450400, "梧州市");
        b.put(450403, "万秀区");
        b.put(450405, "长洲区");
        b.put(450406, "龙圩区");
        b.put(450421, "苍梧县");
        b.put(450422, "藤县");
        b.put(450423, "蒙山县");
        b.put(450481, "岑溪市");
        b.put(450500, "北海市");
        b.put(450502, "海城区");
        b.put(450503, "银海区");
        b.put(450512, "铁山港区");
        b.put(450521, "合浦县");
        b.put(450600, "防城港市");
        b.put(450602, "港口区");
        b.put(450603, "防城区");
        b.put(450621, "上思县");
        b.put(450681, "东兴市");
        b.put(450700, "钦州市");
        b.put(450702, "钦南区");
        b.put(450703, "钦北区");
        b.put(450721, "灵山县");
        b.put(450722, "浦北县");
        b.put(450800, "贵港市");
        b.put(450802, "港北区");
        b.put(450803, "港南区");
        b.put(450804, "覃塘区");
        b.put(450821, "平南县");
        b.put(450881, "桂平市");
        b.put(450900, "玉林市");
        b.put(450902, "玉州区");
        b.put(450903, "福绵区");
        b.put(450921, "容县");
        b.put(450922, "陆川县");
        b.put(450923, "博白县");
        b.put(450924, "兴业县");
        b.put(450981, "北流市");
        b.put(451000, "百色市");
        b.put(451002, "右江区");
        b.put(451021, "田阳县");
        b.put(451022, "田东县");
        b.put(451023, "平果县");
        b.put(451024, "德保县");
        b.put(451026, "那坡县");
        b.put(451027, "凌云县");
        b.put(451028, "乐业县");
        b.put(451029, "田林县");
        b.put(451030, "西林县");
        b.put(451031, "隆林各族自治县");
        b.put(451081, "靖西市");
        b.put(451100, "贺州市");
        b.put(451102, "八步区");
        b.put(451103, "平桂区");
        b.put(451121, "昭平县");
        b.put(451122, "钟山县");
        b.put(451123, "富川瑶族自治县");
        b.put(451200, "河池市");
        b.put(451202, "金城江区");
        b.put(451203, "宜州区");
        b.put(451221, "南丹县");
        b.put(451222, "天峨县");
        b.put(451223, "凤山县");
        b.put(451224, "东兰县");
        b.put(451225, "罗城仫佬族自治县");
        b.put(451226, "环江毛南族自治县");
        b.put(451227, "巴马瑶族自治县");
        b.put(451228, "都安瑶族自治县");
        b.put(451229, "大化瑶族自治县");
        b.put(451300, "来宾市");
        b.put(451302, "兴宾区");
        b.put(451321, "忻城县");
        b.put(451322, "象州县");
        b.put(451323, "武宣县");
        b.put(451324, "金秀瑶族自治县");
        b.put(451381, "合山市");
        b.put(451400, "崇左市");
        b.put(451402, "江州区");
        b.put(451421, "扶绥县");
        b.put(451422, "宁明县");
        b.put(451423, "龙州县");
        b.put(451424, "大新县");
        b.put(451425, "天等县");
        b.put(451481, "凭祥市");
        b.put(460000, "海南省");
        b.put(460100, "海口市");
        b.put(460105, "秀英区");
        b.put(460106, "龙华区");
        b.put(460107, "琼山区");
        b.put(460108, "美兰区");
        b.put(460200, "三亚市");
        b.put(460202, "海棠区");
        b.put(460203, "吉阳区");
        b.put(460204, "天涯区");
        b.put(460205, "崖州区");
        b.put(460300, "三沙市");
        b.put(460400, "儋州市");
        b.put(469001, "五指山市");
        b.put(469002, "琼海市");
        b.put(469005, "文昌市");
        b.put(469006, "万宁市");
        b.put(469007, "东方市");
        b.put(469021, "定安县");
        b.put(469022, "屯昌县");
        b.put(469023, "澄迈县");
        b.put(469024, "临高县");
        b.put(469025, "白沙黎族自治县");
        b.put(469026, "昌江黎族自治县");
        b.put(469027, "乐东黎族自治县");
        b.put(469028, "陵水黎族自治县");
        b.put(469029, "保亭黎族苗族自治县");
        b.put(469030, "琼中黎族苗族自治县");
        b.put(500000, "重庆市");
        b.put(500101, "万州区");
        b.put(500102, "涪陵区");
        b.put(500103, "渝中区");
        b.put(500104, "大渡口区");
        b.put(500105, "江北区");
        b.put(500106, "沙坪坝区");
        b.put(500107, "九龙坡区");
        b.put(500108, "南岸区");
        b.put(500109, "北碚区");
        b.put(500110, "綦江区");
        b.put(500111, "大足区");
        b.put(500112, "渝北区");
        b.put(500113, "巴南区");
        b.put(500114, "黔江区");
        b.put(500115, "长寿区");
        b.put(500116, "江津区");
        b.put(500117, "合川区");
        b.put(500118, "永川区");
        b.put(500119, "南川区");
        b.put(500120, "璧山区");
        b.put(500151, "铜梁区");
        b.put(500152, "潼南区");
        b.put(500153, "荣昌区");
        b.put(500154, "开州区");
        b.put(500155, "梁平区");
        b.put(500156, "武隆区");
        b.put(500229, "城口县");
        b.put(500230, "丰都县");
        b.put(500231, "垫江县");
        b.put(500233, "忠县");
        b.put(500235, "云阳县");
        b.put(500236, "奉节县");
        b.put(500237, "巫山县");
        b.put(500238, "巫溪县");
        b.put(500240, "石柱土家族自治县");
        b.put(500241, "秀山土家族苗族自治县");
        b.put(500242, "酉阳土家族苗族自治县");
        b.put(500243, "彭水苗族土家族自治县");
        b.put(510000, "四川省");
        b.put(510100, "成都市");
        b.put(510104, "锦江区");
        b.put(510105, "青羊区");
        b.put(510106, "金牛区");
        b.put(510107, "武侯区");
        b.put(510108, "成华区");
        b.put(510112, "龙泉驿区");
        b.put(510113, "青白江区");
        b.put(510114, "新都区");
        b.put(510115, "温江区");
        b.put(510116, "双流区");
        b.put(510117, "郫都区");
        b.put(510121, "金堂县");
        b.put(510129, "大邑县");
        b.put(510131, "蒲江县");
        b.put(510132, "新津县");
        b.put(510181, "都江堰市");
        b.put(510182, "彭州市");
        b.put(510183, "邛崃市");
        b.put(510184, "崇州市");
        b.put(510185, "简阳市");
        b.put(510300, "自贡市");
        b.put(510302, "自流井区");
        b.put(510303, "贡井区");
        b.put(510304, "大安区");
        b.put(510311, "沿滩区");
        b.put(510321, "荣县");
        b.put(510322, "富顺县");
        b.put(510400, "攀枝花市");
        b.put(510402, "东区");
        b.put(510403, "西区");
        b.put(510411, "仁和区");
        b.put(510421, "米易县");
        b.put(510422, "盐边县");
        b.put(510500, "泸州市");
        b.put(510502, "江阳区");
        b.put(510503, "纳溪区");
        b.put(510504, "龙马潭区");
        b.put(510521, "泸县");
        b.put(510522, "合江县");
        b.put(510524, "叙永县");
        b.put(510525, "古蔺县");
        b.put(510600, "德阳市");
        b.put(510603, "旌阳区");
        b.put(510604, "罗江区");
        b.put(510623, "中江县");
        b.put(510681, "广汉市");
        b.put(510682, "什邡市");
        b.put(510683, "绵竹市");
        b.put(510700, "绵阳市");
        b.put(510703, "涪城区");
        b.put(510704, "游仙区");
        b.put(510705, "安州区");
        b.put(510722, "三台县");
        b.put(510723, "盐亭县");
        b.put(510725, "梓潼县");
        b.put(510726, "北川羌族自治县");
        b.put(510727, "平武县");
        b.put(510781, "江油市");
        b.put(510800, "广元市");
        b.put(510802, "利州区");
        b.put(510811, "昭化区");
        b.put(510812, "朝天区");
        b.put(510821, "旺苍县");
        b.put(510822, "青川县");
        b.put(510823, "剑阁县");
        b.put(510824, "苍溪县");
        b.put(510900, "遂宁市");
        b.put(510903, "船山区");
        b.put(510904, "安居区");
        b.put(510921, "蓬溪县");
        b.put(510922, "射洪县");
        b.put(510923, "大英县");
        b.put(511000, "内江市");
        b.put(511002, "市中区");
        b.put(511011, "东兴区");
        b.put(511024, "威远县");
        b.put(511025, "资中县");
        b.put(511083, "隆昌市");
        b.put(511100, "乐山市");
        b.put(511102, "市中区");
        b.put(511111, "沙湾区");
        b.put(511112, "五通桥区");
        b.put(511113, "金口河区");
        b.put(511123, "犍为县");
        b.put(511124, "井研县");
        b.put(511126, "夹江县");
        b.put(511129, "沐川县");
        b.put(511132, "峨边彝族自治县");
        b.put(511133, "马边彝族自治县");
        b.put(511181, "峨眉山市");
        b.put(511300, "南充市");
        b.put(511302, "顺庆区");
        b.put(511303, "高坪区");
        b.put(511304, "嘉陵区");
        b.put(511321, "南部县");
        b.put(511322, "营山县");
        b.put(511323, "蓬安县");
        b.put(511324, "仪陇县");
        b.put(511325, "西充县");
        b.put(511381, "阆中市");
        b.put(511400, "眉山市");
        b.put(511402, "东坡区");
        b.put(511403, "彭山区");
        b.put(511421, "仁寿县");
        b.put(511423, "洪雅县");
        b.put(511424, "丹棱县");
        b.put(511425, "青神县");
        b.put(511500, "宜宾市");
        b.put(511502, "翠屏区");
        b.put(511503, "南溪区");
        b.put(511521, "宜宾县");
        b.put(511523, "江安县");
        b.put(511524, "长宁县");
        b.put(511525, "高县");
        b.put(511526, "珙县");
        b.put(511527, "筠连县");
        b.put(511528, "兴文县");
        b.put(511529, "屏山县");
        b.put(511600, "广安市");
        b.put(511602, "广安区");
        b.put(511603, "前锋区");
        b.put(511621, "岳池县");
        b.put(511622, "武胜县");
        b.put(511623, "邻水县");
        b.put(511681, "华蓥市");
        b.put(511700, "达州市");
        b.put(511702, "通川区");
        b.put(511703, "达川区");
        b.put(511722, "宣汉县");
        b.put(511723, "开江县");
        b.put(511724, "大竹县");
        b.put(511725, "渠县");
        b.put(511781, "万源市");
        b.put(511800, "雅安市");
        b.put(511802, "雨城区");
        b.put(511803, "名山区");
        b.put(511822, "荥经县");
        b.put(511823, "汉源县");
        b.put(511824, "石棉县");
        b.put(511825, "天全县");
        b.put(511826, "芦山县");
        b.put(511827, "宝兴县");
        b.put(511900, "巴中市");
        b.put(511902, "巴州区");
        b.put(511903, "恩阳区");
        b.put(511921, "通江县");
        b.put(511922, "南江县");
        b.put(511923, "平昌县");
        b.put(512000, "资阳市");
        b.put(512002, "雁江区");
        b.put(512021, "安岳县");
        b.put(512022, "乐至县");
        b.put(513200, "阿坝藏族羌族自治州");
        b.put(513201, "马尔康市");
        b.put(513221, "汶川县");
        b.put(513222, "理县");
        b.put(513223, "茂县");
        b.put(513224, "松潘县");
        b.put(513225, "九寨沟县");
        b.put(513226, "金川县");
        b.put(513227, "小金县");
        b.put(513228, "黑水县");
        b.put(513230, "壤塘县");
        b.put(513231, "阿坝县");
        b.put(513232, "若尔盖县");
        b.put(513233, "红原县");
        b.put(513300, "甘孜藏族自治州");
        b.put(513301, "康定市");
        b.put(513322, "泸定县");
        b.put(513323, "丹巴县");
        b.put(513324, "九龙县");
        b.put(513325, "雅江县");
        b.put(513326, "道孚县");
        b.put(513327, "炉霍县");
        b.put(513328, "甘孜县");
        b.put(513329, "新龙县");
        b.put(513330, "德格县");
        b.put(513331, "白玉县");
        b.put(513332, "石渠县");
        b.put(513333, "色达县");
        b.put(513334, "理塘县");
        b.put(513335, "巴塘县");
        b.put(513336, "乡城县");
        b.put(513337, "稻城县");
        b.put(513338, "得荣县");
        b.put(513400, "凉山彝族自治州");
        b.put(513401, "西昌市");
        b.put(513422, "木里藏族自治县");
        b.put(513423, "盐源县");
        b.put(513424, "德昌县");
        b.put(513425, "会理县");
        b.put(513426, "会东县");
        b.put(513427, "宁南县");
        b.put(513428, "普格县");
        b.put(513429, "布拖县");
        b.put(513430, "金阳县");
        b.put(513431, "昭觉县");
        b.put(513432, "喜德县");
        b.put(513433, "冕宁县");
        b.put(513434, "越西县");
        b.put(513435, "甘洛县");
        b.put(513436, "美姑县");
        b.put(513437, "雷波县");
        b.put(520000, "贵州省");
        b.put(520100, "贵阳市");
        b.put(520102, "南明区");
        b.put(520103, "云岩区");
        b.put(520111, "花溪区");
        b.put(520112, "乌当区");
        b.put(520113, "白云区");
        b.put(520115, "观山湖区");
        b.put(520121, "开阳县");
        b.put(520122, "息烽县");
        b.put(520123, "修文县");
        b.put(520181, "清镇市");
        b.put(520200, "六盘水市");
        b.put(520201, "钟山区");
        b.put(520203, "六枝特区");
        b.put(520221, "水城县");
        b.put(520281, "盘州市");
        b.put(520300, "遵义市");
        b.put(520302, "红花岗区");
        b.put(520303, "汇川区");
        b.put(520304, "播州区");
        b.put(520322, "桐梓县");
        b.put(520323, "绥阳县");
        b.put(520324, "正安县");
        b.put(520325, "道真仡佬族苗族自治县");
        b.put(520326, "务川仡佬族苗族自治县");
        b.put(520327, "凤冈县");
        b.put(520328, "湄潭县");
        b.put(520329, "余庆县");
        b.put(520330, "习水县");
        b.put(520381, "赤水市");
        b.put(520382, "仁怀市");
        b.put(520400, "安顺市");
        b.put(520402, "西秀区");
        b.put(520403, "平坝区");
        b.put(520422, "普定县");
        b.put(520423, "镇宁布依族苗族自治县");
        b.put(520424, "关岭布依族苗族自治县");
        b.put(520425, "紫云苗族布依族自治县");
        b.put(520500, "毕节市");
        b.put(520502, "七星关区");
        b.put(520521, "大方县");
        b.put(520522, "黔西县");
        b.put(520523, "金沙县");
        b.put(520524, "织金县");
        b.put(520525, "纳雍县");
        b.put(520526, "威宁彝族回族苗族自治县");
        b.put(520527, "赫章县");
        b.put(520600, "铜仁市");
        b.put(520602, "碧江区");
        b.put(520603, "万山区");
        b.put(520621, "江口县");
        b.put(520622, "玉屏侗族自治县");
        b.put(520623, "石阡县");
        b.put(520624, "思南县");
        b.put(520625, "印江土家族苗族自治县");
        b.put(520626, "德江县");
        b.put(520627, "沿河土家族自治县");
        b.put(520628, "松桃苗族自治县");
        b.put(522300, "黔西南布依族苗族自治州");
        b.put(522301, "兴义市");
        b.put(522322, "兴仁县");
        b.put(522323, "普安县");
        b.put(522324, "晴隆县");
        b.put(522325, "贞丰县");
        b.put(522326, "望谟县");
        b.put(522327, "册亨县");
        b.put(522328, "安龙县");
        b.put(522600, "黔东南苗族侗族自治州");
        b.put(522601, "凯里市");
        b.put(522622, "黄平县");
        b.put(522623, "施秉县");
        b.put(522624, "三穗县");
        b.put(522625, "镇远县");
        b.put(522626, "岑巩县");
        b.put(522627, "天柱县");
        b.put(522628, "锦屏县");
        b.put(522629, "剑河县");
        b.put(522630, "台江县");
        b.put(522631, "黎平县");
        b.put(522632, "榕江县");
        b.put(522633, "从江县");
        b.put(522634, "雷山县");
        b.put(522635, "麻江县");
        b.put(522636, "丹寨县");
        b.put(522700, "黔南布依族苗族自治州");
        b.put(522701, "都匀市");
        b.put(522702, "福泉市");
        b.put(522722, "荔波县");
        b.put(522723, "贵定县");
        b.put(522725, "瓮安县");
        b.put(522726, "独山县");
        b.put(522727, "平塘县");
        b.put(522728, "罗甸县");
        b.put(522729, "长顺县");
        b.put(522730, "龙里县");
        b.put(522731, "惠水县");
        b.put(522732, "三都水族自治县");
        b.put(530000, "云南省");
        b.put(530100, "昆明市");
        b.put(530102, "五华区");
        b.put(530103, "盘龙区");
        b.put(530111, "官渡区");
        b.put(530112, "西山区");
        b.put(530113, "东川区");
        b.put(530114, "呈贡区");
        b.put(530115, "晋宁区");
        b.put(530124, "富民县");
        b.put(530125, "宜良县");
        b.put(530126, "石林彝族自治县");
        b.put(530127, "嵩明县");
        b.put(530128, "禄劝彝族苗族自治县");
        b.put(530129, "寻甸回族彝族自治县");
        b.put(530181, "安宁市");
        b.put(530300, "曲靖市");
        b.put(530302, "麒麟区");
        b.put(530303, "沾益区");
        b.put(530304, "马龙区");
        b.put(530322, "陆良县");
        b.put(530323, "师宗县");
        b.put(530324, "罗平县");
        b.put(530325, "富源县");
        b.put(530326, "会泽县");
        b.put(530381, "宣威市");
        b.put(530400, "玉溪市");
        b.put(530402, "红塔区");
        b.put(530403, "江川区");
        b.put(530422, "澄江县");
        b.put(530423, "通海县");
        b.put(530424, "华宁县");
        b.put(530425, "易门县");
        b.put(530426, "峨山彝族自治县");
        b.put(530427, "新平彝族傣族自治县");
        b.put(530428, "元江哈尼族彝族傣族自治县");
        b.put(530500, "保山市");
        b.put(530502, "隆阳区");
        b.put(530521, "施甸县");
        b.put(530523, "龙陵县");
        b.put(530524, "昌宁县");
        b.put(530581, "腾冲市");
        b.put(530600, "昭通市");
        b.put(530602, "昭阳区");
        b.put(530621, "鲁甸县");
        b.put(530622, "巧家县");
        b.put(530623, "盐津县");
        b.put(530624, "大关县");
        b.put(530625, "永善县");
        b.put(530626, "绥江县");
        b.put(530627, "镇雄县");
        b.put(530628, "彝良县");
        b.put(530629, "威信县");
        b.put(530681, "水富市");
        b.put(530700, "丽江市");
        b.put(530702, "古城区");
        b.put(530721, "玉龙纳西族自治县");
        b.put(530722, "永胜县");
        b.put(530723, "华坪县");
        b.put(530724, "宁蒗彝族自治县");
        b.put(530800, "普洱市");
        b.put(530802, "思茅区");
        b.put(530821, "宁洱哈尼族彝族自治县");
        b.put(530822, "墨江哈尼族自治县");
        b.put(530823, "景东彝族自治县");
        b.put(530824, "景谷傣族彝族自治县");
        b.put(530825, "镇沅彝族哈尼族拉祜族自治县");
        b.put(530826, "江城哈尼族彝族自治县");
        b.put(530827, "孟连傣族拉祜族佤族自治县");
        b.put(530828, "澜沧拉祜族自治县");
        b.put(530829, "西盟佤族自治县");
        b.put(530900, "临沧市");
        b.put(530902, "临翔区");
        b.put(530921, "凤庆县");
        b.put(530922, "云县");
        b.put(530923, "永德县");
        b.put(530924, "镇康县");
        b.put(530925, "双江拉祜族佤族布朗族傣族自治县");
        b.put(530926, "耿马傣族佤族自治县");
        b.put(530927, "沧源佤族自治县");
        b.put(532300, "楚雄彝族自治州");
        b.put(532301, "楚雄市");
        b.put(532322, "双柏县");
        b.put(532323, "牟定县");
        b.put(532324, "南华县");
        b.put(532325, "姚安县");
        b.put(532326, "大姚县");
        b.put(532327, "永仁县");
        b.put(532328, "元谋县");
        b.put(532329, "武定县");
        b.put(532331, "禄丰县");
        b.put(532500, "红河哈尼族彝族自治州");
        b.put(532501, "个旧市");
        b.put(532502, "开远市");
        b.put(532503, "蒙自市");
        b.put(532504, "弥勒市");
        b.put(532523, "屏边苗族自治县");
        b.put(532524, "建水县");
        b.put(532525, "石屏县");
        b.put(532527, "泸西县");
        b.put(532528, "元阳县");
        b.put(532529, "红河县");
        b.put(532530, "金平苗族瑶族傣族自治县");
        b.put(532531, "绿春县");
        b.put(532532, "河口瑶族自治县");
        b.put(532600, "文山壮族苗族自治州");
        b.put(532601, "文山市");
        b.put(532622, "砚山县");
        b.put(532623, "西畴县");
        b.put(532624, "麻栗坡县");
        b.put(532625, "马关县");
        b.put(532626, "丘北县");
        b.put(532627, "广南县");
        b.put(532628, "富宁县");
        b.put(532800, "西双版纳傣族自治州");
        b.put(532801, "景洪市");
        b.put(532822, "勐海县");
        b.put(532823, "勐腊县");
        b.put(532900, "大理白族自治州");
        b.put(532901, "大理市");
        b.put(532922, "漾濞彝族自治县");
        b.put(532923, "祥云县");
        b.put(532924, "宾川县");
        b.put(532925, "弥渡县");
        b.put(532926, "南涧彝族自治县");
        b.put(532927, "巍山彝族回族自治县");
        b.put(532928, "永平县");
        b.put(532929, "云龙县");
        b.put(532930, "洱源县");
        b.put(532931, "剑川县");
        b.put(532932, "鹤庆县");
        b.put(533100, "德宏傣族景颇族自治州");
        b.put(533102, "瑞丽市");
        b.put(533103, "芒市");
        b.put(533122, "梁河县");
        b.put(533123, "盈江县");
        b.put(533124, "陇川县");
        b.put(533300, "怒江傈僳族自治州");
        b.put(533301, "泸水市");
        b.put(533323, "福贡县");
        b.put(533324, "贡山独龙族怒族自治县");
        b.put(533325, "兰坪白族普米族自治县");
        b.put(533400, "迪庆藏族自治州");
        b.put(533401, "香格里拉市");
        b.put(533422, "德钦县");
        b.put(533423, "维西傈僳族自治县");
        b.put(540000, "西藏自治区");
        b.put(540100, "拉萨市");
        b.put(540102, "城关区");
        b.put(540103, "堆龙德庆区");
        b.put(540104, "达孜区");
        b.put(540121, "林周县");
        b.put(540122, "当雄县");
        b.put(540123, "尼木县");
        b.put(540124, "曲水县");
        b.put(540127, "墨竹工卡县");
        b.put(540200, "日喀则市");
        b.put(540202, "桑珠孜区");
        b.put(540221, "南木林县");
        b.put(540222, "江孜县");
        b.put(540223, "定日县");
        b.put(540224, "萨迦县");
        b.put(540225, "拉孜县");
        b.put(540226, "昂仁县");
        b.put(540227, "谢通门县");
        b.put(540228, "白朗县");
        b.put(540229, "仁布县");
        b.put(540230, "康马县");
        b.put(540231, "定结县");
        b.put(540232, "仲巴县");
        b.put(540233, "亚东县");
        b.put(540234, "吉隆县");
        b.put(540235, "聂拉木县");
        b.put(540236, "萨嘎县");
        b.put(540237, "岗巴县");
        b.put(540300, "昌都市");
        b.put(540302, "卡若区");
        b.put(540321, "江达县");
        b.put(540322, "贡觉县");
        b.put(540323, "类乌齐县");
        b.put(540324, "丁青县");
        b.put(540325, "察雅县");
        b.put(540326, "八宿县");
        b.put(540327, "左贡县");
        b.put(540328, "芒康县");
        b.put(540329, "洛隆县");
        b.put(540330, "边坝县");
        b.put(540400, "林芝市");
        b.put(540402, "巴宜区");
        b.put(540421, "工布江达县");
        b.put(540422, "米林县");
        b.put(540423, "墨脱县");
        b.put(540424, "波密县");
        b.put(540425, "察隅县");
        b.put(540426, "朗县");
        b.put(540500, "山南市");
        b.put(540502, "乃东区");
        b.put(540521, "扎囊县");
        b.put(540522, "贡嘎县");
        b.put(540523, "桑日县");
        b.put(540524, "琼结县");
        b.put(540525, "曲松县");
        b.put(540526, "措美县");
        b.put(540527, "洛扎县");
        b.put(540528, "加查县");
        b.put(540529, "隆子县");
        b.put(540530, "错那县");
        b.put(540531, "浪卡子县");
        b.put(540600, "那曲市");
        b.put(540602, "色尼区");
        b.put(540621, "嘉黎县");
        b.put(540622, "比如县");
        b.put(540623, "聂荣县");
        b.put(540624, "安多县");
        b.put(540625, "申扎县");
        b.put(540626, "索县");
        b.put(540627, "班戈县");
        b.put(540628, "巴青县");
        b.put(540629, "尼玛县");
        b.put(540630, "双湖县");
        b.put(542500, "阿里地区");
        b.put(542521, "普兰县");
        b.put(542522, "札达县");
        b.put(542523, "噶尔县");
        b.put(542524, "日土县");
        b.put(542525, "革吉县");
        b.put(542526, "改则县");
        b.put(542527, "措勤县");
        b.put(610000, "陕西省");
        b.put(610100, "西安市");
        b.put(610102, "新城区");
        b.put(610103, "碑林区");
        b.put(610104, "莲湖区");
        b.put(610111, "灞桥区");
        b.put(610112, "未央区");
        b.put(610113, "雁塔区");
        b.put(610114, "阎良区");
        b.put(610115, "临潼区");
        b.put(610116, "长安区");
        b.put(610117, "高陵区");
        b.put(610118, "鄠邑区");
        b.put(610122, "蓝田县");
        b.put(610124, "周至县");
        b.put(610200, "铜川市");
        b.put(610202, "王益区");
        b.put(610203, "印台区");
        b.put(610204, "耀州区");
        b.put(610222, "宜君县");
        b.put(610300, "宝鸡市");
        b.put(610302, "渭滨区");
        b.put(610303, "金台区");
        b.put(610304, "陈仓区");
        b.put(610322, "凤翔县");
        b.put(610323, "岐山县");
        b.put(610324, "扶风县");
        b.put(610326, "眉县");
        b.put(610327, "陇县");
        b.put(610328, "千阳县");
        b.put(610329, "麟游县");
        b.put(610330, "凤县");
        b.put(610331, "太白县");
        b.put(610400, "咸阳市");
        b.put(610402, "秦都区");
        b.put(610403, "杨陵区");
        b.put(610404, "渭城区");
        b.put(610422, "三原县");
        b.put(610423, "泾阳县");
        b.put(610424, "乾县");
        b.put(610425, "礼泉县");
        b.put(610426, "永寿县");
        b.put(610428, "长武县");
        b.put(610429, "旬邑县");
        b.put(610430, "淳化县");
        b.put(610431, "武功县");
        b.put(610481, "兴平市");
        b.put(610482, "彬州市");
        b.put(610500, "渭南市");
        b.put(610502, "临渭区");
        b.put(610503, "华州区");
        b.put(610522, "潼关县");
        b.put(610523, "大荔县");
        b.put(610524, "合阳县");
        b.put(610525, "澄城县");
        b.put(610526, "蒲城县");
        b.put(610527, "白水县");
        b.put(610528, "富平县");
        b.put(610581, "韩城市");
        b.put(610582, "华阴市");
        b.put(610600, "延安市");
        b.put(610602, "宝塔区");
        b.put(610603, "安塞区");
        b.put(610621, "延长县");
        b.put(610622, "延川县");
        b.put(610623, "子长县");
        b.put(610625, "志丹县");
        b.put(610626, "吴起县");
        b.put(610627, "甘泉县");
        b.put(610628, "富县");
        b.put(610629, "洛川县");
        b.put(610630, "宜川县");
        b.put(610631, "黄龙县");
        b.put(610632, "黄陵县");
        b.put(610700, "汉中市");
        b.put(610702, "汉台区");
        b.put(610703, "南郑区");
        b.put(610722, "城固县");
        b.put(610723, "洋县");
        b.put(610724, "西乡县");
        b.put(610725, "勉县");
        b.put(610726, "宁强县");
        b.put(610727, "略阳县");
        b.put(610728, "镇巴县");
        b.put(610729, "留坝县");
        b.put(610730, "佛坪县");
        b.put(610800, "榆林市");
        b.put(610802, "榆阳区");
        b.put(610803, "横山区");
        b.put(610822, "府谷县");
        b.put(610824, "靖边县");
        b.put(610825, "定边县");
        b.put(610826, "绥德县");
        b.put(610827, "米脂县");
        b.put(610828, "佳县");
        b.put(610829, "吴堡县");
        b.put(610830, "清涧县");
        b.put(610831, "子洲县");
        b.put(610881, "神木市");
        b.put(610900, "安康市");
        b.put(610902, "汉滨区");
        b.put(610921, "汉阴县");
        b.put(610922, "石泉县");
        b.put(610923, "宁陕县");
        b.put(610924, "紫阳县");
        b.put(610925, "岚皋县");
        b.put(610926, "平利县");
        b.put(610927, "镇坪县");
        b.put(610928, "旬阳县");
        b.put(610929, "白河县");
        b.put(611000, "商洛市");
        b.put(611002, "商州区");
        b.put(611021, "洛南县");
        b.put(611022, "丹凤县");
        b.put(611023, "商南县");
        b.put(611024, "山阳县");
        b.put(611025, "镇安县");
        b.put(611026, "柞水县");
        b.put(620000, "甘肃省");
        b.put(620100, "兰州市");
        b.put(620102, "城关区");
        b.put(620103, "七里河区");
        b.put(620104, "西固区");
        b.put(620105, "安宁区");
        b.put(620111, "红古区");
        b.put(620121, "永登县");
        b.put(620122, "皋兰县");
        b.put(620123, "榆中县");
        b.put(620200, "嘉峪关市");
        b.put(620300, "金昌市");
        b.put(620302, "金川区");
        b.put(620321, "永昌县");
        b.put(620400, "白银市");
        b.put(620402, "白银区");
        b.put(620403, "平川区");
        b.put(620421, "靖远县");
        b.put(620422, "会宁县");
        b.put(620423, "景泰县");
        b.put(620500, "天水市");
        b.put(620502, "秦州区");
        b.put(620503, "麦积区");
        b.put(620521, "清水县");
        b.put(620522, "秦安县");
        b.put(620523, "甘谷县");
        b.put(620524, "武山县");
        b.put(620525, "张家川回族自治县");
        b.put(620600, "武威市");
        b.put(620602, "凉州区");
        b.put(620621, "民勤县");
        b.put(620622, "古浪县");
        b.put(620623, "天祝藏族自治县");
        b.put(620700, "张掖市");
        b.put(620702, "甘州区");
        b.put(620721, "肃南裕固族自治县");
        b.put(620722, "民乐县");
        b.put(620723, "临泽县");
        b.put(620724, "高台县");
        b.put(620725, "山丹县");
        b.put(620800, "平凉市");
        b.put(620802, "崆峒区");
        b.put(620821, "泾川县");
        b.put(620822, "灵台县");
        b.put(620823, "崇信县");
        b.put(620825, "庄浪县");
        b.put(620826, "静宁县");
        b.put(620881, "华亭市");
        b.put(620900, "酒泉市");
        b.put(620902, "肃州区");
        b.put(620921, "金塔县");
        b.put(620922, "瓜州县");
        b.put(620923, "肃北蒙古族自治县");
        b.put(620924, "阿克塞哈萨克族自治县");
        b.put(620981, "玉门市");
        b.put(620982, "敦煌市");
        b.put(621000, "庆阳市");
        b.put(621002, "西峰区");
        b.put(621021, "庆城县");
        b.put(621022, "环县");
        b.put(621023, "华池县");
        b.put(621024, "合水县");
        b.put(621025, "正宁县");
        b.put(621026, "宁县");
        b.put(621027, "镇原县");
        b.put(621100, "定西市");
        b.put(621102, "安定区");
        b.put(621121, "通渭县");
        b.put(621122, "陇西县");
        b.put(621123, "渭源县");
        b.put(621124, "临洮县");
        b.put(621125, "漳县");
        b.put(621126, "岷县");
        b.put(621200, "陇南市");
        b.put(621202, "武都区");
        b.put(621221, "成县");
        b.put(621222, "文县");
        b.put(621223, "宕昌县");
        b.put(621224, "康县");
        b.put(621225, "西和县");
        b.put(621226, "礼县");
        b.put(621227, "徽县");
        b.put(621228, "两当县");
        b.put(622900, "临夏回族自治州");
        b.put(622901, "临夏市");
        b.put(622921, "临夏县");
        b.put(622922, "康乐县");
        b.put(622923, "永靖县");
        b.put(622924, "广河县");
        b.put(622925, "和政县");
        b.put(622926, "东乡族自治县");
        b.put(622927, "积石山保安族东乡族撒拉族自治县");
        b.put(623000, "甘南藏族自治州");
        b.put(623001, "合作市");
        b.put(623021, "临潭县");
        b.put(623022, "卓尼县");
        b.put(623023, "舟曲县");
        b.put(623024, "迭部县");
        b.put(623025, "玛曲县");
        b.put(623026, "碌曲县");
        b.put(623027, "夏河县");
        b.put(630000, "青海省");
        b.put(630100, "西宁市");
        b.put(630102, "城东区");
        b.put(630103, "城中区");
        b.put(630104, "城西区");
        b.put(630105, "城北区");
        b.put(630121, "大通回族土族自治县");
        b.put(630122, "湟中县");
        b.put(630123, "湟源县");
        b.put(630200, "海东市");
        b.put(630202, "乐都区");
        b.put(630203, "平安区");
        b.put(630222, "民和回族土族自治县");
        b.put(630223, "互助土族自治县");
        b.put(630224, "化隆回族自治县");
        b.put(630225, "循化撒拉族自治县");
        b.put(632200, "海北藏族自治州");
        b.put(632221, "门源回族自治县");
        b.put(632222, "祁连县");
        b.put(632223, "海晏县");
        b.put(632224, "刚察县");
        b.put(632300, "黄南藏族自治州");
        b.put(632321, "同仁县");
        b.put(632322, "尖扎县");
        b.put(632323, "泽库县");
        b.put(632324, "河南蒙古族自治县");
        b.put(632500, "海南藏族自治州");
        b.put(632521, "共和县");
        b.put(632522, "同德县");
        b.put(632523, "贵德县");
        b.put(632524, "兴海县");
        b.put(632525, "贵南县");
        b.put(632600, "果洛藏族自治州");
        b.put(632621, "玛沁县");
        b.put(632622, "班玛县");
        b.put(632623, "甘德县");
        b.put(632624, "达日县");
        b.put(632625, "久治县");
        b.put(632626, "玛多县");
        b.put(632700, "玉树藏族自治州");
        b.put(632701, "玉树市");
        b.put(632722, "杂多县");
        b.put(632723, "称多县");
        b.put(632724, "治多县");
        b.put(632725, "囊谦县");
        b.put(632726, "曲麻莱县");
        b.put(632800, "海西蒙古族藏族自治州");
        b.put(632801, "格尔木市");
        b.put(632802, "德令哈市");
        b.put(632803, "茫崖市");
        b.put(632821, "乌兰县");
        b.put(632822, "都兰县");
        b.put(632823, "天峻县");
        b.put(640000, "宁夏回族自治区");
        b.put(640100, "银川市");
        b.put(640104, "兴庆区");
        b.put(640105, "西夏区");
        b.put(640106, "金凤区");
        b.put(640121, "永宁县");
        b.put(640122, "贺兰县");
        b.put(640181, "灵武市");
        b.put(640200, "石嘴山市");
        b.put(640202, "大武口区");
        b.put(640205, "惠农区");
        b.put(640221, "平罗县");
        b.put(640300, "吴忠市");
        b.put(640302, "利通区");
        b.put(640303, "红寺堡区");
        b.put(640323, "盐池县");
        b.put(640324, "同心县");
        b.put(640381, "青铜峡市");
        b.put(640400, "固原市");
        b.put(640402, "原州区");
        b.put(640422, "西吉县");
        b.put(640423, "隆德县");
        b.put(640424, "泾源县");
        b.put(640425, "彭阳县");
        b.put(640500, "中卫市");
        b.put(640502, "沙坡头区");
        b.put(640521, "中宁县");
        b.put(640522, "海原县");
        b.put(650000, "新疆维吾尔自治区");
        b.put(650100, "乌鲁木齐市");
        b.put(650102, "天山区");
        b.put(650103, "沙依巴克区");
        b.put(650104, "新市区");
        b.put(650105, "水磨沟区");
        b.put(650106, "头屯河区");
        b.put(650107, "达坂城区");
        b.put(650109, "米东区");
        b.put(650121, "乌鲁木齐县");
        b.put(650200, "克拉玛依市");
        b.put(650202, "独山子区");
        b.put(650203, "克拉玛依区");
        b.put(650204, "白碱滩区");
        b.put(650205, "乌尔禾区");
        b.put(650400, "吐鲁番市");
        b.put(650402, "高昌区");
        b.put(650421, "鄯善县");
        b.put(650422, "托克逊县");
        b.put(650500, "哈密市");
        b.put(650502, "伊州区");
        b.put(650521, "巴里坤哈萨克自治县");
        b.put(650522, "伊吾县");
        b.put(652300, "昌吉回族自治州");
        b.put(652301, "昌吉市");
        b.put(652302, "阜康市");
        b.put(652323, "呼图壁县");
        b.put(652324, "玛纳斯县");
        b.put(652325, "奇台县");
        b.put(652327, "吉木萨尔县");
        b.put(652328, "木垒哈萨克自治县");
        b.put(652700, "博尔塔拉蒙古自治州");
        b.put(652701, "博乐市");
        b.put(652702, "阿拉山口市");
        b.put(652722, "精河县");
        b.put(652723, "温泉县");
        b.put(652800, "巴音郭楞蒙古自治州");
        b.put(652801, "库尔勒市");
        b.put(652822, "轮台县");
        b.put(652823, "尉犁县");
        b.put(652824, "若羌县");
        b.put(652825, "且末县");
        b.put(652826, "焉耆回族自治县");
        b.put(652827, "和静县");
        b.put(652828, "和硕县");
        b.put(652829, "博湖县");
        b.put(652900, "阿克苏地区");
        b.put(652901, "阿克苏市");
        b.put(652922, "温宿县");
        b.put(652923, "库车县");
        b.put(652924, "沙雅县");
        b.put(652925, "新和县");
        b.put(652926, "拜城县");
        b.put(652927, "乌什县");
        b.put(652928, "阿瓦提县");
        b.put(652929, "柯坪县");
        b.put(653000, "克孜勒苏柯尔克孜自治州");
        b.put(653001, "阿图什市");
        b.put(653022, "阿克陶县");
        b.put(653023, "阿合奇县");
        b.put(653024, "乌恰县");
        b.put(653100, "喀什地区");
        b.put(653101, "喀什市");
        b.put(653121, "疏附县");
        b.put(653122, "疏勒县");
        b.put(653123, "英吉沙县");
        b.put(653124, "泽普县");
        b.put(653125, "莎车县");
        b.put(653126, "叶城县");
        b.put(653127, "麦盖提县");
        b.put(653128, "岳普湖县");
        b.put(653129, "伽师县");
        b.put(653130, "巴楚县");
        b.put(653131, "塔什库尔干塔吉克自治县");
        b.put(653200, "和田地区");
        b.put(653201, "和田市");
        b.put(653221, "和田县");
        b.put(653222, "墨玉县");
        b.put(653223, "皮山县");
        b.put(653224, "洛浦县");
        b.put(653225, "策勒县");
        b.put(653226, "于田县");
        b.put(653227, "民丰县");
        b.put(654000, "伊犁哈萨克自治州");
        b.put(654002, "伊宁市");
        b.put(654003, "奎屯市");
        b.put(654004, "霍尔果斯市");
        b.put(654021, "伊宁县");
        b.put(654022, "察布查尔锡伯自治县");
        b.put(654023, "霍城县");
        b.put(654024, "巩留县");
        b.put(654025, "新源县");
        b.put(654026, "昭苏县");
        b.put(654027, "特克斯县");
        b.put(654028, "尼勒克县");
        b.put(654200, "塔城地区");
        b.put(654201, "塔城市");
        b.put(654202, "乌苏市");
        b.put(654221, "额敏县");
        b.put(654223, "沙湾县");
        b.put(654224, "托里县");
        b.put(654225, "裕民县");
        b.put(654226, "和布克赛尔蒙古自治县");
        b.put(654300, "阿勒泰地区");
        b.put(654301, "阿勒泰市");
        b.put(654321, "布尔津县");
        b.put(654322, "富蕴县");
        b.put(654323, "福海县");
        b.put(654324, "哈巴河县");
        b.put(654325, "青河县");
        b.put(654326, "吉木乃县");
        b.put(659001, "石河子市");
        b.put(659002, "阿拉尔市");
        b.put(659003, "图木舒克市");
        b.put(659004, "五家渠市");
        b.put(659005, "北屯市");
        b.put(659006, "铁门关市");
        b.put(659007, "双河市");
        b.put(659008, "可克达拉市");
        b.put(659009, "昆玉市");
        b.put(710000, "台湾省");
        b.put(810000, "香港特别行政区");
        b.put(820000, "澳门特别行政区");

        DISTRICT_CODE_MAPPING = b.build();
        AREA_CODE_LIST = new ArrayList<>(DISTRICT_CODE_MAPPING.keySet());
    }

    /* 香港身份首字母对应数字 */
    /*private static final Map<String, Integer> HK_FIRST_CODE = 
        new ImmutableMap.Builder<String, Integer>()
        .put("A", 1)
        .put("B", 2)
        .put("C", 3)
        .put("R", 18)
        .put("U", 21)
        .put("Z", 26)
        .put("X", 24)
        .put("W", 23)
        .put("O", 15)
        .put("N", 14)
        .build();*/

}
