package cn.rl520.druid.config;

import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.jdbc.datasource.lookup.AbstractRoutingDataSource;

import java.lang.reflect.Field;
import java.util.HashMap;
import java.util.Map;

/**
 * @author wwb
 * @version 1.0
 **/

public class RlDynamicDataSource extends AbstractRoutingDataSource implements ApplicationContextAware {

    private static final ThreadLocal<String> contextHolder = new ThreadLocal<>();


    /**
     * 注入Spring 上下文，从中获取DataSource Bean
     */
    private ApplicationContext applicationContext;

    /**
     * 每一次执行SQL语句之前都会执行此方法,以确定本次访问数据库所对应的数据源的key
     * 这里将数据源的bean name作为数据源的key 以方便后续调度数据数据源
     */
    @Override
    protected Object determineCurrentLookupKey() {
        try {
            // 如果没有添加动态数据源，则添加一次。否则不处理
            // 注意,由于系统调用过此方法,targetDataSources的size可能为0也可能为1
            if(getTargetSource().size() == 0 || getTargetSource().size() == 1) {
                Map<Object,Object> dataSources = new HashMap<Object,Object>();
                dataSources.putAll(applicationContext.getBeansOfType(RlDynamicDataSource.class));
                super.setTargetDataSources(dataSources);
                super.afterPropertiesSet();
            }
        } catch (Exception ex) {
            // ignore
        }
        Object key = getDataSource();
        // 如果没有指定数据源,则使用默认的数据源
        if(key == null){
            key = "baseDataSource";
        }
        return key;
    }

    public static void setDataSource(String dataSource) {
        contextHolder.set(dataSource);
    }

    public static String getDataSource() {
        return contextHolder.get();
    }

    public static void clearDataSource() {
        contextHolder.remove();
    }


    /**
     * 通过反射获取AbstractRoutingDataSource的targetDataSources属性
     * @return targetDataSources的值
     */
    @SuppressWarnings("unchecked")
    public Map<Object, Object> getTargetSource() throws NoSuchFieldException, SecurityException, IllegalArgumentException, IllegalAccessException {
        Field field = AbstractRoutingDataSource.class.getDeclaredField("targetDataSources");
        field.setAccessible(true);
        return (Map<Object, Object>) field.get(this);
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }


}
