package cn.sinozg.applet.common.customizer;

import cn.sinozg.applet.common.annotation.DictTrans;
import cn.sinozg.applet.common.annotation.EnumField;
import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.core.model.DictDataVo;
import cn.sinozg.applet.common.enums.DictType;
import cn.sinozg.applet.common.utils.DictUtil;
import cn.sinozg.applet.common.utils.PojoUtil;
import cn.sinozg.applet.common.utils.StringUtil;
import io.swagger.v3.core.converter.AnnotatedType;
import io.swagger.v3.oas.models.media.Schema;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springdoc.core.customizers.PropertyCustomizer;
import org.springframework.stereotype.Component;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

/**
 * 将字段的枚举值写入到 swagger
 * @author xieyubin
 * @Description
 * @Copyright Copyright (c) 2023
 * @since 2023-12-10 12:04
 */
@Component
public class EnumFieldCustomizer implements PropertyCustomizer {

    public static final Logger log = LoggerFactory.getLogger(EnumFieldCustomizer.class);

    @Override
    public Schema<?> customize(Schema property, AnnotatedType type) {
        Annotation[] annotations = type.getCtxAnnotations();
        if (ArrayUtils.isNotEmpty(annotations)) {
            String enumInfo = null;
            for (Annotation a : annotations) {
                if (a instanceof DictTrans) {
                    DictTrans e = (DictTrans) a;
                    enumInfo = dictPair(e);
                    break;
                } else if (a instanceof EnumField) {
                    EnumField e = (EnumField) a;
                    try {
                        enumInfo = enumInfo(e);
                    } catch (Exception ex) {
                        log.error("获取枚举值错误！", ex);
                    }
                }
            }
            if (StringUtils.isNotBlank(enumInfo)) {
                if (StringUtils.isNotBlank(property.getDescription())) {
                    enumInfo = property.getDescription() + "<br/>" + enumInfo;
                }
                property.setDescription(enumInfo);
            }
        }
        return property;
    }

    /**
     * 获取枚举值的键值对
     * @param ef 枚举值
     * @return 信息
     * @throws Exception 异常
     */
    private String enumInfo (EnumField ef) throws Exception {
        StringBuilder info = new StringBuilder(StringUtils.EMPTY);
        Class<?> clazz = ef.clazz();
        if (clazz.isEnum()) {
            Field key = clazz.getDeclaredField(ef.key());
            key.setAccessible(true);
            Field name = clazz.getDeclaredField(ef.name());
            name.setAccessible(true);
            Object[] os = clazz.getEnumConstants();
            Map<String, String> map = new TreeMap<>();
            for (Object o : os) {
                map.put(PojoUtil.cast(key.get(o)), PojoUtil.cast(name.get(o)));
            }
            for (Map.Entry<String, String> entry : map.entrySet()) {
                info.append(entry.getKey()).append(BaseConstants.COLON).append(entry.getValue()).append(StringUtils.SPACE);
            }
        }
        return wrapUp(info, false, false);
    }

    /**
     * dict 键对值
     * @param dict dict
     * @return 数据
     */
    private String dictPair (DictTrans dict){
        StringBuilder info = new StringBuilder(StringUtils.EMPTY);
        DictType type = dict.type();
        boolean error = false;
        if (ArrayUtils.isNotEmpty(dict.pairs())) {
            for (String pair : dict.pairs()) {
                info.append(pair).append(StringUtils.SPACE);
            }
        } else {
            List<DictDataVo> dictList = DictUtil.getDictCache(type);
            if (CollectionUtils.isEmpty(dictList)) {
                info.append("数据字典未定义！ ");
                error = true;
            } else {
                for (DictDataVo d : dictList) {
                    info.append(d.getDictValue()).append(BaseConstants.COLON).append(d.getDictLabel());
                }
            }
        }
        return wrapUp(info, error, true);
    }

    /**
     * 转html 语言
     * @param info 键对值信息
     * @param error 是否错误
     * @return 文本
     */
    private String wrapUp(StringBuilder info, boolean error, boolean isDict){
        String color = String.format("<span><font color=\"%s\">", error ? "red" : "green");
        info.insert(0, color);
        StringUtil.delLast(info);
        info.append("</font></span>");
        if (isDict) {
            info.insert(0, "(");
            info.append(")");
        }
        return info.toString();
    }
}
