package cn.sinozg.applet.common.aspectj;

import cn.sinozg.applet.common.annotation.RateLimiter;
import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.constant.BaseRedisKeys;
import cn.sinozg.applet.common.exception.CavException;
import cn.sinozg.applet.common.utils.OsUtil;
import cn.sinozg.applet.common.utils.PojoWebUtil;
import cn.sinozg.applet.common.utils.WebUtil;
import javax.annotation.Resource;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.redisson.api.RRateLimiter;
import org.redisson.api.RateIntervalUnit;
import org.redisson.api.RateType;
import org.redisson.api.RedissonClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.annotation.Order;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Component;

/**
 * 实现自定义注解对接口的限流
 *
 * @author xyb
 * @Description
 * @Copyright Copyright (c) 2024
 * @since 2024-07-13 21:25
 */
@Order(2)
@Aspect
@Component
public class RateLimiterAspect {
    @Resource
    private RedissonClient redissonClient;

    private final Logger log = LoggerFactory.getLogger(RateLimiterAspect.class);

    @Around("@annotation(rl)")
    public Object around(ProceedingJoinPoint jp, RateLimiter rl) throws Throwable {
        // 获取到 id
        String id = rl.id();
        if (StringUtils.isBlank(id)) {
            id = PojoWebUtil.methodKey(jp, BaseConstants.COLON);
        }
        String ak = rl.key();
        if (StringUtils.isNotBlank(ak)) {
            ak = PojoWebUtil.getElValue(jp, ak);
        }
        if (StringUtils.isBlank(ak)) {
            ak = OsUtil.getHostIp();
        }
        String key = String.format(BaseRedisKeys.RATE_LIMITER, id, ak);
        RRateLimiter limiter = redissonClient.getRateLimiter(key);
        limiter.trySetRate(RateType.OVERALL, rl.limit(), 1, unit(rl));
        // 设置过期时间 加倍
        limiter.expire(rl.unit().duration(2));
        if (!limiter.tryAcquire()) {
            log.error("限流，ID 为{}, key 为 {}", id, ak);
            // 设置HTTP状态码为429 Too Many Requests
            WebUtil.response().setStatus(HttpStatus.TOO_MANY_REQUESTS.value());
            // 阻止执行
            throw new CavException("Too Many Requests");
        }
        // 继续执行目标方法
        return jp.proceed();
    }

    /**
     * 转化枚举值
     * @param anon 注解
     * @return 枚举值
     */
    private RateIntervalUnit unit(RateLimiter anon){
        switch (anon.unit()) {
            case DAYS:
                return RateIntervalUnit.DAYS;
            case HOURS:
                return RateIntervalUnit.HOURS;
            case SECONDS:
                return RateIntervalUnit.SECONDS;
            default:
                return RateIntervalUnit.MINUTES;
        }
    }
}
