/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.utils;

import cn.sinozg.applet.common.annotation.DictTrans;
import cn.sinozg.applet.common.annotation.I18nField;
import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.core.model.I18nName;
import cn.sinozg.applet.common.exception.CavException;
import cn.sinozg.applet.common.recode.DictPairCache;
import jakarta.servlet.http.HttpServletRequest;
import java.lang.reflect.Field;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 字典工具类
 * @Author: xyb
 * @Description:
 * @Date: 2022-11-14 下午 10:03
 **/
public class DictTransUtil {

    private static final Logger log = LoggerFactory.getLogger(DictTransUtil.class);

    private static final Map<String, DictPairCache> DICT_MAP = new ConcurrentHashMap<>();

    private DictTransUtil(){
    }
    /**
     * 数据转义
     * @param value 原始值
     * @param dict 对应的注解
     * @return  转义后的值
     */
    public static String dictTrans (String value, DictTrans dict){
        String targetValue = null;
        if (StringUtils.isNotBlank(value)) {
            if (ArrayUtils.isNotEmpty(dict.pairs())) {
                targetValue = dictPair(value, dict.pairs());
            } else {
                targetValue = DictUtil.getDictLabel(dict.type() ,value);
            }
        }
        return targetValue;
    }

    /**
     * 解析数据转意中自定义的国际化配置
     * @param value 值
     * @param pairs 配置
     * @return 值
     */
    public static String dictPair (String value, String[] pairs) {
        if (ArrayUtils.isNotEmpty(pairs)) {
            DictPairCache cache = dictPairCache(pairs);
            boolean isI18n = cache.i18n();
            Locale locale = null;
            if (isI18n) {
                locale = DictUtil.dictLocale();
            }
            if (locale == null) {
                locale = Locale.CHINA;
            }
            Map<String, I18nName> map = cache.map();
            if (StringUtils.isNotBlank(value)) {
                return I18nUtil.getDictName(map.get(value), locale);
            } else {
                StringBuilder result = new StringBuilder();
                for (Map.Entry<String, I18nName> e : map.entrySet()) {
                    result.append(e.getKey()).append(BaseConstants.COLON);
                    if (e.getValue() != null) {
                        result.append(e.getValue().getName()).append(StringUtils.SPACE);
                    }
                    result.append(StringUtils.SPACE);
                }
                return result.toString();
            }
        }
        return null;
    }

    /**
     * 对对象进行国际化翻译
     * @param t 对象
     * @param request 请求
     * @param <T> 类型
     */
    public static <T> void setBeanI18n(T t, HttpServletRequest request) {
        if (t == null) {
            return;
        }
        if (request == null) {
            request = WebUtil.request();
        }
        List<Field> fields = PojoUtil.allField(t.getClass());
        String name;
        try {
            for (Field field : fields) {
                I18nField fieldAnnotation = field.getAnnotation(I18nField.class);
                if (fieldAnnotation != null) {
                    name = I18nUtil.propertyName(request, fieldAnnotation.value());
                    PojoUtil.setValue(t, field, name);
                }
            }
        } catch (Exception e) {
            log.error("设置值错误。", e);
            throw new CavException("BIZ000100062");
        }
    }

    /**
     * 缓存自定义的 数据字典
     * @param pairs 参数
     * @return 缓存
     */
    private static DictPairCache dictPairCache(String[] pairs) {
        String key = StringUtils.join(pairs, BaseConstants.COMMA);
        if (!DICT_MAP.containsKey(key)) {
            Map<String, I18nName> map = new HashMap<>();
            boolean isI18n = false;
            String k, v;
            for (String pair : pairs) {
                k = StringUtils.substringBefore(pair, BaseConstants.COLON).trim();
                v = StringUtils.substringAfter(pair, BaseConstants.COLON).trim();
                if (!isI18n && StringUtils.startsWith(v, BaseConstants.JSON_START) && StringUtils.endsWith(v, BaseConstants.JSON_END)) {
                    isI18n = true;
                }
                I18nName i18nName;
                if (isI18n) {
                    i18nName = JsonUtil.toPojo(v, I18nName.class);
                } else {
                    i18nName = new I18nName();
                    i18nName.setName(k);
                }
                map.put(k, i18nName);
            }
            DICT_MAP.put(key, new DictPairCache(isI18n, map));
        }
        return DICT_MAP.get(key);
    }
}
