/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.serializer;

import cn.sinozg.applet.common.annotation.Regex;
import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.enums.RegexType;
import cn.sinozg.applet.common.utils.I18nUtil;
import cn.sinozg.applet.common.utils.IdCardUtil;
import jakarta.validation.ConstraintValidator;
import jakarta.validation.ConstraintValidatorContext;
import org.apache.commons.lang3.StringUtils;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 自定义验证注解实现
 * @Author: xyb
 * @Description:
 * @Date: 2023-09-29 下午 04:43
 **/
public class RegexValidator implements ConstraintValidator<Regex, String> {

    private Regex regex;

    @Override
    public void initialize(Regex annotation) {
        regex = annotation;
    }

    @Override
    public boolean isValid(String value, ConstraintValidatorContext context) {
        // 值为空
        if (StringUtils.isBlank(value)) {
            // 必填
            if (regex.required()) {
                setMessage(true, context);
            }
            return !regex.required();
        }
        Pattern pattern = Pattern.compile(regex.type().getRegexp());
        Matcher matcher = pattern.matcher(value);
        boolean result = matcher.matches();
        // 单独 再次校验身份证
        if (result && regex.type() == RegexType.ID_CARD) {
            result = StringUtils.isBlank(IdCardUtil.idCard(value));
        }
        // 返回枚举对应的错误信息
        if (!result) {
            setMessage(false, context);
        }
        return result;
    }

    /**
     * 获取验证信息
     * @param required 是否必填
     * @param context 上下文
     */
    private void setMessage(boolean required, ConstraintValidatorContext context) {
        context.disableDefaultConstraintViolation();
        String message = regex.message();
        if (StringUtils.isNotBlank(message)) {
            // 国际化
            if (StringUtils.startsWith(message, BaseConstants.JSON_START) && StringUtils.endsWith(message, BaseConstants.JSON_END)) {
                message = StringUtils.substringBetween(message, BaseConstants.JSON_START, BaseConstants.JSON_END);
                message = I18nUtil.propertyName(message);
            }
        } else {
            message = regex.type().getMsg() + (required ? "不能为空！" : "格式不正确！");
        }
        context.buildConstraintViolationWithTemplate(message)
                .addConstraintViolation();
    }
}