/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.serializer;

import cn.sinozg.applet.common.annotation.ConditionalField;
import cn.sinozg.applet.common.annotation.ConditionalRequired;
import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.utils.I18nUtil;
import cn.sinozg.applet.common.utils.JsonUtil;
import jakarta.validation.ConstraintValidator;
import jakarta.validation.ConstraintValidatorContext;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Strings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Field;
import java.util.Objects;

/**
 * @author xieyubin
 * @Description
 * @Copyright Copyright (c) 2025
 * @since 2025-07-28 23:39
 */
public class ConditionalRequiredValidator implements ConstraintValidator<ConditionalRequired, Object> {

    private static final Logger log = LoggerFactory.getLogger(ConditionalRequiredValidator.class);
    private ConditionalField[] rules;

    @Override
    public void initialize(ConditionalRequired annotation) {
        this.rules = annotation.value();
    }

    @Override
    public boolean isValid(Object value, ConstraintValidatorContext context) {
        if (value == null) {
            return true;
        }
        Class<?> clazz = value.getClass();
        boolean valid = true;
        for (ConditionalField rule : rules) {
            try {
                Field dependsField = clazz.getDeclaredField(rule.dependsOn());
                dependsField.setAccessible(true);
                Object dependsValue = dependsField.get(value);
                Field field = clazz.getDeclaredField(rule.field());
                field.setAccessible(true);
                Object fieldValue = field.get(value);
                if (ArrayUtils.isEmpty(rule.eq()) && ArrayUtils.isEmpty(rule.ne())) {
                    break;
                }
                boolean needNotBlank = false;
                String depends = Objects.toString(dependsValue, StringUtils.EMPTY);
                if (ArrayUtils.isNotEmpty(rule.eq())) {
                    if (Strings.CS.equalsAny(depends, rule.eq())) {
                        needNotBlank = true;
                    }
                }
                if (ArrayUtils.isNotEmpty(rule.ne())) {
                    if (!Strings.CS.equalsAny(depends, rule.ne())) {
                        needNotBlank = true;
                    }
                }
                if (needNotBlank) {
                    if (fieldValue == null || fieldValue.toString().trim().isEmpty()) {
                        valid = false;
                        setMessage(rule, context);
                    }
                }

            } catch (Exception e) {
                log.error("反射取值异常:", e);
                throw new RuntimeException(e);
            }
        }
        return valid;
    }

    private void setMessage(ConditionalField rule, ConstraintValidatorContext context) {
        context.disableDefaultConstraintViolation();
        String message = rule.message();
        if (StringUtils.isNotBlank(message) && JsonUtil.isJson(message)) {
            message = StringUtils.substringBetween(message, BaseConstants.JSON_START, BaseConstants.JSON_END);
            message = I18nUtil.propertyName(message);
        }
        context.buildConstraintViolationWithTemplate(message)
                .addPropertyNode(rule.field())
                .addConstraintViolation();
    }
}
