package cn.sinozg.applet.common.utils;

import cn.sinozg.applet.common.constant.OmConstants;
import cn.sinozg.applet.common.core.model.JsonNodeInfo;
import cn.sinozg.applet.common.enums.NodeType;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.Function;


/**
* json 工具类
* @Author: xyb
* @Description:
* @Date: 2022-11-14 下午 10:04
**/
public class JsonUtil {

    protected final static ObjectMapper OM = OmConstants.getOm();

    protected static final Logger log = LoggerFactory.getLogger(JsonUtil.class);
    /**
     * 美化json
     *
     * @param obj 对象
     * @return 美化后的json
     */
    public static String printerJson (Object obj){
        try {
            return OM.writerWithDefaultPrettyPrinter().writeValueAsString(obj);
        } catch (JsonProcessingException e) {
            log.error("对象转json错误，{}", e.getMessage());
            log.error("对象转json错误!", e);
        }
        return null;
    }

    /**
     * 转换为 JSON 字符串
     *
     * @param obj 对象
     * @return  json
     */
    public static String toJson(Object obj) {
        try {
            return OM.writeValueAsString(obj);
        } catch (JsonProcessingException e) {
            log.error("对象转json错误，{}", e.getMessage());
            log.error("对象转json错误!", e);
        }
        return null;
    }

    /**
     * json 字符串转为map
     * @param json json
     * @return map
     */
    public static <T> Map<String, T> toMap (String json){
        try {
            return OM.readValue(json, new TypeReference<Map<String, T>>() {});
        } catch (JsonProcessingException e) {
            log.error("json转map错误，{}", e.getMessage());
            log.error("json转map错误!", e);
        }
        return null;
    }

    /**
     * map转为 bean
     * @param map map
     * @param clazz 类型
     * @return bean
     * @param <T> 类型
     */
    public static <T> T mapPojo (Map<String, Object> map, Class<T> clazz){
        if (MapUtils.isEmpty(map) || clazz == null) {
            return null;
        }
        String json = toJson(map);
        return toPojo(json, clazz);
    }

    /**
     * 转换为 JavaBean
     *
     * @param json json
     * @param clazz 类型
     * @return 实体对象
     */
    public static <T> T toPojo(String json, Class<T> clazz) {
        if (StringUtils.isBlank(json)) {
            return null;
        }
        try {
            OM.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
            return OM.readValue(json, clazz);
        } catch (JsonProcessingException e) {
            log.error("json转bean错误，{}", e.getMessage());
            log.error("json转bean错误!", e);
        }
        return null;
    }

    /**
     * 反序列化 嵌套泛型带参数的 示例：<p/>
     * bean 结构如下：<p/>
     * <pre> {@code
     * public class IotResponse<T>
     *     private T data;
     *
     *
     * public class IotPageResponse<T>
     *
     *     private List<T> list;
     *
     * public class IotCardInfoResponse
     * }</pre>
     *
     * 使用方式：<p/>
     * <pre> {@code
     *   Function<ObjectMapper, JavaType> f = om -> {
     *             TypeFactory factory = om.getTypeFactory();
     *             JavaType pageType = factory.constructParametricType(IotPageResponse.class, IotCardInfoResponse.class);
     *             return factory.constructParametricType(IotResponse.class, pageType);
     *   };
     *  IotResponse<IotPageResponse<IotCardInfoResponse>> response = JsonUtil.toPojo(json, f);
     * }</pre>
     *
     * @param json json
     * @param function 获取泛型的函数
     * @return 结果
     * @param <T> 实体
     */
    public static <T> T toPojo(String json, Function<ObjectMapper, JavaType> function){
        if (StringUtils.isBlank(json)) {
            return null;
        }
        try {
            OM.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
            JavaType javaType = function.apply(OM);
            return OM.readValue(json, javaType);
        } catch (JsonProcessingException e) {
            log.error("json转bean错误，{}", e.getMessage());
            log.error("json转bean错误!", e);
        }
        return null;
    }

    /**
     * 转换为 JavaBean 集合
     *
     * @param json json
     * @param clazz 类型
    * @return 对象集合
     */
    public static <T> List<T> toPojos(String json, Class<T> clazz) {
        if (StringUtils.isBlank(json)) {
            return null;
        }
        try {
            JavaType javaType = getCollectionType(ArrayList.class, clazz);
            return OM.readValue(json, javaType);
        } catch (JsonProcessingException e) {
            log.error("json转bean集合错误，{}", e.getMessage());
            log.error("json转bean集合错误!", e);
        }
        return null;
    }

    /**
     * JsonNode 转bean
     * @param node node
     * @param clazz 类型
     * @return 实体
     * @param <T> 实体对象类型
     */
    public static <T> T nodeBean (JsonNode node, Class<T> clazz){
        try {
            OM.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
            return OM.treeToValue(node, clazz);
        } catch (JsonProcessingException e) {
            log.error("jsonNode转bean错误，{}", e.getMessage());
            log.error("jsonNode转bean错误!", e);
        }
        return null;
    }


    /**
     *
     * 获取泛型的 Collection Type
     * @param collectionClass 泛型的Collection
     * @param elementClasses 元素类
     * @return JavaType Java类型
     */
    public static JavaType getCollectionType(Class<?> collectionClass, Class<?>... elementClasses) {
        return OM.getTypeFactory().constructParametricType(collectionClass, elementClasses);
    }
    /**
     * 获取json节点的值
     *
     * @param json json
     * @param key 值
     * @return 对应的值
     */
    public static String key2Val(String json, String key) {
        try {
            JsonNode node = OM.readTree(json);
            if (node != null) {
                node = node.findValue(key);
                if (node != null) {
                    return node.toString();
                }
            }
        } catch (JsonProcessingException e) {
            log.error("获取json节点的值错误，{}", e.getMessage());
        }
        return null;
    }

    /**
     * 获取json节点的值 可以多层获取
     * @param json json
     * @param infos 每层的信息
     * @return 值
     */
    public static String key2Val (String json, List<JsonNodeInfo> infos){
        if (StringUtils.isBlank(json) || CollectionUtils.isEmpty(infos)) {
            return null;
        }
        JsonNode temp = null;
        try {
            JsonNode node = OM.readTree(json);
            if (node != null) {
                temp = nodeValue(node, infos.get(0));
                for (int i = 1, j = infos.size(); i < j; i++) {
                    if (temp == null) {
                        break;
                    }
                    temp = nodeValue(temp, infos.get(i));
                }
            }
        } catch (JsonProcessingException e) {
            log.error("获取json节点的值错误，{}", e.getMessage());
            log.error("获取json节点的值错误!", e);
        }
        if (temp == null) {
            return null;
        }
        return temp.asText();
    }

    /**
     * 获取节点信息
     * @param node 节点
     * @param info 具体信息
     * @return node
     */
    private static JsonNode nodeValue (JsonNode node, JsonNodeInfo info){
        JsonNode result = null;
        if (info.getType() == NodeType.FIND) {
            result = node.findValue(info.getValue());
        } else if (info.getType() == NodeType.AT) {
            result = node.at(info.getValue());
        } else {
            if (node.isArray()) {
                int index = info.getIndex();
                if (index < 0) {
                    index = node.size() + index;
                }
                result = node.get(index);
            }
        }
        return result;
    }

    /**
     * 获取对象
     * @return 对象
     */
    public static ObjectMapper om (){
        return OM;
    }

    /**
     * 对象转 JsonNode
     * @param t 入参
     * @return node节点
     * @param <T> 类型
     */
    public static <T extends Serializable> JsonNode toNode (T t){
        return OM.valueToTree(t);
    }
}
