package cn.sinozg.applet.common.utils;

import cn.sinozg.applet.common.enums.ChronoDate;

import java.time.DayOfWeek;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.Period;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoField;
import java.time.temporal.ChronoUnit;
import java.time.temporal.Temporal;
import java.time.temporal.TemporalAccessor;
import java.time.temporal.TemporalAdjuster;
import java.time.temporal.TemporalAdjusters;
import java.time.temporal.TemporalQuery;
import java.time.temporal.TemporalUnit;
import java.time.temporal.ValueRange;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
* 时间工具类
* @Author: xyb
* @Description: 
* @Date: 2022-11-14 下午 10:03
**/
public class DateUtil {
    public static final String YYYY = "yyyy";

    public static final String YYYY_MM = "yyyy-MM";

    public static final String YYYY_MM_DD = "yyyy-MM-dd";

    public static final String YYYYMMDD = "yyyyMMdd";

    public static final String YYYYMMDDHH = "yyyyMMddHH";

    public static final String YYYYMMDDHHMM = "yyyyMMddHHmm";

    public static final String YYYYMMDDHHMMSS = "yyyyMMddHHmmss";

    /** 端名称 */
    public static final String MDHSS = "MMddHHmmssSSS";

    public static final String YYYY_MM_DD_HH_MM_SS = "yyyy-MM-dd HH:mm:ss";

    public static final String HHMM = "HHmm";

    public static final String HH_MM = "HH:mm";

    public static final String HHMMSS = "HHmmss";

    public static final String HH_MM_SS = "HH:mm:ss";

    public static final String[] YMD_PATTERNS = {YYYY_MM_DD, "yyyy/MM/dd", "yyyy.MM.dd", YYYYMMDD};

    public static final String[] YMDHMS_PATTERNS = {YYYY_MM_DD_HH_MM_SS, "yyyy/MM/dd HH:mm:ss", "yyyy.MM.dd HH:mm:ss", YYYYMMDDHHMMSS};

    public static final String[] HMS_PATTERNS = {HH_MM_SS, HHMMSS};

    /** 北京时区 */
    public static final ZoneOffset ZONE_BJ = ZoneOffset.of("+8");

    public static String getDateTime () {
        return formatDateTime(LocalDateTime.now());
    }

    /**
     * 获取当前日期, 默认格式为yyyy-MM-dd
     *
     * @return String
     */
    public static String getDate() {
        return formatDate(LocalDate.now(), YYYY_MM_DD);
    }

    public static String getTime (){
        return formatTime(LocalTime.now(), HH_MM_SS);
    }


    /**
     * 根据 ymd 增加几天后返回
     * @param ymd 日期
     * @param daysToAdd 偏移量
     * @param unit 年月日
     * @return 下一天
     */
    public static String nextDate (String ymd, int daysToAdd, TemporalUnit unit){
        LocalDate date = parseDate(ymd).plus(daysToAdd, unit);
        if (unit == ChronoUnit.WEEKS || unit == ChronoUnit.MONTHS || unit == ChronoUnit.YEARS) {
            date = date.plusDays(daysToAdd < 0 ? 1 : -1);
        }
        return formatDate(date, YYYY_MM_DD);
    }

    /**
     * 将字符串 转为时间
     *
     * @param input   字符串
     * @return 时间
     */
    public static LocalDateTime parseDateTime(String input) {
        return parseDateTime(input, YYYY_MM_DD_HH_MM_SS);
    }
    /**
     * 将字符串 转为时间
     *
     * @param input   字符串
     * @param pattern 格式化
     * @return 时间
     */
    public static LocalDateTime parseDateTime(String input, String pattern) {
        return parse(pattern, input, LocalDateTime::from);
    }

    public static LocalDate parseDate(String input) {
        return parseDate(input, YYYY_MM_DD);
    }

    public static LocalDate parseDate(String input, String pattern) {
        return parse(pattern, input, LocalDate::from);
    }

    public static LocalTime parseTime(String input) {
        return parseTime(input, HH_MM_SS);
    }
    public static LocalTime parseTime(String input, String pattern) {
        return parse(pattern, input, LocalTime::from);
    }

    public static LocalDateTime ymdToLdt(String input){
        input = input + " 00:00:00";
        return parseDateTime(input);
    }
    public static LocalDateTime ymdToLdtEnd(String input){
        input = input + " 23:59:59";
        return parseDateTime(input);
    }

    /**
     * 讲秒转为时间差 显示为 *天*小时*分钟*秒
     * @param seconds 秒
     * @return 时间差
     */
    public static String poorSeconds(int seconds){
        LocalDateTime ldt = LocalDateTime.now();
        return getDatePoor(ldt, ldt.plusSeconds(seconds));
    }

    /**
     * 计算两个时间差 显示为 *天*小时*分钟*秒
     * @param startDate 开始日期
     * @param endDate 结束日期
     * @return 时间差
     */
    public static String getDatePoor(LocalDateTime startDate, LocalDateTime endDate) {
        if (startDate == null || endDate == null) {
            return null;
        }
        Duration duration = Duration.between(startDate, endDate);
        if (duration.isNegative()) {
            return null;
        }
        // 如果时分秒小于开始时间 要减少一天
        LocalDate endLd = endDate.toLocalDate();
        if (startDate.toLocalTime().isAfter(endDate.toLocalTime())) {
            endLd = endLd.minusDays(1);
        }
        Period period = Period.between(startDate.toLocalDate(), endLd);
        StringBuilder poor = new StringBuilder();
        appendPoor(poor, period.getYears(), "年");
        appendPoor(poor, period.getMonths(), "月");
        appendPoor(poor, period.getDays(), "天");

        appendPoor(poor,duration.toHours() % 24, "小时");
        appendPoor(poor,duration.toMinutes() % 60, "分钟");
        double second = ArithUtil.div(duration.toMillis(), 1000);
        second = ArithUtil.round(second, 0);
        appendPoor(poor, (long) (second % 60), "秒");
        return poor.toString();
    }

    /**
     * 拼接时间差
     * @param poor 字符串
     * @param num 数量
     * @param unit 单位
     */
    private static void appendPoor (StringBuilder poor, long num, String unit){
        if (num > 0) {
            poor.append(num).append(unit);
        }
    }

    /**
     * 计算两个时间差
     */
    public static long diffTime(LocalDateTime startDate, LocalDateTime endDate) {
        Duration duration = Duration.between(startDate, endDate);
        return duration.toMillis();
    }

    /**
     * 判断 target 是否大于source
     * 只考虑了 正常的字符串日期 ，后期完善
     * @param source 原始
     * @param target 目标
     * @return 是否相等
     */
    public static boolean compareTime (Object source, Object target){
        // 为空或者类型不一致直接返回
        if (source == null || !source.getClass().isInstance(target)) {
            return false;
        }
        if (source instanceof String) {
            String s = (String)source;
            String t = (String)target;
            Temporal t1 = null;
            Temporal t2 = null;
            if (t.length() == s.length()) {
                if (t.length()  == YYYY_MM_DD_HH_MM_SS.length()) {
                    t1 = parseDateTime(s);
                    t2 = parseDateTime(t);
                } else if (t.length()  == YYYY_MM_DD.length()) {
                    t1 = parseDate(s);
                    t2 = parseDate(t);
                } else if (t.length()  == HH_MM_SS.length()) {
                    t1 = parseTime(s);
                    t2 = parseTime(t);
                }
                return compare(t1, t2);
            }
        }
        return compare(source, target);
    }

    /**
     * 时间比较
     * @param source 原始
     * @param target 目标
     * @return 是否相等
     */
    private static boolean compare (Object source, Object target) {
        if (target == null || !(source instanceof Temporal)) {
            return false;
        }
        // 时间
        if (source instanceof LocalDateTime) {
            return ((LocalDateTime)source).isBefore((LocalDateTime)target);
        } else if (source instanceof LocalDate) {
            return ((LocalDate)source).isBefore((LocalDate)target);
        } else if (source instanceof LocalTime) {
            return ((LocalTime)source).isBefore((LocalTime)target);
        }
        return false;
    }

    /**
     * 计算年纪
     * @param birthDay 生日
     * @return 年纪
     */
    public static int getAge(LocalDateTime birthDay){
        //出生日期晚于当前时间，无法计算
        LocalDateTime now = LocalDateTime.now();
        if (birthDay == null || birthDay.isAfter(now)) {
            throw new IllegalArgumentException("The birthDay is before Now.It's unbelievable!");
        }
        return (int) birthDay.until(now, ChronoUnit.YEARS);
    }



    /**
     * 当前时间向推几小时
     *
     * @param hour 小时
     * @return String
     */
    public static String getBeforeByHourTime(int hour) {
        LocalDateTime ldt = LocalDateTime.now();
        ldt = ldt.minusHours(hour);
        return formatDateTime(ldt, null);
    }


    /**
     * 当前时间向推N天 n 为正数
     *
     * @param day 小时
     * @return String  YYYY_MM_DD_HH_MM_SS
     */
    public static String getBeforeByDayTime(int day) {
        LocalDateTime now = LocalDateTime.now();
        now = now.minusDays(day);
        return formatDateTime(now, null);
    }

    /**
     * 当前时间向推N天
     * 获取时间戳
     *
     * @param day 小时
     * @return String
     */
    public static Long getBeforeByDayLongTime(int day) {
        LocalDateTime now = LocalDateTime.now();
        now = now.minusDays(-day);
        return now.toEpochSecond(ZONE_BJ);
    }

    /**
     * 毫秒数据
     * @param ldt 日期
     * @return 时间
     */
    public static Long epochMilli (LocalDateTime ldt){
        return epochMilli(ldt, ZONE_BJ);
    }
    /**
     * 毫秒数据
     * @param ldt 日期
     * @return 毫秒数据
     */
    public static Long epochMilli (LocalDateTime ldt, ZoneOffset offset){
        if (ldt == null) {
            return 0L;
        }
        return ldt.toInstant(offset).toEpochMilli();
    }
    public static LocalDateTime toLdt(long milli){
        return toLdt(milli, ZoneOffset.ofHours(8));
    }

    /**
     * 处理腾讯穿过来的 时间
     * @param milli 毫秒
     * @return 时间
     */
    public static LocalDateTime ldt (long milli){
        return toLdt(milli, ZoneOffset.ofHours(8));
    }

    public static LocalDateTime toLdt(long milli, ZoneId zoneId){
        return Instant.ofEpochMilli(milli).atZone(zoneId).toLocalDateTime();
    }

    public static String formatDateTime(LocalDateTime dateTime) {
        return formatDateTime(dateTime, YYYY_MM_DD_HH_MM_SS);
    }

    /**
     * 按pattern格式化时间-默认yyyy-MM-dd HH:mm:ss格式
     *
     * @param dateTime LocalDateTime对象
     * @param pattern  要格式化的字符串
     * @return 格式化日期
     */
    public static String formatDateTime(LocalDateTime dateTime, String pattern) {
        return format(Objects.toString(pattern, YYYY_MM_DD_HH_MM_SS), dateTime);
    }
    public static String formatDate(LocalDate dateTime) {
        return formatDate(dateTime, YYYY_MM_DD);
    }
    public static String formatDate(LocalDate dateTime, String pattern) {
        return format(Objects.toString(pattern, YYYY_MM_DD), dateTime);
    }

    public static String formatTime(LocalTime dateTime) {
        return formatTime(dateTime, HH_MM_SS);
    }

    public static String formatTime(LocalTime dateTime, String pattern) {
        return format(Objects.toString(pattern, HH_MM_SS), dateTime);
    }


    /**
     * 格式化日期
     * @param pattern 格式
     * @param temporal 日期信息
     * @return 日期
     */
    private static String format(String pattern, TemporalAccessor temporal){
        if (temporal == null) {
            return null;
        }
        return formatter(pattern).format(temporal);
    }

    /**
     * 转为日期
     * @param pattern 格式
     * @param input 输入
     * @param query query
     * @return 对应的日期
     * @param <T> 日期
     */
    private static <T> T parse (String pattern, String input, TemporalQuery<T> query){
        return formatter(pattern).parse(input, query);
    }

    /**
     * 获取到 DateTimeFormatter
     * @param pattern pattern
     * @return DateTimeFormatter
     */
    public static DateTimeFormatter formatter(String pattern){
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern(pattern);
        Objects.requireNonNull(formatter, "formatter");
        return formatter;
    }

    /**
     * 获取日期间隔内所有的日期
     * @param beginTime 开始日期
     * @param endTime 结束日期
     * @return 所有的天数
     */
    public static List<String> getAllLds(String beginTime, String endTime) {
        LocalDate startDate = parseDate(beginTime);
        LocalDate endDate = parseDate(endTime);
        List<String> dates = new ArrayList<>();
        while (!endDate.isBefore(startDate)) {
            dates.add(formatDate(startDate, null));
            startDate = startDate.plusDays(1);
        }
        return dates;
    }

    /**
     * 根据时间 算取对应的开始日期或者结束时间
     * @param input 时间
     * @param amountToAdd 偏移的量
     * @param chrono 获取的类型
     * @param begin 是否为开始时间
     * @return 开始时间或者结束时间
     */
    public static LocalDateTime beginOrEndDateTime (LocalDateTime input, long amountToAdd, ChronoDate chrono, boolean begin){
        if (input == null || chrono == null) {
            return null;
        }
        LocalDate localTime = input.toLocalDate();
        localTime = beginOrEndDate(localTime, amountToAdd, chrono, begin);
        return LocalDateTime.of(localTime, begin ? LocalTime.MIN : LocalTime.MAX);
    }
    /**
     * 根据类型获取日期的结束时间
     * @param input 时间
     * @param chrono 类型
     * @return 结束时间
     */
    public static LocalDate endDate (LocalDate input, ChronoDate chrono){
        return beginOrEndDate(input, 0, chrono, false);
    }
    /**
     * 根据类型获取日期的开始时间
     * @param input 时间
     * @param chrono 类型
     * @return 开始时间
     */
    public static LocalDate beginDate (LocalDate input, ChronoDate chrono){
        return beginOrEndDate(input, 0, chrono, true);
    }

    /**
     * 根据时间 算取对应的开始日期或者结束时间
     * @param input 时间
     * @param amountToAdd 偏移的量
     * @param chrono 获取的类型
     * @param begin 是否为开始时间
     * @return 开始时间或者结束时间
     */
    public static LocalDate beginOrEndDate (LocalDate input, long amountToAdd, ChronoDate chrono, boolean begin) {
        if (input == null || chrono == null) {
            return null;
        }
        TemporalAdjuster adjuster = null;
        ChronoUnit unit = null;
        switch (chrono) {
            // 年
            case YEARS:
                unit = ChronoUnit.YEARS;
                adjuster = begin ? TemporalAdjusters.firstDayOfYear() : TemporalAdjusters.lastDayOfYear();
                break;
            // 上半年
            case FIRST_HALF_YEAR:
                unit = ChronoUnit.YEARS;
                adjuster = begin ? TemporalAdjusters.firstDayOfYear() :
                        temporal -> temporal.with(ChronoField.DAY_OF_YEAR, 1)
                                .plus(6, ChronoUnit.MONTHS)
                                .plus(-1, ChronoUnit.MONTHS);
                break;
            // 下半年
            case NEXT_HALF_YEAR:
                unit = ChronoUnit.YEARS;
                adjuster = !begin ? TemporalAdjusters.lastDayOfYear() :
                        temporal -> temporal.with(ChronoField.DAY_OF_YEAR, 1)
                                .plus(6, ChronoUnit.MONTHS);
                break;
            // 季度
            case QUARTER:
                unit = ChronoUnit.MONTHS;
                amountToAdd *= 3;
                adjuster = temporal -> {
                    int month = temporal.get(ChronoField.MONTH_OF_YEAR);
                    int monthDiff = (begin ? 2 : 0) - (month + 2) % 3;
                    Temporal t = temporal.plus(monthDiff, ChronoUnit.MONTHS);
                    ValueRange range = t.range(ChronoField.DAY_OF_MONTH);
                    return t.with(ChronoField.DAY_OF_MONTH, begin ? range.getMinimum() : range.getMaximum());
                };
                break;
            // 月
            case MONTHS:
                unit = ChronoUnit.MONTHS;
                adjuster = begin ? TemporalAdjusters.firstDayOfMonth() : TemporalAdjusters.lastDayOfMonth();
                break;
            // 周
            case WEEKS:
                unit = ChronoUnit.WEEKS;
                adjuster = temporal -> {
                    int daysDiff = DayOfWeek.MONDAY.getValue() - temporal.get(ChronoField.DAY_OF_WEEK);
                    daysDiff = begin ? daysDiff : daysDiff + 6;
                    return temporal.plus(daysDiff, ChronoUnit.DAYS);
                };
                break;
            // 天
            case DAYS:
                unit = ChronoUnit.DAYS;
                adjuster = temporal -> temporal;
                break;
            default:
        }
        return input.plus(amountToAdd, unit).with(adjuster);
    }
}
