package cn.sinozg.applet.common.utils;

import java.math.BigDecimal;
import java.math.RoundingMode;


/**
* 精确的浮点数运算
* @Author: xyb
* @Description:
* @Date: 2022-11-14 下午 09:52
**/
public class ArithUtil {

    /**
     * 默认除法运算精度
     */
    private static final int DEF_DIV_SCALE = 10;

    private static final int ZERO = 0;

    /**
     * 这个类不能实例化
     */
    private ArithUtil() {
    }

    /**
     * 提供精确的加法运算。
     *
     * @param ds 被加数
     * @return 多个参数的和
     */
    public static double add(double ... ds) {
        BigDecimal bd = new BigDecimal(ZERO);
        for (double d : ds) {
            bd = bd.add(new BigDecimal(Double.toString(d)));
        }
        return bd.doubleValue();
    }

    /**
     * 提供精确的减法运算。
     *
     * @param ds 被减数
     * @return 多个参数的差
     */
    public static double sub(double ... ds) {
        BigDecimal bd = new BigDecimal(String.valueOf(ds[ZERO]));
        int t = ds.length;
        for (int i = 1; i < t; i++) {
            bd = bd.subtract(new BigDecimal(Double.toString(ds[i])));
        }
        return bd.doubleValue();
    }

    /**
     * 提供精确的乘法运算。
     *
     * @param ds 被乘数
     * @return 多个参数的积
     */
    public static double mul(double ... ds) {
        BigDecimal bd = BigDecimal.ONE;
        for (double d : ds) {
            bd = bd.multiply(new BigDecimal(Double.toString(d)));
        }
        return bd.doubleValue();
    }

    /**
     * 提供（相对）精确的除法运算，当发生除不尽的情况时，精确到
     * 小数点以后10位，以后的数字四舍五入。
     *
     * @param ds 被除数
     * @return 两多个参数的商
     */
    public static double div(double ... ds) {
        return div(DEF_DIV_SCALE, ds);
    }

    /**
     * 提供（相对）精确的除法运算。当发生除不尽的情况时，由scale参数指
     * 定精度，以后的数字四舍五入。
     * @param scale 表示表示需要精确到小数点以后几位。
     * @param ds    除数
     * @return 两个参数的商
     */
    public static double div(int scale, double ... ds) {
        if (scale < ZERO) {
            throw new IllegalArgumentException("需要精确到小数点以后几位的值不正确，请检查！");
        }
        int t = ds.length;
        BigDecimal bd = BigDecimal.valueOf(ds[ZERO]);
        for (int i = 1; i < t; i++) {
            if (ds[i] == ZERO) {
                return ZERO;
            }
            bd = bd.divide(BigDecimal.valueOf(ds[i]), scale, RoundingMode.HALF_DOWN);
        }
        return bd.doubleValue();
    }

    /**
     * 提供精确的小数位四舍五入处理。
     *
     * @param v     需要四舍五入的数字
     * @param scale 小数点后保留几位
     * @return 四舍五入后的结果
     */
    public static double round(double v, int scale) {
        return round(v, scale, RoundingMode.HALF_UP);
    }

    /**
     * 提供精确的小数位四舍五入处理。
     * @param v 需要四舍五入的数字
     * @param scale 小数点后保留几位
     * @param roundingMode 取舍类型
     * @return 结果
     */
    public static double round (double v, int scale, RoundingMode roundingMode){
        if (scale < ZERO) {
            throw new IllegalArgumentException("需要精确到小数点以后几位的值不正确，请检查！");
        }
        if (roundingMode == null) {
            throw new IllegalArgumentException("处理类型不能为空，请检查！");
        }
        BigDecimal b = BigDecimal.valueOf(v);
        return b.divide(BigDecimal.ONE, scale, roundingMode).doubleValue();
    }
}
