package cn.sinozg.applet.common.utils;

import cn.sinozg.applet.common.constant.OmConstants;
import cn.sinozg.applet.common.core.model.JsonNodeInfo;
import cn.sinozg.applet.common.core.model.MapJsonTemp;
import cn.sinozg.applet.common.enums.NodeType;
import cn.sinozg.applet.common.exception.CavException;
import cn.sinozg.applet.common.function.FunctionException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.Function;


/**
* json 工具类
* @Author: xyb
* @Description:
* @Date: 2022-11-14 下午 10:04
**/
public class JsonUtil {

    protected final static ObjectMapper OM = OmConstants.getOm();

    protected static final Logger log = LoggerFactory.getLogger(JsonUtil.class);

    /**
     * 美化json
     *
     * @param obj 对象
     * @return 美化后的json
     */
    public static String printerJson (Object obj){
        return tryParse(m -> m.writerWithDefaultPrettyPrinter().writeValueAsString(obj));
    }

    /**
     * 转换为 JSON 字符串
     *
     * @param obj 对象
     * @return  json
     */
    public static String toJson(Object obj) {
        return tryParse(m -> m.writeValueAsString(obj));
    }

    /**
     * json 字符串转为map
     * @param json json
     * @return map
     */
    public static Map<String, Object> toMap (String json){
        Object value = tryParse(m -> m.readValue(json, Map.class));
        return PojoUtil.cast(value);
    }

    /**
     * json 字符串转为map
     * value的类型一样
     * @param json json
     * @param clazz value 类型
     * @return map
     */
    public static <T> Map<String, T> toMap (String json, Class<T> clazz){
        if (StringUtils.isNotEmpty(json) && clazz != null) {
            json = "{\"map\":" + json + "}";
            MapJsonTemp<T> temp = toPojo(json, om -> om.getTypeFactory().constructParametricType(MapJsonTemp.class, clazz));
            if (temp != null) {
                return temp.getMap();
            }
        }
        return null;
    }

    /**
     * map转为 bean
     * @param map map
     * @param clazz 类型
     * @return bean
     * @param <T> 类型
     */
    public static <T> T mapPojo (Map<String, Object> map, Class<T> clazz){
        if (MapUtils.isEmpty(map) || clazz == null) {
            return null;
        }
        String json = toJson(map);
        return toPojo(json, clazz);
    }

    /**
     * 转换为 JavaBean
     *
     * @param json json
     * @param clazz 类型
     * @return 实体对象
     */
    public static <T> T toPojo(String json, Class<T> clazz) {
        if (StringUtils.isBlank(json)) {
            return null;
        }
        return tryParse(m -> {
            m.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
            return m.readValue(json, clazz);
        });
    }

    /**
     * 反序列化 嵌套泛型带参数的 示例：<p/>
     * bean 结构如下：<p/>
     * <pre> {@code
     * public class IotResponse<T>
     *     private T data;
     *
     *
     * public class IotPageResponse<T>
     *
     *     private List<T> list;
     *
     * public class IotCardInfoResponse
     * }</pre>
     *
     * 使用方式：<p/>
     * <pre> {@code
     *   Function<ObjectMapper, JavaType> f = om -> {
     *             TypeFactory factory = om.getTypeFactory();
     *             JavaType pageType = factory.constructParametricType(IotPageResponse.class, IotCardInfoResponse.class);
     *             return factory.constructParametricType(IotResponse.class, pageType);
     *   };
     *  IotResponse<IotPageResponse<IotCardInfoResponse>> response = JsonUtil.toPojo(json, f);
     * }</pre>
     *
     * @param json json
     * @param function 获取泛型的函数
     * @return 结果
     * @param <T> 实体
     */
    public static <T> T toPojo(String json, Function<ObjectMapper, JavaType> function){
        if (StringUtils.isBlank(json)) {
            return null;
        }
        return tryParse(m -> {
            m.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
            JavaType javaType = function.apply(m);
            return m.readValue(json, javaType);
        });
    }

    /**
     * 转换为 JavaBean 集合
     *
     * @param json json
     * @param clazz 类型
    * @return 对象集合
     */
    public static <T> List<T> toPojos(String json, Class<T> clazz) {
        if (StringUtils.isBlank(json)) {
            return null;
        }
        return tryParse(m -> {
            JavaType javaType = getCollectionType(ArrayList.class, clazz);
            return m.readValue(json, javaType);
        });
    }

    /**
     * JsonNode 转bean
     * @param node node
     * @param clazz 类型
     * @return 实体
     * @param <T> 实体对象类型
     */
    public static <T> T nodeBean (JsonNode node, Class<T> clazz){
        return tryParse(m -> {
            m.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
            return m.treeToValue(node, clazz);
        });
    }

    /**
     *
     * 获取泛型的 Collection Type
     * @param collectionClass 泛型的Collection
     * @param elementClasses 元素类
     * @return JavaType Java类型
     */
    public static JavaType getCollectionType(Class<?> collectionClass, Class<?>... elementClasses) {
        return OM.getTypeFactory().constructParametricType(collectionClass, elementClasses);
    }

    /**
     * 获取json节点的值
     *
     * @param json json
     * @param key 值
     * @return 对应的值
     */
    public static String key2Val(String json, String key) {
        return tryParse(m -> {
            JsonNode node = m.readTree(json);
            if (node != null) {
                node = node.findValue(key);
                if (node != null) {
                    return node.toString();
                }
            }
            return null;
        });
    }

    /**
     * 获取json节点的值 可以多层获取
     * 最后节点是最后节点才可以
     * @param json json
     * @param infos 每层的信息
     * @return 值
     */
    public static String key2Val (String json, List<JsonNodeInfo> infos) {
        JsonNode node = key2Node(json, infos);
        if (node != null) {
            return node.asText();
        }
        return null;
    }

    /**
     * 获取到json节点的值 可以多层获取
     * @param json json
     * @param infos 每层的信息
     * @return 节点
     */
    public static JsonNode key2Node (String json, List<JsonNodeInfo> infos) {
        if (StringUtils.isBlank(json) || CollectionUtils.isEmpty(infos)) {
            return null;
        }
        return tryParse(m -> {
            JsonNode node = m.readTree(json);
            JsonNode n = null;
            if (node != null) {
                n = nodeValue(node, infos.get(0));
                for (int i = 1, j = infos.size(); i < j; i++) {
                    if (n == null) {
                        break;
                    }
                    n = nodeValue(n, infos.get(i));
                }
            }
            return n;
        });
    }

    /**
     * 获取节点信息
     * @param node 节点
     * @param info 具体信息
     * @return node
     */
    private static JsonNode nodeValue (JsonNode node, JsonNodeInfo info){
        JsonNode result = null;
        if (info.getType() == NodeType.FIND) {
            result = node.findValue(info.getValue());
        } else if (info.getType() == NodeType.AT) {
            result = node.at(info.getValue());
        } else {
            if (node.isArray()) {
                int index = info.getIndex();
                if (index < 0) {
                    index = node.size() + index;
                }
                result = node.get(index);
            }
        }
        return result;
    }

    /**
     * 获取对象
     * @return 对象
     */
    public static ObjectMapper om (){
        return OM;
    }

    /**
     * 对象转 JsonNode
     * @param t 入参
     * @return node节点
     * @param <T> 类型
     */
    public static <T extends Serializable> JsonNode toNode (T t){
        return OM.valueToTree(t);
    }

    /**
     * 解析json
     * @param parse 执行函数
     * @return 返回数据
     * @param <T> 返回类型
     */
    public static <T> T tryParse(FunctionException<ObjectMapper, T> parse) {
        return tryParse(parse, OM);
    }

    /**
     * 解析json
     * @param parse 执行函数
     * @param om om
     * @return 返回数据
     * @param <T> 返回类型
     */
    protected static <T> T tryParse(FunctionException<ObjectMapper, T> parse, ObjectMapper om) {
        try {
            return parse.apply(om);
        } catch (Exception e) {
            log.error("json 处理异常!", e);
            throw new CavException("BIZ000100040");
        }
    }
}
