package cn.sinozg.applet.common.utils;

import cn.sinozg.applet.common.core.ThreadFun;
import cn.sinozg.applet.common.handler.ThreadPoolExecutorHandler;
import com.alibaba.ttl.threadpool.TtlExecutors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.time.Duration;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;


/**
 * 线程池对象
 * @Author: xyb
 * @Description
 * @Date: 2022/8/15 20:38
 */
public class ThreadPool {

    private static final int SHUT_TIME = 120;
    private static final Logger log = LoggerFactory.getLogger(ThreadPool.class);
    private ThreadPool(){
    }
    /**
     * 多线程下可以获取用户信息的 线程池
     */
    private static final ExecutorService TTL_SERVICE = TtlExecutors.getTtlExecutorService(new ThreadPoolExecutorHandler("biz-ttl-pool"));

    /**
     * 创建线程池 普通线程池
     */
    private static final ThreadPoolExecutor SINGLE_THREAD_POOL = new ThreadPoolExecutorHandler("common-pool", (r, executor) -> {
        try {
            executor.getQueue().put(r);
        } catch (InterruptedException e) {
            log.error("初始化普通线程错误！", e);
        }
    });

    public static void submit(Runnable runnable){
        SINGLE_THREAD_POOL.submit(runnable);
    }

    public static void execute (Runnable runnable){
        SINGLE_THREAD_POOL.execute(runnable);
    }

    /**
     * 可以在子线程里面获取到用户信息
     * @param r 线程方法
     */
    public static void ttlExecute (Runnable r){
        assert TTL_SERVICE != null;
        TTL_SERVICE.execute(r);
    }

    /**
     * 停止线程池
     * 先使用shutdown, 停止接收新任务并尝试完成所有已存在任务.
     * 如果超时, 则调用shutdownNow, 取消在workQueue中Pending的任务,并中断所有阻塞函数.
     * 如果仍人超時，則強制退出.
     * 另对在shutdown时线程本身被调用中断做了处理.
     */
    public static void shutdownAndAwaitTermination() {
        shutdownAndAwaitTermination(SINGLE_THREAD_POOL);
    }

    /**
     * 停止线程池
     * 先使用shutdown, 停止接收新任务并尝试完成所有已存在任务.
     * 如果超时, 则调用shutdownNow, 取消在workQueue中Pending的任务,并中断所有阻塞函数.
     * 如果仍人超時，則強制退出.
     * 另对在shutdown时线程本身被调用中断做了处理.
     */
    public static void shutdownAndAwaitTermination(ExecutorService pool) {
        if (pool != null && !pool.isShutdown()) {
            pool.shutdown();
            try {
                if (!pool.awaitTermination(SHUT_TIME, TimeUnit.SECONDS)) {
                    pool.shutdownNow();
                    if (!pool.awaitTermination(SHUT_TIME, TimeUnit.SECONDS)) {
                        log.info("Pool did not terminate");
                    }
                }
            } catch (InterruptedException ie) {
                log.error("线程闭关失败！", ie);
                pool.shutdownNow();
                Thread.currentThread().interrupt();
            }
        }
    }

    /**
     * 打印线程异常信息
     */
    public static void printException(Runnable r, Throwable t) {
        if (t == null && r instanceof Future<?> future) {
            try {
                if (future.isDone()) {
                    future.get();
                }
            } catch (CancellationException ce) {
                t = ce;
            } catch (ExecutionException ee) {
                t = ee.getCause();
            } catch (InterruptedException ie) {
                // ignore/reset
                Thread.currentThread().interrupt();
            }
        }
        if (t != null) {
            log.error(t.getMessage(), t);
        }
    }

    /**
     * 创建一个 延迟线程池
     * @param poolName 名称
     * @return 延迟线程池
     */
    public static ScheduledThreadPoolExecutor scheduledThread(String poolName){
        return new ScheduledThreadPoolExecutor(1, ThreadFun.threadFactory(poolName));
    }

    /**
     * 执行延迟任务 下次任务开始时间以上次任务结束时间开始 加上延迟的间隔时间
     * @param executor 线程池
     * @param command 任务
     * @param initialDelay 延迟时间 秒
     * @param delay 延迟间隔时间
     * @return 返回结果
     */
    public static ScheduledFuture<?> scheduleWithFixedDelay(ScheduledThreadPoolExecutor executor, Runnable command, Duration initialDelay, Duration delay){
        return executor.scheduleWithFixedDelay(command, initialDelay.getSeconds(), delay.getSeconds(), TimeUnit.SECONDS);
    }

    /**
     * 执行延迟任务 任务执行时间超过间隔时间的 直接执行
     * @param executor 线程池
     * @param command 任务
     * @param initialDelay 延迟时间
     * @param period 间隔时间
     * @return 返回结果
     */
    public static ScheduledFuture<?> scheduleAtFixedRate(ScheduledThreadPoolExecutor executor,Runnable command, Duration initialDelay, Duration period){
        return executor.scheduleAtFixedRate(command, initialDelay.getSeconds(), period.getSeconds(), TimeUnit.SECONDS);
    }
}
