/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.handler;

import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.core.ThreadFun;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;


/**
 * 重写线程池 打印当前的线程信息
 * @Author: xyb
 * @Description:
 * @Date: 2023-04-28 下午 01:59
 **/
public class ThreadPoolExecutorHandler extends ThreadPoolExecutor {
    /** 线程池名称 */
    private final String poolName;
    /** 线程池维护线程所允许的空闲时间 */
    private static final long KEEP_ALIVE_TIME = 300;

    /** 队列最大长度 */
    private static final int QUEUE_CAPACITY = 8192;
    private static final Logger log = LoggerFactory.getLogger(BaseConstants.BIZ_CUSTOM_LOG);

    /**
     * 初始化线程池
     * 核心数默认为 cpu+1 对列数为 8192 拒绝策略为抛出异常
     *
     * @param poolName 线程名称
     */
    public ThreadPoolExecutorHandler(String poolName){
        this(poolName, new ThreadPoolExecutor.CallerRunsPolicy());
    }

    /**
     * 初始化线程池
     * 核心数默认为 cpu+1 对列数为 8192
     * @param poolName 线程名称
     * @param handler 拒绝策略
     */
    public ThreadPoolExecutorHandler(String poolName, RejectedExecutionHandler handler){
        this(ThreadFun.CORE_POOL_SIZE + 1, ThreadFun.CORE_POOL_SIZE + 1, KEEP_ALIVE_TIME, TimeUnit.SECONDS, new LinkedBlockingDeque<>(QUEUE_CAPACITY), poolName, handler);
    }

    /**
     * 初始化线程池
     * @param corePoolSize 核心数
     * @param maximumPoolSize 最大核心数
     * @param keepAliveTime 空闲时间
     * @param unit 空闲时间单位
     * @param workQueue 对列
     * @param poolName 线程名称
     * @param handler 拒绝策略
     */
    public ThreadPoolExecutorHandler(int corePoolSize, int maximumPoolSize, long keepAliveTime, @NotNull TimeUnit unit, @NotNull BlockingQueue<Runnable> workQueue, String poolName, RejectedExecutionHandler handler) {
        super(corePoolSize, maximumPoolSize, keepAliveTime, unit, workQueue, ThreadFun.threadFactory(poolName), handler);
        this.poolName = poolName;
    }

    /**
     * 打印线程池运行状态
     * @param t the thread that will run task {@code r}
     * @param r the task that will be executed
     */
    @Override
    protected void beforeExecute(Thread t, Runnable r) {
        if (log.isDebugEnabled()) {
            BlockingQueue<Runnable> queue = this.getQueue();
            log.debug("{}: 初始线程数: {}, 核心线程数: {}, 正在执行的任务数量: {}, 池中存在的最大线程数: {}, 最大允许的线程数: {}," +
                            "已完成任务数量: {}, 历史任务总数: {}, 队列里缓存的任务数量: {}, 队列剩余大小：{}",
                    this.poolName, this.getPoolSize(), this.getCorePoolSize(), this.getActiveCount(), this.getLargestPoolSize(), this.getMaximumPoolSize(),
                    this.getCompletedTaskCount(), this.getTaskCount(), queue.size(), queue.remainingCapacity());
        }
        super.beforeExecute(t, r);
    }
}