/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.utils;

import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.handler.SslSocket;
import java.nio.charset.Charset;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.X509TrustManager;
import okhttp3.Call;
import okhttp3.Callback;
import okhttp3.ConnectionPool;
import okhttp3.FormBody;
import okhttp3.Headers;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;
import okhttp3.ResponseBody;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * HTTP 请求工具类<p/>
 * 封装okhttp
 * @Author: xyb
 * @Description:
 * @Date: 2023-03-22 下午 03:28
 **/
public class HttpUtil {
    /** 读取超时时间毫秒 */
    private static final int READ_TIMEOUT = 10;
    /** 写数据超时时间毫秒 */
    private static final int WRITE_TIMEOUT = 30;
    /** 连接超时时间 */
    private static final int CONNECT_TIMEOUT = 30;
    private static volatile OkHttpClient client = null;
    private static volatile OkHttpClient ssl_client = null;
    private static volatile OkHttpClient client302 = null;
    private static final int MAX_IDLE_CONNECTION = 5;
    private static final long KEEP_ALIVE_DURATION = 5;

    private static final Logger log = LoggerFactory.getLogger(HttpUtil.class);
    private HttpUtil() {
    }

    /**
     * 获取单例的okhttp client对象，并配置通用信息
     *
     * @return 客户端实例
     */
    public static OkHttpClient getInstance() {
        if (client == null) {
            synchronized (HttpUtil.class) {
                if (client == null) {
                    client = build(false, true);
                }
            }
        }
        return client;
    }

    /**
     * 获取单例的okhttp client对象，并配置通用信息
     * 没有越过证书
     * @return 客户端实例
     */
    public static OkHttpClient sslInstance() {
        if (ssl_client == null) {
            synchronized (HttpUtil.class) {
                if (ssl_client == null) {
                    ssl_client = build(false, false);
                }
            }
        }
        return ssl_client;
    }

    /**
     * 获取单例的okhttp client对象，用于302请求
     *
     * @return 客户端实例
     */
    public static OkHttpClient getInstance302() {
        if (client302 == null) {
            synchronized (HttpUtil.class) {
                if (client302 == null) {
                    client302 = build(true, true);
                }
            }
        }
        return client302;
    }


    /**
     * 请求 302
     * @param url 地址
     * @param headerMap 头map
     * @return 返回location
     */
    public static String location302(String url, Map<String, String> headerMap){
        Request.Builder builder = builder(url, headerMap);
        Call call = getInstance302().newCall(builder.get().build());
        Headers headers = responseBody(call, 3);
        if (headers != null) {
            return headers.get("Location");
        }
        return null;
    }
    /**
     * 同步GET请求
     *
     * @param url 请求地址
     * @return 返回get结果
     */
    public static byte[] getBytes(String url) {
        Call call = baseGetCall(url, null);
        return responseBody(call, 2);
    }

    /**
     * 同步GET请求
     *
     * @param url 请求地址
     * @return 返回get结果
     */
    public static String doGet(String url) {
        return doGet(url, null);
    }

    /**
     * 同步GET请求
     *
     * @param url 请求地址
     * @param headerMap 请求头
     * @return 返回get结果
     */
    public static String doGet(String url, Map<String, String> headerMap) {
        Call call = baseGetCall(url, headerMap);
        return responseBody(call, 1);
    }

    /**
     * 获取网络文件
     * @param url 地址
     * @param headerMap 头
     * @return 文件名称 文件字节
     */
    public static ImmutablePair<String, byte[]> getFile (String url, Map<String, String> headerMap){
        Call call = baseGetCall(url, headerMap);
        return responseBody(call, 4);
    }

    /**
     * 异步GET请求
     *
     * @param url 请求地址
     * @param callback 回调
     */
    public static void doGetAsync(String url, Callback callback) {
        doGetAsync(url, null, callback);
    }

    /**
     * 异步GET请求
     *
     * @param url 请求地址
     * @param headerMap 请求头
     * @param callback 回调
     */
    public static void doGetAsync(String url, Map<String, String> headerMap, Callback callback) {
        Call call = baseGetCall(url, headerMap);
        call.enqueue(callback);
    }


    /**
     * json 请求
     * @param url 参数
     * @param headerMap 请求头
     * @param r 入参
     * @param clazz 出参
     * @return 结果
     * @param <T> 出参类型
     * @param <R> 入参类型
     */
    public static <T, R> T doPost(String url, Map<String, String> headerMap, R r, Class<T> clazz) {
        String jsonParams = JsonUtil.toJson(r);
        String result = doPost(url, headerMap, jsonParams);
        if (StringUtils.isNotBlank(result)) {
            return JsonUtil.toPojo(result, clazz);
        }
        return null;
    }

    /**
     * 同步POST请求 form
     *
     * @param url 请求地址
     * @param mapParams 请求参数
     * @return post结果
     */
    public static String doPost(String url, Map<String, String> mapParams) {
        return doPost(url, null, mapParams);
    }
    /**
     * 同步POST请求 form
     *
     * @param url 请求地址
     * @param headerMap 请求头
     * @param mapParams 请求参数
     * @return post结果
     */
    public static String doPost(String url, Map<String, String> headerMap, Map<String, String> mapParams) {
        Call call = basePostCallForm(url, headerMap, mapParams);
        return responseBody(call, 1);
    }


    /**
     * 同步POST请求 json
     *
     * @param url 请求地址
     * @param headerMap 请求头
     * @param jsonParams json参数
     * @return post结果
     */
    public static String doPost(String url, Map<String, String> headerMap, String jsonParams) {
        Call call = basePostCallJson(url, headerMap, jsonParams, false);
        return responseBody(call, 1);
    }

    /**
     * 同步POST请求 强制使用 json请求 参数为非json
     *
     * @param url 请求地址
     * @param headerMap 请求头
     * @param params 非json参数
     * @return post结果
     */
    public static String doPostJson(String url, Map<String, String> headerMap, String params) {
        Call call = basePostCallJson(url, headerMap, params, true);
        return responseBody(call, 1);
    }


    /**
     * 同步POST请求
     *
     * @param url 请求地址
     * @param params 请求参数
     * @return post结果
     */
    public static String doPost(String url, String params) {
        return doPost(url, null, params);
    }

    /**
     * 异步POST请求
     *
     * @param url 请求地址
     * @param headerMap 请求头
     * @param mapParams nap请求参数
     * @param callback 回调
     */
    public static void doPostAsync(String url, Map<String, String> headerMap, Map<String, String> mapParams, Callback callback) {
        Call call = basePostCallForm(url, headerMap, mapParams);
        call.enqueue(callback);
    }


    /**
     * 异步POST请求
     *
     * @param url 请求地址
     * @param mapParams 请求参数
     * @param callback 回调
     */
    public static void doPostAsync(String url, Map<String, String> mapParams, Callback callback) {
        doPostAsync(url, null, mapParams, callback);
    }

    /**
     * 异步POST请求
     *
     * @param url 请求地址
     * @param headerMap 请求头
     * @param jsonParams json请求参数
     * @param callback 回调
     */
    public static void doPostAsync(String url, Map<String, String> headerMap, String jsonParams, Callback callback) {
        Call call = basePostCallJson(url, headerMap, jsonParams, false);
        call.enqueue(callback);
    }

    /**
     * 异步POST请求
     *
     * @param url 请求地址
     * @param jsonParams json参数
     * @param callback 回调
     */
    public static void doPostAsync(String url, String jsonParams, Callback callback) {
        doPostAsync(url, null, jsonParams, callback);
    }

    /**
     * 构建客户端
     * @param skipSsl 是否跳过证书
     * @param c 其他参数
     * @return 客户端
     */
    public static OkHttpClient build (boolean skipSsl, Consumer<OkHttpClient.Builder> c){
        return build(false, skipSsl, c);
    }

    /**
     * 构建实例
     * @param redirects 是否
     * @param skipSsl 是否忽略证书
     * @return 实例
     */
    private static OkHttpClient build (boolean redirects, boolean skipSsl){
        return build(redirects, skipSsl, null);
    }

    /**
     * 构建实例
     * @param redirects 是否
     * @param skipSsl 是否忽略证书
     * @param c 其他需要构建的参数
     * @return 实例
     */
    private static OkHttpClient build (boolean redirects, boolean skipSsl, Consumer<OkHttpClient.Builder> c){
        OkHttpClient.Builder builder = new OkHttpClient().newBuilder()
                .readTimeout(READ_TIMEOUT, TimeUnit.SECONDS)
                .writeTimeout(WRITE_TIMEOUT, TimeUnit.SECONDS)
                .connectTimeout(CONNECT_TIMEOUT, TimeUnit.SECONDS)
                .connectionPool(new ConnectionPool(MAX_IDLE_CONNECTION, KEEP_ALIVE_DURATION, TimeUnit.MINUTES));
        if (redirects) {
            builder.followRedirects(false);
        }
        if (skipSsl) {
            X509TrustManager manager = SslSocket.trustManager();
            builder.sslSocketFactory(SslSocket.socketFactory(manager), manager)
                    .hostnameVerifier(SslSocket.hostnameVerifier());
        } else {
            ImmutablePair<SSLSocketFactory, X509TrustManager> pair = SslSocket.sslSocketFactory();
            if (pair != null) {
                builder.sslSocketFactory(pair.getKey(), pair.getValue());
            }
        }
        if (c != null) {
            c.accept(builder);
        }
        return builder.build();
    }

    /**
     * 获取通用的GET请求Call
     *
     * @param url 请求地址
     * @param headerMap 头
     * @return call对象
     */
    private static Call baseGetCall(String url, Map<String, String> headerMap) {
        Request request = builder(url, headerMap).build();
        return getInstance().newCall(request);
    }

    /**
     * 获取POST发送请求参数的call
     *
     * @param url 请求地址
     * @param headerMap 请求头
     * @param mapParams 参数map
     * @return call对象
     */
    private static Call basePostCallForm(String url, Map<String, String> headerMap, Map<String, String> mapParams) {
        FormBody.Builder builder = new FormBody.Builder();
        if (mapParams != null) {
            for (String key : mapParams.keySet()) {
                builder.add(key, mapParams.get(key));
            }
        }
        Request.Builder requestBuilder = builder(url, headerMap);
        requestBuilder.post(builder.build());
        Request request = requestBuilder.build();
        return getInstance().newCall(request);
    }

    /**
     * 获取post请求发送json串的call
     *
     * @param url 请求地址
     * @param headerMap 请求头
     * @param params 请求参数
     * @return call对象
     */
    private static Call basePostCallJson(String url, Map<String, String> headerMap, String params, boolean isJson) {
        MediaType type;
        boolean jsonRequest = isJson || StringUtils.startsWith(params, BaseConstants.JSON_START)
                && StringUtils.endsWith(params, BaseConstants.JSON_END);
        if (jsonRequest) {
            type = MediaType.parse(org.springframework.http.MediaType.APPLICATION_JSON_VALUE);
        } else {
            type = MediaType.parse(org.springframework.http.MediaType.APPLICATION_XML_VALUE);
        }
        assert type != null;
        type.charset(Charset.defaultCharset());
        RequestBody body = RequestBody.Companion.create(params, type);
        Request.Builder requestBuilder = builder(url, headerMap);
        requestBuilder.post(body);
        Request request = requestBuilder.build();
        return getInstance().newCall(request);
    }

    /**
     * 创建builder 对象
     *
     * @param url    请求地址
     * @param header 请求头
     * @return build
     */
    private static Request.Builder builder(String url, Map<String, String> header) {
        Request.Builder builder = new Request.Builder();
        builder.url(url);
        if (MapUtils.isNotEmpty(header)) {
            for (Map.Entry<String, String> entry : header.entrySet()) {
                builder.addHeader(entry.getKey(), entry.getValue());
            }
        }
        return builder;
    }

    /**
     * 获取执行的结果，可以
     * @param call call
     * @param type 类型 1:string 2:bytes 3:headers 4:文件
     * @return 结果
     * @param <T> 结果类型
     */
    private static <T> T responseBody (Call call, int type) {
        try (Response response = call.execute();
             ResponseBody body = response.body()){
            Object result;
            int isBody = 1, isArray = 2, isHeader = 3;
            if (type == isBody) {
                result = body.string();
            } else if (type == isArray) {
                result = body.bytes();
            } else if (type == isHeader) {
                result = response.headers();
            } else {
                String fileName = FileUtil.getHeaderFileName(response);
                result = Pair.of(fileName, body.bytes());
            }
            return PojoUtil.cast(result) ;
        } catch (Exception e) {
            log.error("http请求错误：", e);
        }
        return null;
    }
}