/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.utils;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Strings;
import org.apache.commons.lang3.math.NumberUtils;

import java.util.List;

/**
 * 密码强度检查
 *
 * @Author: xyb
 * @Description:
 * @Date: 2023-05-10 下午 04:24
 **/
public class CipherStrength {
    /** 数字 */
    private static final int NUM = 1;
    /** 小写 */
    private static final int SMALL_LETTER = 2;
    /** 大写 */
    private static final int CAPITAL_LETTER = 3;
    /** 其他字符 */
    private static final int OTHER_CHAR = 4;

    /** 数字0 */
    private static final int NUM_0 = 0;
    /** 数字1 */
    private static final int NUM_1 = 1;
    /** 数字2 */
    private static final int NUM_2 = 2;
    /** 数字3 */
    private static final int NUM_3 = 3;
    /** 数字4 */
    private static final int NUM_4 = 4;
    /** 数字6 */
    private static final int NUM_6 = 6;
    /** 数字7 */
    private static final int NUM_7 = 7;
    /** 数字8 */
    private static final int NUM_8 = 8;
    /** 数字10 */
    private static final int NUM_10 = 10;
    /** 数字12 */
    private static final int NUM_12 = 12;
    /** 数字13 */
    private static final int NUM_13 = 13;
    /** 数字16 */
    private static final int NUM_16 = 16;
    /** 字符 0 */
    private static final int CHAR_0 = 48;
    /** 字符 9 */
    private static final int CHAR_9 = 57;
    /** 字符 A */
    private static final int CHAR_A = 65;
    /** 字符 Z */
    private static final int CHAR_Z = 90;
    /** 字符 a */
    private static final int CHAR_L_A = 97;
    /** 字符 z */
    private static final int CHAR_L_Z = 122;
    private static final String[] DICTIONARY = new String[]{"password", "abc123", "iloveyou", "adobe123", "123123", "sunshine", "1314520", "a1b2c3", "123qwe", "aaa111", "qweasd", "admin", "passwd"};

    private CipherStrength() {
    }

    /**
     * 密码强度检测
     * <p>[0, 4)弱密码
     * <p>[4, 7)中等密码
     * <p>[7, 10)强密码
     * <p>[10, 13)非常强
     * <p>[13, +∞)极端强
     * @param password 密码
     * @return 强度
     */
    public static int passwordStrength(String password) {
        String nullString = "null";
        if (StringUtils.isBlank(password) || Strings.CI.equals(password, nullString)) {
            throw new IllegalArgumentException("password is empty");
        }
        // 密码基本组合 长度
        int score = lengthPassword(password);
        int level = score;
        // 其他复杂的组合
        score = combinationScore(password);
        level += score;

        // 是否为基本的密码字符
        score = basePassword(password);
        level = level - score;

        // 密码是否分段相等
        score = sectionEqual(NUM_2, password);
        level = level - score;

        score = sectionEqual(NUM_3, password);
        level = level - score;

        // 密码是否为日期
        score = datePassword(password);
        level = level - score;

        if (isCharEqual(password) || level < NUM_3) {
            level = NUM_0;
        }
        return level;
    }

    /**
     * 返回密码强度
     * @param password 密码
     * @return 密码强度
     */
    public static String passwordStrengthLevel(String password){
        int level = passwordStrength(password);
        if (level < NUM_4) {
            return "EASY";
        } else if (level < NUM_7) {
            return "MEDIUM";
        } else if (level < NUM_10) {
            return "STRONG";
        } else if (level < NUM_13) {
            return "VERY_STRONG";
        }
        return "EXTREMELY_STRONG";
    }

    /**
     * 判断密码是否为弱密码
     * @param password 密码
     * @return 是否为弱密码
     */
    public static boolean weakPassword (String password){
        int level = passwordStrength(password);
        return level < NUM_7;
    }
    /**
     * 字符类型
     * @param c 字符
     * @return 类型
     */
    private static int checkCharacterType(char c) {
        if (c >= CHAR_0 && c <= CHAR_9) {
            return NUM;
        } else if (c >= CHAR_A && c <= CHAR_Z) {
            return CAPITAL_LETTER;
        } else if (c >= CHAR_L_A && c <= CHAR_L_Z) {
            return SMALL_LETTER;
        } else {
            return OTHER_CHAR;
        }
    }

    /**
     * 计算字符串类型的长度
     * @param password 密码
     * @param type 类型
     * @return 长度
     */
    private static int countLetter(String password, int type) {
        int count = 0;
        if (StringUtils.isNotBlank(password)) {
            char[] arrays = password.toCharArray();
            for (char c : arrays) {
                if (checkCharacterType(c) == type) {
                    ++count;
                }
            }
        }
        return count;
    }

    /**
     * 密码的组合模式
     * @param password 密码
     * @return 分数
     */
    private static int combinationScore(String password){
        int score = 0;
        int len = password.length();
        // 判断密码是否还有大写字母有level++
        if (len > NUM_4 && countLetter(password, CAPITAL_LETTER) > NUM_0) {
            score++;
        }
        // 判断密码是否还有特殊字符有level++
        if (len > NUM_6 && countLetter(password, OTHER_CHAR) > NUM_0) {
            score++;
        }

        // 密码长度大于4并且2种类型组合
        if (len > NUM_4 && combination(password, NUM_2, NUM_0, null)) {
            score++;
        }
        // 密码长度大于6并且3中类型组合 4
        if (len > NUM_6 && combination(password, NUM_3, NUM_0, null)) {
            score++;
        }
        // 密码长度大于8并且4种类型组合
        if (len > NUM_8 && combination(password, NUM_4, NUM_0, null)) {
            score++;
        }
        // 密码长度大于6并且2种类型组合每种类型长度大于等于3或者2
        if (len > NUM_6 && combination(password, NUM_2, NUM_3, NUM_2)) {
            score++;
        }
        // 密码长度大于8并且3种类型组合每种类型长度大于等于3或者）
        if (len > NUM_8 && combination(password, NUM_3, NUM_2, null)) {
            score++;
        }
        // 密码长度大于10并且4种类型组合每种类型长度大于等于2
        if (len > NUM_10 && combination(password, NUM_4, NUM_2, null)) {
            score++;
        }
        return score;
    }

    /**
     * 判断密码的组合方式
     * @param password 密码
     * @param combinationNum 组合数
     * @param standardNum 长度
     * @param otherNum 其他的长度
     * @return 是否满足条件
     */
    private static boolean combination(String password, int combinationNum, int standardNum, Integer otherNum){
        if (otherNum == null) {
            otherNum = standardNum;
        }
        List<List<Integer>> combination = AlgoUtil.c(List.of(NUM, SMALL_LETTER, CAPITAL_LETTER, OTHER_CHAR), combinationNum);
        if (CollectionUtils.isEmpty(combination)) {
            return true;
        }
        for (List<Integer> list : combination) {
            boolean temp = true;
            for (Integer i : list) {
                temp &= countLetter(password, i) > (i == OTHER_CHAR ? otherNum : standardNum);
            }
            if (temp) {
                return temp;
            }
        }
        return false;
    }

    /**
     * 密码长度分数
     * @param password 密码
     * @return 长度
     */
    private static int lengthPassword(String password){
        int score = 0;
        // 判断密码是否含有数字有level++
        if (countLetter(password, NUM) > NUM_0) {
            score++;
        }
        // 判断密码是否含有小写字母有level++
        if (countLetter(password, SMALL_LETTER) > NUM_0) {
            score++;
        }
        // 其他字符长度 大于3位
        if (countLetter(password, OTHER_CHAR) >= NUM_3) {
            score++;
        }
        // 其他字符长度 大于6位
        if (countLetter(password, OTHER_CHAR) >= NUM_6) {
            score++;
        }
        int length = password.length();
        if (length > NUM_12) {
            score++;
        }
        if (length >= NUM_16) {
            score++;
        }
        if (length <= NUM_6) {
            score--;
        }
        if (length <= NUM_4) {
            score--;
        }
        return score;
    }

    /**
     * 是否为基本密码
     * @param password 密码
     * @return 权重
     */
    private static int basePassword(String password){
        // 是否为基本的密码字符
        int score = 0;
        String[] letter = {"abcdefghijklmnopqrstuvwxyz", "ABCDEFGHIJKLMNOPQRSTUVWXYZ"};
        if (indexOf(password, letter)) {
            score ++;
        }
        String[] keyboard = {"qwertyuiop", "asdfghjkl", "zxcvbnm"};
        if (indexOf(password, keyboard)) {
            score ++;
        }

        String[] digits = {"01234567890", "09876543210"};
        if (NumberUtils.isDigits(password) && indexOf(password, digits)) {
            score ++;
        }
        int len = password.length();
        // 全数字 或者全字母
        if (countLetter(password, NUM) == len || countLetter(password, SMALL_LETTER) == len || countLetter(password, CAPITAL_LETTER) == len) {
            score ++;
        }
        for (String s : DICTIONARY) {
            if (password.equals(s) || s.contains(password) || password.contains(s)) {
                score --;
                break;
            }
        }
        return score;
    }

    /**
     * 密码是否为日期
     * @param password 日期
     * @return 权重
     */
    private static int datePassword (String password){
        int length = password.length();
        boolean isDate = false;
        if (NumberUtils.isDigits(password) && length >= NUM_6) {
            String pattern;
            if (length == NUM_8) {
                pattern = "yyyyMMdd";
            } else {
                pattern = "yyMMdd";
            }
            // 如果是日期 20230510 或者 230510
            isDate = Boolean.TRUE.equals(TirsciUtil.tryCatch(() -> {
                DateUtil.parseDate(password, pattern);
                return true;
            }));
        }
        return isDate ? NUM_1 : NUM_0;
    }

    /**
     * 分段相关
     *
     * @param sectionNum 分段
     * @param password 密码
     * @return  level 级别
     */
    public static int sectionEqual (int sectionNum, String password){
        int level = 0;
        if (password.length() % sectionNum != NUM_0) {
            return level;
        }
        int div = password.length() / sectionNum;
        if (div == NUM_1) {
            return level;
        }
        String part1;
        String part2;
        boolean sectionEqual = false;
        boolean charEqual = false;
        for (int i = NUM_0; i < sectionNum - 1; i++) {
            part1 = password.substring(i * div, (i + 1) * div);
            part2 = password.substring((i + 1) * div , (i + 2) * div);
            sectionEqual = part1.equals(part2);
            charEqual = isCharEqual(part1) && isCharEqual(part2);
        }
        if (sectionEqual) {
            level += NUM_1;
        }
        if (charEqual) {
            level += NUM_1;
        }
        return level;
    }

    /**
     * 判断密码是否为 弱密码
     * @param password 密码
     * @param resources 弱密码库
     * @return 是否存在
     */
    private static boolean indexOf(String password, String[] resources){
        for (String r : resources) {
            if (r.indexOf(password) > NUM_0) {
                return true;
            }
        }
        return false;
    }

    /**
     * 判断字符串的每个字符是否相等
     *
     * @param str 字符
     * @return 是否相等
     */
    private static boolean isCharEqual(String str) {
        return str.replace(str.charAt(NUM_0), ' ').trim().isEmpty();
    }
}