/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.exception;

import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.utils.TirsciUtil;

import java.util.Objects;


/**
 * 全局业务异常
 * @Author: xyb
 * @Description:
 * @Date: 2022-11-14 下午 09:41
 **/
public class CavException extends RuntimeException {

	/** 业务code */
	private String code;

	/** 异常信息参数 */
	private final Object[] params;

	/**
	 * 业务异常
	 * @param code 业务异常dode
	 */
	public CavException(String code) {
		this(code, (Object) null);
	}

	/**
	 * 直接复制异常，抛出到页面
	 * @param e 异常
	 */
	public CavException(Exception e){
		this(Objects.toString(e.getMessage(), BaseConstants.GLOBAL_EXCEPTION_MESSAGE), e);
	}

	/**
	 * 业务异常，复制了其他异常的栈堆信息
	 * @param code 错误信息
	 * @param e 异常
	 */
	public CavException(String code, Exception e) {
		this(
				e instanceof CavException ? ((CavException) e).getCode() : code,
				e instanceof CavException ? null : e,
				e instanceof CavException ? ((CavException) e).getParameters() : null);
	}

	/**
	 * 业务异常
	 * @param code 业务异常dode
	 * @param params 业务参数
	 */
	public CavException(String code, Object... params) {
		this(code, null, params);
	}

	/**
	 * 业务异常
	 * @param code 业务异常dode
	 * @param e 抛出来的 异常
	 * @param params 参数
	 */
	public CavException (String code, Exception e, Object... params){
		super(code, e);
		// 如果是国际化的
		if (code.startsWith(BaseConstants.EXP_PREFIX)) {
			this.code = code;
			this.params = params;
		} else {
			this.code = BaseConstants.DEFAULT_CODE;
			Object[] arrays = new Object[1];
			arrays[0] = TirsciUtil.format(code, params);
			this.params = arrays;
		}
	}

	/**
	 * 获取 业务code
	 *
	 * @return code 业务code
	 */
	public String getCode() {
		return this.code;
	}

	/**
	 * 设置 业务code
	 *
	 * @param code 业务code
	 */
	public void setCode(String code) {
		this.code = code;
	}

	/**
	 * 获取 异常信息参数
	 *
	 * @return params 异常信息参数
	 */
	public Object[] getParameters() {
		return this.params;
	}
}
