/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.suport;

import org.apache.commons.fileupload2.core.DiskFileItem;
import org.jetbrains.annotations.NotNull;
import org.springframework.util.FileCopyUtils;
import org.springframework.web.multipart.MultipartFile;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;

/**
 * Spring MultipartFile adapter, wrapping a Servlet Part object.
 * @see org.springframework.web.multipart.support.StandardMultipartHttpServletRequest
 * @author xyb
 * @Description
 * @Copyright Copyright (c) 2024
 * @since 2024-07-15 22:01
 */
public class StandardMultipartFile implements MultipartFile, Serializable  {

    private final DiskFileItem fileItem;

    private final String filename;

    public StandardMultipartFile(DiskFileItem fileItem, String filename) {
        this.fileItem = fileItem;
        this.filename = filename;
    }

    @Override
    public @NotNull String getName() {
        return this.fileItem.getName();
    }

    @Override
    public String getOriginalFilename() {
        return this.filename;
    }

    @Override
    public String getContentType() {
        return this.fileItem.getContentType();
    }

    @Override
    public boolean isEmpty() {
        return (this.fileItem.getSize() == 0);
    }

    @Override
    public long getSize() {
        return this.fileItem.getSize();
    }

    @Override
    public byte @NotNull [] getBytes() throws IOException {
        return FileCopyUtils.copyToByteArray(this.fileItem.getInputStream());
    }

    @Override
    public @NotNull InputStream getInputStream() throws IOException {
        return this.fileItem.getInputStream();
    }

    @Override
    public void transferTo(File dest) throws IOException, IllegalStateException {
        this.fileItem.write(Paths.get(dest.getPath()));
        if (dest.isAbsolute() && !dest.exists()) {
            // Servlet Part.write is not guaranteed to support absolute file paths:
            // may translate the given path to a relative location within a temp dir
            // (e.g. on Jetty whereas Tomcat and Undertow detect absolute paths).
            // At least we offloaded the file from memory storage; it'll get deleted
            // from the temp dir eventually in any case. And for our user's purposes,
            // we can manually copy it to the requested location as a fallback.
            FileCopyUtils.copy(this.fileItem.getInputStream(), Files.newOutputStream(dest.toPath()));
        }
    }

    @Override
    public void transferTo(@NotNull Path dest) throws IOException, IllegalStateException {
        FileCopyUtils.copy(this.fileItem.getInputStream(), Files.newOutputStream(dest));
    }
}
