/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.utils;

import cn.sinozg.applet.common.exception.CavException;
import org.apache.commons.lang3.StringUtils;

import java.time.LocalDate;
import java.util.regex.Pattern;

/**
 * @author xieyubin
 * @Description
 * @Copyright Copyright (c) 2023
 * @since 2023-11-04 22:05
 */
public class IdCardUtil {
    /** 行政区划 大陆地区 */
    private static final String REGEX = "([1|6][1-5]|2[1-3]|3[1-7]|4[1-6]|5[0-4])([0-2][0-9]|[3-5][0-4]|90)([0-5|7-9][0-9])";
    /** 行政区划 港澳台 81 82 83 外国人9开头，二三位是受理地代码，四五六为国家代码  */
    private static final String NON_MAINLAND_REGEX = "(?:8[123]0{4}|9[1-5][0-7]\\d{3})";

    private static final int[] POWER = {7, 9, 10, 5, 8, 4, 2, 1, 6, 3, 7, 9, 10, 5, 8, 4, 2};

    private static final int LEN_15 = 15;

    private static final int LEN_18 = 18;

    private static final String DIGITAL_REGEX = "^[0-9]*$";
    private IdCardUtil() {
    }

    /**
     * 检查身份证号码
     * @param idCardNum 身份证号码
     */
    public static void checkIdCard(String idCardNum){
        String msg = idCard(idCardNum);
        if (StringUtils.isNotBlank(msg)) {
            throw new CavException("BIZ000100033");
        }
    }

    /**
     * 校验身份证号码是否正确
     * @param idCardNum 身份证号码
     * @return 错误信息
     */
    public static String idCard (String idCardNum){
        idCardNum = StringUtils.trimToEmpty(idCardNum);
        if (StringUtils.isBlank(idCardNum)) {
            return "身份证号码为空";
        }
        int len = idCardNum.length();
        if (!(len == LEN_18 || len == LEN_15)) {
            return "身份证号码必须是18或者15位！";
        }
        String idCardPre = StringUtils.substring(idCardNum, 0, len - 1);
        if (!idCardPre.matches(DIGITAL_REGEX)) {
            return "身份证号码除最后一位外必须为数字！";
        }
        if (len == LEN_15) {
            idCardNum = idCard15To18(idCardNum);
        }
        return isIdCard(idCardNum);
    }

    /**
     * 15位身份证转 18位
     * @param idCardNum 15位
     * @return 18位
     */
    public static String idCard15To18(String idCardNum){
        idCardNum = StringUtils.trimToEmpty(idCardNum);
        // 不是15位
        if (StringUtils.isBlank(idCardNum) || idCardNum.length() != LEN_15) {
            return StringUtils.EMPTY;
        }
        if (!idCardNum.matches(DIGITAL_REGEX)) {
            throw new RuntimeException("身份证号码除最后一位外必须为数字！");
        }
        String idCard17 = StringUtils.substring(idCardNum, 0, 6) + "19" + StringUtils.substring(idCardNum, 6);
        String powerCode = powerStr(idCard17);
        return idCard17 + powerCode;
    }

    /**
     * 是否位大陆地区身份证
     * @param idNm 身份证
     * @return 是否有错误
     */
    private static String isIdCard (String idNm){
        String ymd = StringUtils.substring(idNm, 6, 14);
        LocalDate ld;
        // 通过日期格式化 判断日期合法性
        try {
            ld = DateUtil.parseDate(ymd, DateUtil.YYYYMMDD);
            String newYmd = DateUtil.formatDate(ld, DateUtil.YYYYMMDD);
            if (!newYmd.equals(ymd)) {
                return "身份证出生日期不正确！";
            }
        } catch (Exception e) {
            return "身份证出生日期不正确！";
        }
        LocalDate now = LocalDate.now();
        // 150年
        LocalDate bf = now.minusYears(150);
        if (ld.isAfter(now) || ld.isBefore(bf)) {
            return "出生日期超出日期范围！";
        }
        // 省市区校验
        String pcc = StringUtils.substring(idNm, 0, 6);
        if (!(Pattern.matches(REGEX, pcc) || Pattern.matches(NON_MAINLAND_REGEX, pcc))) {
            return "未定义的行政区划代码！";
        }
        // 最后检验码
        String lastCode = StringUtils.substring(idNm, 17);
        String idNm17 = StringUtils.substring(idNm, 0, 17);
        String powerCode = powerStr(idNm17);
        if (!powerCode.equals(lastCode)) {
            return "非法的身份证号码！";
        }
        return StringUtils.EMPTY;
    }

    /**
     * 获取到校验码
     * @param nm 身份证
     * @return 校验码
     */
    private static String powerStr (String nm){
        int sum = 0;
        for (int i = 0, j = 17; i < j; i++) {
            sum += (nm.charAt(i) - 48) * POWER[i];
        }
        int t = sum % 11;
        int c = t < 2 ? 49 : (t > 2 ? 60 : 90);
        return Character.toString((char) (c - t));
    }
}
