package cn.sinozg.applet.quartz.use.ext;

import cn.sinozg.applet.common.utils.DateUtil;
import org.apache.commons.collections4.CollectionUtils;
import org.quartz.CalendarIntervalTrigger;
import org.quartz.DailyTimeIntervalTrigger;
import org.quartz.DateBuilder;
import org.quartz.ScheduleBuilder;
import org.quartz.TimeOfDay;
import org.quartz.Trigger;
import org.quartz.TriggerBuilder;
import org.quartz.spi.MutableTrigger;

import java.time.LocalDateTime;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;

/**
 * @author xieyubin
 * @Description
 * @Copyright Copyright (c) 2024
 * @since 2024-01-15 14:35
 */
public class DailyTimeIntervalMonthScheduleBuilder extends ScheduleBuilder<DailyTimeIntervalMonthTrigger> {

    private int interval = 1;
    private DateBuilder.IntervalUnit intervalUnit = DateBuilder.IntervalUnit.MINUTE;
    private Set<Integer> daysOfMonth;
    private TimeOfDay startTimeOfDay;
    private TimeOfDay endTimeOfDay;
    private int repeatCount = DailyTimeIntervalTrigger.REPEAT_INDEFINITELY;

    private int misfireInstruction = CalendarIntervalTrigger.MISFIRE_INSTRUCTION_SMART_POLICY;

    /**
     * A set of all days of the week.
     * <p>
     * The set contains all values between (the integers from 1 through 31).
     */
    public static final Set<Integer> ALL_DAYS_OF_THE_MONTH;

    private static final TimeOfDay STARTING_DAILY = TimeOfDay.hourAndMinuteOfDay(0, 0);

    private static final TimeOfDay ENDING_DAILY = TimeOfDay.hourMinuteAndSecondOfDay(23, 59, 59);

    static {
        int maxDay = 31;
        Set<Integer> t = new HashSet<>(31);
        for (int i = 1; i <= maxDay; i++) {
            t.add(i);
        }
        ALL_DAYS_OF_THE_MONTH = Collections.unmodifiableSet(t);
    }

    protected DailyTimeIntervalMonthScheduleBuilder() {
    }

    /**
     * Create a DailyTimeIntervalScheduleBuilder.
     *
     * @return the new DailyTimeIntervalScheduleBuilder
     */
    public static DailyTimeIntervalMonthScheduleBuilder dailyTimeIntervalSchedule() {
        return new DailyTimeIntervalMonthScheduleBuilder();
    }

    /**
     * Build the actual Trigger -- NOT intended to be invoked by end users,
     * but will rather be invoked by a TriggerBuilder which this
     * ScheduleBuilder is given to.
     *
     * @see TriggerBuilder#withSchedule(ScheduleBuilder)
     */
    @Override
    public MutableTrigger build() {

        DailyTimeIntervalTriggerMonthImpl st = new DailyTimeIntervalTriggerMonthImpl();
        st.setRepeatInterval(interval);
        st.setRepeatIntervalUnit(intervalUnit);
        st.setMisfireInstruction(misfireInstruction);
        st.setRepeatCount(repeatCount);

        if (daysOfMonth != null) {
            st.setDaysOfMonth(daysOfMonth);
        } else {
            st.setDaysOfMonth(ALL_DAYS_OF_THE_MONTH);
        }
        if (startTimeOfDay != null) {
            st.setStartTimeOfDay(startTimeOfDay);
        } else {
            st.setStartTimeOfDay(STARTING_DAILY);
        }
        if (endTimeOfDay != null)  {
            st.setEndTimeOfDay(endTimeOfDay);
        } else {
            st.setEndTimeOfDay(ENDING_DAILY);
        }
        return st;
    }

    /**
     * Specify the time unit and interval for the Trigger to be produced.
     *
     * @param timeInterval the interval at which the trigger should repeat.
     * @param unit         the time unit (IntervalUnit) of the interval. The only intervals that are valid for this type of
     *                     trigger are {@link DateBuilder.IntervalUnit#SECOND}, {@link DateBuilder.IntervalUnit#MINUTE}, and {@link DateBuilder.IntervalUnit#HOUR}.
     * @return the updated DailyTimeIntervalScheduleBuilder
     * @see DailyTimeIntervalTrigger#getRepeatInterval()
     * @see DailyTimeIntervalTrigger#getRepeatIntervalUnit()
     */
    public DailyTimeIntervalMonthScheduleBuilder withInterval(int timeInterval, DateBuilder.IntervalUnit unit) {
        boolean params = unit == null || !(unit.equals(DateBuilder.IntervalUnit.SECOND) ||
                unit.equals(DateBuilder.IntervalUnit.MINUTE) || unit.equals(DateBuilder.IntervalUnit.HOUR));
        if (params) {
            throw new IllegalArgumentException("Invalid repeat IntervalUnit (must be SECOND, MINUTE or HOUR).");
        }
        validateInterval(timeInterval);
        this.interval = timeInterval;
        this.intervalUnit = unit;
        return this;
    }

    /**
     * Specify an interval in the IntervalUnit.SECOND that the produced
     * Trigger will repeat at.
     *
     * @param intervalInSeconds the number of seconds at which the trigger should repeat.
     * @return the updated DailyTimeIntervalScheduleBuilder
     * @see DailyTimeIntervalTrigger#getRepeatInterval()
     * @see DailyTimeIntervalTrigger#getRepeatIntervalUnit()
     */
    public DailyTimeIntervalMonthScheduleBuilder withIntervalInSeconds(int intervalInSeconds) {
        withInterval(intervalInSeconds, DateBuilder.IntervalUnit.SECOND);
        return this;
    }

    /**
     * Specify an interval in the IntervalUnit.MINUTE that the produced
     * Trigger will repeat at.
     *
     * @param intervalInMinutes the number of minutes at which the trigger should repeat.
     * @return the updated CalendarIntervalScheduleBuilder
     * @see DailyTimeIntervalTrigger#getRepeatInterval()
     * @see DailyTimeIntervalTrigger#getRepeatIntervalUnit()
     */
    public DailyTimeIntervalMonthScheduleBuilder withIntervalInMinutes(int intervalInMinutes) {
        withInterval(intervalInMinutes, DateBuilder.IntervalUnit.MINUTE);
        return this;
    }

    /**
     * Specify an interval in the IntervalUnit.HOUR that the produced
     * Trigger will repeat at.
     *
     * @param intervalInHours the number of hours at which the trigger should repeat.
     * @return the updated DailyTimeIntervalScheduleBuilder
     * @see DailyTimeIntervalTrigger#getRepeatInterval()
     * @see DailyTimeIntervalTrigger#getRepeatIntervalUnit()
     */
    public DailyTimeIntervalMonthScheduleBuilder withIntervalInHours(int intervalInHours) {
        withInterval(intervalInHours, DateBuilder.IntervalUnit.HOUR);
        return this;
    }

    /**
     * Set the trigger to fire on the given days of the month.
     *
     * @param onDaysOfMonth a
     * @return the updated DailyTimeIntervalScheduleBuilder
     */
    public DailyTimeIntervalMonthScheduleBuilder onDaysOfTheMonth(Set<Integer> onDaysOfMonth) {
        if (CollectionUtils.isEmpty(onDaysOfMonth)) {
            throw new IllegalArgumentException("Days of week must be an non-empty set.");
        }
        for (Integer day : onDaysOfMonth) {
            if (!ALL_DAYS_OF_THE_MONTH.contains(day)) {
                throw new IllegalArgumentException("Invalid value for day of week: " + day);
            }
        }
        this.daysOfMonth = onDaysOfMonth;
        return this;
    }

    /**
     * Set the trigger to fire on the given days of the month.
     *
     * @param onDaysOfMonth a
     * @return the updated DailyTimeIntervalScheduleBuilder
     */
    public DailyTimeIntervalMonthScheduleBuilder onDaysOfTheMonth(Integer... onDaysOfMonth) {
        Set<Integer> daysAsSet = new HashSet<>(64);
        Collections.addAll(daysAsSet, onDaysOfMonth);
        return onDaysOfTheMonth(daysAsSet);
    }


    /**
     * Set the trigger to fire on all days of the month.
     *
     * @return the updated DailyTimeIntervalScheduleBuilder
     */
    public DailyTimeIntervalMonthScheduleBuilder onEveryDay() {
        this.daysOfMonth = ALL_DAYS_OF_THE_MONTH;
        return this;
    }

    /**
     * Set the trigger to begin firing each day at the given time.
     *
     * @return the updated DailyTimeIntervalScheduleBuilder
     */
    public DailyTimeIntervalMonthScheduleBuilder startingDailyAt(TimeOfDay timeOfDay) {
        if (timeOfDay == null) {
            throw new IllegalArgumentException("Start time of day cannot be null!");
        }
        this.startTimeOfDay = timeOfDay;
        return this;
    }

    /**
     * Set the startTimeOfDay for this trigger to end firing each day at the given time.
     *
     * @return the updated DailyTimeIntervalScheduleBuilder
     */
    public DailyTimeIntervalMonthScheduleBuilder endingDailyAt(TimeOfDay timeOfDay) {
        this.endTimeOfDay = timeOfDay;
        return this;
    }

    /**
     * Calculate and set the endTimeOfDay using count, interval and starTimeOfDay. This means
     * that these must be set before this method is call.
     *
     * @return the updated DailyTimeIntervalScheduleBuilder
     */
    public DailyTimeIntervalMonthScheduleBuilder endingDailyAfterCount(int count) {
        if (count <= 0) {
            throw new IllegalArgumentException("Ending daily after count must be a positive number!");
        }
        if (startTimeOfDay == null) {
            throw new IllegalArgumentException("You must set the startDailyAt() before calling this endingDailyAfterCount()!");
        }
        Date today = new Date();
        long startTimeOfDayDateMillis = startTimeOfDay.getTimeOfDayForDate(today).getTime();
        long maxEndTimeOfDayDateMillis = ENDING_DAILY.getTimeOfDayForDate(today).getTime();
        long remainingMillisInDay = maxEndTimeOfDayDateMillis - startTimeOfDayDateMillis;
        long intervalInMillis;
        if (intervalUnit == DateBuilder.IntervalUnit.SECOND) {
            intervalInMillis = interval * 1000L;

        } else if (intervalUnit == DateBuilder.IntervalUnit.MINUTE) {
            intervalInMillis = interval * 1000L * 60;
        } else if(intervalUnit ==DateBuilder.IntervalUnit.HOUR) {
            intervalInMillis =interval * 1000L * 60 * 24;
        } else {
            throw new IllegalArgumentException("The IntervalUnit: " + intervalUnit + " is invalid for this trigger.");
        }
        if (remainingMillisInDay - intervalInMillis <= 0) {
            throw new IllegalArgumentException("The startTimeOfDay is too late with given Interval and IntervalUnit values.");
        }
        long maxNumOfCount = (remainingMillisInDay / intervalInMillis);
        if (count > maxNumOfCount) {
            throw new IllegalArgumentException("The given count " + count + " is too large! The max you can set is " + maxNumOfCount);
        }
        long incrementInMillis = (count - 1) * intervalInMillis;
        long endTimeOfDayMillis = startTimeOfDayDateMillis + incrementInMillis;
        if (endTimeOfDayMillis > maxEndTimeOfDayDateMillis) {
            throw new IllegalArgumentException("The given count " + count + " is too large! The max you can set is " + maxNumOfCount);
        }
        LocalDateTime endLdt = DateUtil.ldt(endTimeOfDayMillis);
        endTimeOfDay = TimeOfDay.hourMinuteAndSecondOfDay(endLdt.getHour(), endLdt.getMinute(), endLdt.getSecond());
        return this;
    }

    /**
     * If the Trigger misfires, use the
     * {@link Trigger#MISFIRE_INSTRUCTION_IGNORE_MISFIRE_POLICY} instruction.
     *
     * @return the updated DailyTimeIntervalScheduleBuilder
     * @see Trigger#MISFIRE_INSTRUCTION_IGNORE_MISFIRE_POLICY
     */
    public DailyTimeIntervalMonthScheduleBuilder withMisfireHandlingInstructionIgnoreMisfires() {
        misfireInstruction = Trigger.MISFIRE_INSTRUCTION_IGNORE_MISFIRE_POLICY;
        return this;
    }

    /**
     * If the Trigger misfires, use the
     * {@link DailyTimeIntervalTrigger#MISFIRE_INSTRUCTION_DO_NOTHING} instruction.
     *
     * @return the updated DailyTimeIntervalScheduleBuilder
     * @see DailyTimeIntervalTrigger#MISFIRE_INSTRUCTION_DO_NOTHING
     */
    public DailyTimeIntervalMonthScheduleBuilder withMisfireHandlingInstructionDoNothing() {
        misfireInstruction = DailyTimeIntervalTrigger.MISFIRE_INSTRUCTION_DO_NOTHING;
        return this;
    }

    /**
     * If the Trigger misfires, use the
     * {@link DailyTimeIntervalTrigger#MISFIRE_INSTRUCTION_FIRE_ONCE_NOW} instruction.
     *
     * @return the updated DailyTimeIntervalScheduleBuilder
     * @see DailyTimeIntervalTrigger#MISFIRE_INSTRUCTION_FIRE_ONCE_NOW
     */
    public DailyTimeIntervalMonthScheduleBuilder withMisfireHandlingInstructionFireAndProceed() {
        misfireInstruction = CalendarIntervalTrigger.MISFIRE_INSTRUCTION_FIRE_ONCE_NOW;
        return this;
    }

    /**
     * Set number of times for interval to repeat.
     *
     * <p>Note: if you want total count = 1 (at start time) + repeatCount</p>
     *
     * @return the new DailyTimeIntervalScheduleBuilder
     */
    public DailyTimeIntervalMonthScheduleBuilder withRepeatCount(int repeatCount) {
        this.repeatCount = repeatCount;
        return this;
    }

    private void validateInterval(int timeInterval) {
        if(timeInterval <= 0) {
            throw new IllegalArgumentException("Interval must be a positive value.");
        }
    }
}