package cn.sinozg.applet.quartz.use.handler;

import cn.sinozg.applet.common.holder.UserContextHolder;
import cn.sinozg.applet.common.utils.SpringUtil;
import cn.sinozg.applet.quartz.use.enums.JobKeyEnum;
import cn.sinozg.applet.quartz.use.service.JobLogBaseService;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;
import org.quartz.DisallowConcurrentExecution;
import org.quartz.JobDataMap;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;
import org.quartz.PersistJobDataAfterExecution;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.scheduling.quartz.QuartzJobBean;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.time.LocalDateTime;


/**
 * 基础 Job 调用者，负责调用 {@link JobHandler#execute(String)} 执行任务
 *
 * @Author: xyb
 * @Description:
 * @Date: 2023-12-31 上午 11:59
 **/
@DisallowConcurrentExecution
@PersistJobDataAfterExecution
public class JobHandlerInvoker extends QuartzJobBean {

    private static final Logger log = LoggerFactory.getLogger(JobHandlerInvoker.class);

    @Resource
    private JobLogBaseService jobLogService;

    @Override
    protected void executeInternal(JobExecutionContext context) throws JobExecutionException {
        // 第一步，获得 Job 数据
        JobDataMap dataMap = context.getMergedJobDataMap();
        String jobId = dataMap.getString(JobKeyEnum.JOB_ID.name());
        String jobName = dataMap.getString(JobKeyEnum.JOB_NAME.name());
        String beanName = dataMap.getString(JobKeyEnum.JOB_BEAN_NAME.name());
        String tenantId = dataMap.getString(JobKeyEnum.JOB_TENANT.name());
        String args = dataMap.getString(JobKeyEnum.JOB_ARGS.name());
        int retryCount = MapUtils.getInteger(dataMap, JobKeyEnum.JOB_RETRY_COUNT.name(), 0);
        int retryInterval = MapUtils.getInteger(dataMap, JobKeyEnum.JOB_RETRY_INTERVAL.name(), 0);
        // 第二步，执行任务
        String jobLogId = null;
        String data = null;
        Throwable exception = null;
        try {
            UserContextHolder.setTenantId(tenantId);
            // 记录 Job 日志（初始）
            jobLogId = jobLogService.createJobLog(jobId, jobName, LocalDateTime.now(), beanName, args, context.getRefireCount() + 1);
            // 执行任务
            data = this.executeInternal(beanName, args);
        } catch (Throwable ex) {
            exception = ex;
        }
        // 第三步，记录执行日志
        this.updateJobLogResultAsync(jobLogId, data, exception, context);

        // 第四步，处理有异常的情况
        handleException(exception, context.getRefireCount(), retryCount, retryInterval);
    }

    /**
     * 执行任务
     * @param beanName 名称
     * @param args 参数
     * @return 返回结果
     * @throws Exception 异常
     */
    private String executeInternal(String beanName, String args) throws Exception {
        // 获得 JobHandler 对象
        JobHandler jobHandler = SpringUtil.beanOfNmTp(beanName, JobHandler.class);
        Assert.notNull(jobHandler, "JobHandler 不会为空");
        // 执行任务
        return jobHandler.execute(args);
    }

    /**
     * 更新 记录日志
     * @param jobLogId logId
     * @param data 数据
     * @param exception 异常
     * @param context 上下文
     */
    private void updateJobLogResultAsync(String jobLogId, String data, Throwable exception, JobExecutionContext context) {
        // 处理是否成功
        boolean success = true;
        if (exception != null) {
            success = false;
            data = ExceptionUtils.getRootCauseMessage(exception);
        }
        // 更新日志
        try {
            jobLogService.updateJobLogResultAsync(jobLogId, LocalDateTime.now(), success, data);
        } catch (Exception ex) {
            log.error("执行定时任务失败 jobKey：{}，logId：{} {}， data： {}", context.getJobDetail().getKey(), jobLogId, success, data);
        }
    }

    /**
     * 异常处理
     * @param exception 异常
     * @param refireCount 再试次数
     * @param retryCount 重试次数
     * @param retryInterval 重试间隔
     * @throws JobExecutionException 异常
     */
    private void handleException(Throwable exception, int refireCount, int retryCount, int retryInterval) throws JobExecutionException {
        // 如果有异常，则进行重试
        if (exception == null) {
            return;
        }
        // 情况一：如果到达重试上限，则直接抛出异常即可
        if (refireCount >= retryCount) {
            throw new JobExecutionException(exception);
        }
        // 情况二：如果未到达重试上限，则 sleep 一定间隔时间，然后重试
        // 这里使用 sleep 来实现，主要还是希望实现比较简单。因为，同一时间，不会存在大量失败的 Job。
        if (retryInterval > 0) {
            try {
                Thread.sleep(retryInterval);
            } catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
        }
        // 第二个参数，refireImmediately = true，表示立即重试
        throw new JobExecutionException(exception, true);
    }

}
