package cn.sinozg.applet.quartz.use.util;

import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.exception.CavException;
import cn.sinozg.applet.common.utils.DateUtil;
import cn.sinozg.applet.common.utils.StringUtil;
import cn.sinozg.applet.quartz.use.enums.JobTypeEnum;
import cn.sinozg.applet.quartz.use.module.TaskScheduleModel;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.quartz.CronExpression;

import java.text.ParseException;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * Quartz Cron 表达式的工具类
 *
 * @Author: xyb
 * @Description:
 * @Date: 2023-12-31 下午 12:13
 **/
public class CronUtils {

    /**
     * 校验 CRON 表达式是否有效
     *
     * @param cronExpression CRON 表达式
     * @return 是否有效
     */
    public static boolean isValid(String cronExpression) {
        return CronExpression.isValidExpression(cronExpression);
    }

    /**
     * 基于 CRON 表达式，获得下 n 个满足执行的时间
     *
     * @param cronExpression CRON 表达式
     * @param n              数量
     * @return 满足条件的执行时间
     */
    public static List<LocalDateTime> getNextTimes(String cronExpression, int n) {
        // 获得 CronExpression 对象
        CronExpression cron;
        try {
            cron = new CronExpression(cronExpression);
        } catch (ParseException e) {
            throw new IllegalArgumentException(e.getMessage());
        }
        // 从当前开始计算，n 个满足条件的
        Date now = new Date();
        List<LocalDateTime> nextTimes = new ArrayList<>(n);
        for (int i = 0; i < n; i++) {
            Date nextTime = cron.getNextValidTimeAfter(now);
            nextTimes.add(DateUtil.toLdt(nextTime.getTime()));
            // 切换现在，为下一个触发时间；
            now = nextTime;
        }
        return nextTimes;
    }

    /**
     * 方法摘要：构建Cron表达式
     *
     * @param model 参数
     * @return String
     */
    public static String cronExpression(TaskScheduleModel model) {
        StringBuilder cron = new StringBuilder();
        JobTypeEnum type = model.getJobType();
        if (null == type) {
            throw new CavException("执行周期未配置！");
        }
        if (null == model.getSecond() && null == model.getMinute() && null == model.getHour()) {
            // 时或分或秒参数未配置
            throw new CavException("时或分或秒参数未配置！");
        }
        // 每隔几秒
        if (type.equals(JobTypeEnum.SECOND) && model.getSecond() != null) {
            cron.append("0/").append(model.getSecond())
                    .append(" * * * * ?");
        }
        // 每隔几分钟
        if (type.equals(JobTypeEnum.MINUTE) && model.getMinute() != null) {
            cron.append("* 0/").append(model.getMinute())
                    .append(" * * * ?");
        }
        // 每隔几小时
        if (type.equals(JobTypeEnum.HOUR) && model.getHour() != null) {
            cron.append("* * 0/").append(model.getHour())
                    .append(" * * ?");
        }
        if (null != model.getSecond() && null != model.getMinute() && null != model.getHour()) {
            // 秒 分 小时
            cron.append(model.getSecond()).append(StringUtils.SPACE)
                    .append(model.getMinute()).append(StringUtils.SPACE)
                    .append(model.getHour()).append(StringUtils.SPACE);
            // 每天
            if (type.equals(JobTypeEnum.DAY)) {
                cron.append("* * ?");
                // 按每周
            } else if (type.equals(JobTypeEnum.WEEK)) {
                // 一个月中第几天 月份 周
                cron.append("? * ");
                days(cron, model.getDayOfWeeks());
                //按每月
            } else if (type.equals(JobTypeEnum.MONTH)) {
                // 一个月中的哪几天
                days(cron, model.getDayOfMonths());
                // 月份 周
                cron.append(" * ?");

            }
        }
        return cron.toString();
    }


    /**
     * 方法摘要：生成计划的详细描述
     *
     * @param model model
     * @return String
     */
    public static String createDescription(TaskScheduleModel model) {
        StringBuilder description = new StringBuilder();

        if (null != model.getSecond() && null != model.getMinute() && null != model.getHour()) {
            // 按每天
            JobTypeEnum type = model.getJobType();
            if (type.equals(JobTypeEnum.DAY)) {
                description.append("每天");
                description.append(model.getHour()).append("时");
                description.append(model.getMinute()).append("分");
                description.append(model.getSecond()).append("秒");
                description.append("执行");
                //按每周
            } else if (type.equals(JobTypeEnum.WEEK)) {
                if (CollectionUtils.isNotEmpty(model.getDayOfWeeks())) {
                    StringBuilder days = new StringBuilder();
                    if (isAll(model.getDayOfWeeks())) {
                        days.append("每天");
                    } else {
                        for (int i : model.getDayOfWeeks()) {
                            days.append(i).append("周");
                        }
                    }
                    description.append("每周的").append(days).append(" ");
                }
                if (null != model.getSecond() && null != model.getMinute() && null != model.getHour()) {
                    description.append(",");
                    description.append(model.getHour()).append("时");
                    description.append(model.getMinute()).append("分");
                    description.append(model.getSecond()).append("秒");
                }
                description.append("执行");
                //按每月
            } else if (type.equals(JobTypeEnum.MONTH)) {
                //选择月份
                if (CollectionUtils.isNotEmpty(model.getDayOfMonths())) {
                    StringBuilder days = new StringBuilder();
                    if (isAll(model.getDayOfMonths())) {
                        days.append("每天");
                    } else {
                        for (int i : model.getDayOfMonths()) {
                            days.append(i).append("号");
                        }
                    }
                    description.append("每月的").append(days).append(" ");
                }
                description.append(model.getHour()).append("时");
                description.append(model.getMinute()).append("分");
                description.append(model.getSecond()).append("秒");
                description.append("执行");
            }

        }
        return description.toString();
    }

    /**
     * 按照天去 追加
     * @param cron 表达式
     * @param days 天数
     */
    private static void days (StringBuilder cron, List<Integer> days){
        if (isAll(days)) {
            cron.append(BaseConstants.ALL);
        } else {
            days.forEach(d -> cron.append(BaseConstants.COMMA).append(d));
            StringUtil.delLast(cron);
        }
    }

    /**
     * 是不是每天
     * @param days 每天
     * @return 是否每天
     */
    private static boolean isAll (List<Integer> days){
        return days.contains(0);
    }
}
