package cn.sinozg.applet.quartz.use.scheduler;

import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.exception.CavException;
import cn.sinozg.applet.common.function.BiConsumerException;
import cn.sinozg.applet.common.function.ConsumerException;
import cn.sinozg.applet.common.utils.DateUtil;
import cn.sinozg.applet.common.utils.PojoUtil;
import cn.sinozg.applet.quartz.use.enums.JobKeyEnum;
import cn.sinozg.applet.quartz.use.enums.JobOptEnum;
import cn.sinozg.applet.quartz.use.enums.JobTypeEnum;
import cn.sinozg.applet.quartz.use.ext.DailyTimeIntervalMonthScheduleBuilder;
import cn.sinozg.applet.quartz.use.handler.JobHandlerInvoker;
import cn.sinozg.applet.quartz.use.module.SchedulerBaseParams;
import cn.sinozg.applet.quartz.use.module.SchedulerCronParams;
import cn.sinozg.applet.quartz.use.module.SchedulerCycleOnceParams;
import cn.sinozg.applet.quartz.use.module.SchedulerCycleTimesParams;
import cn.sinozg.applet.quartz.use.module.SchedulerSingleParams;
import cn.sinozg.applet.quartz.use.module.TaskScheduleModel;
import cn.sinozg.applet.quartz.use.util.CronUtils;
import org.apache.commons.collections4.CollectionUtils;
import org.quartz.CronScheduleBuilder;
import org.quartz.DailyTimeIntervalScheduleBuilder;
import org.quartz.DateBuilder;
import org.quartz.JobBuilder;
import org.quartz.JobDataMap;
import org.quartz.JobDetail;
import org.quartz.JobKey;
import org.quartz.ScheduleBuilder;
import org.quartz.Scheduler;
import org.quartz.SimpleScheduleBuilder;
import org.quartz.TimeOfDay;
import org.quartz.Trigger;
import org.quartz.TriggerBuilder;
import org.quartz.TriggerKey;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.time.LocalTime;
import java.util.Date;


/**
 * {@link org.quartz.Scheduler} 的管理器，负责创建任务
 * <p>
 * 考虑到实现的简洁性，我们使用 jobName 作为唯一标识，即：
 * 1. Job 的 {@link JobDetail#getKey()}<br/>
 * 2. Trigger 的 {@link Trigger#getKey()}
 * <p>
 * 另外，jobName 对应到 Spring Bean 的名字，直接调用
 *
 * @Author: xyb
 * @Description:
 * @Date: 2023-12-31 下午 12:12
 **/
public class SchedulerManager {

    private static final Logger log = LoggerFactory.getLogger(SchedulerManager.class);

    private final Scheduler scheduler;

    public SchedulerManager(Scheduler scheduler) {
        this.scheduler = scheduler;
    }

    /**
     * 添加 Job 到 Quartz 中
     *
     * @param params         任务具体信息
     */
    public <T extends SchedulerBaseParams> void addJob(T params) {
        validateScheduler();
        // 创建 JobDetail 对象
        String beanName = params.getBeanName();
        JobDetail jobDetail = JobBuilder.newJob(JobHandlerInvoker.class)
                .usingJobData(JobKeyEnum.JOB_ID.name(), params.getId())
                .usingJobData(JobKeyEnum.JOB_TENANT.name(), params.getTenantId())
                .usingJobData(JobKeyEnum.JOB_NAME.name(), params.getJobName())
                .usingJobData(JobKeyEnum.JOB_BEAN_NAME.name(), beanName)
                .withIdentity(params.getId()).build();
        // 创建 Trigger 对象
        Trigger trigger = this.buildTrigger(params);
        // 新增调度
        this.schedulerRun(jobDetail, trigger, JobOptEnum.ADD, scheduler::scheduleJob);
    }


    /**
     * 更新 Job 到 Quartz
     *
     * @param params      任务具体信息
     */
    public <T extends SchedulerBaseParams> void updateJob(T params) {
        validateScheduler();
        // 创建新 Trigger 对象
        Trigger trigger = this.buildTrigger(params);
        // 修改调度
        this.schedulerRun(new TriggerKey(params.getId()), trigger, JobOptEnum.UPDATE, scheduler::rescheduleJob);
    }

    /**
     * 删除 Quartz 中的 Job
     *
     * @param id 任务处理器的名字
     */
    public void deleteJob(String id) {
        validateScheduler();
        this.schedulerRun(new JobKey(id), JobOptEnum.DEL, scheduler::deleteJob);
    }

    /**
     * 暂停 Quartz 中的 Job
     *
     * @param id 任务处理器的名字
     */
    public void pauseJob(String id) {
        validateScheduler();
        this.schedulerRun(new JobKey(id), JobOptEnum.PAUSE, scheduler::pauseJob);
    }

    /**
     * 启动 Quartz 中的 Job
     *
     * @param id 任务处理器的名字
     */
    public void resumeJob(String id) {
        validateScheduler();
        this.schedulerRun(new JobKey(id), JobOptEnum.RESUME, scheduler::resumeJob);
        this.schedulerRun(new TriggerKey(id), JobOptEnum.RESUME, scheduler::resumeTrigger);
    }

    /**
     * 立即触发一次 Quartz 中的 Job
     *
     * @param params    任务信息
     */
    public <T extends SchedulerBaseParams> void triggerJob(T params) {
        validateScheduler();
        // 触发任务 无需重试，所以不设置 retryCount 和 retryInterval
        JobDataMap data = new JobDataMap();
        String beanName = params.getBeanName();
        data.put(JobKeyEnum.JOB_ID.name(), params.getId());
        data.put(JobKeyEnum.JOB_TENANT.name(), params.getTenantId());
        data.put(JobKeyEnum.JOB_BEAN_NAME.name(), beanName);
        data.put(JobKeyEnum.JOB_ARGS.name(), params.getArgs());
        this.schedulerRun(new JobKey(params.getId()), data, JobOptEnum.RESUME, scheduler::triggerJob);
    }

    /**
     * 构建 Trigger
     * @param params 参数
     * @return Trigger
     * @param <T> Trigger
     */
    private <T extends SchedulerBaseParams> Trigger buildTrigger(T params) {
        boolean isSingle = params instanceof SchedulerSingleParams;
        ScheduleBuilder<? extends Trigger> schedule;
        Date d = null;
        if (isSingle) {
            schedule = SimpleScheduleBuilder.simpleSchedule()
                    .withRepeatCount(0);
            SchedulerSingleParams p = (SchedulerSingleParams) params;
            d = new Date(DateUtil.epochMilli(p.getSingleTime()));
        } else if (params instanceof SchedulerCycleTimesParams timesParams) {
            schedule = dailyBuilder(timesParams);
        } else {
            String cron;
            if (params instanceof SchedulerCronParams p) {
                cron = p.getCron();
            } else {
                SchedulerCycleOnceParams p = (SchedulerCycleOnceParams) params;
                // 生成cron
                TaskScheduleModel model = new TaskScheduleModel();
                model.setJobType(JobTypeEnum.MONTH);
                model.setDayOfMonths(p.getRunDays());
                LocalTime time = p.getRunTime();
                model.setHour(time.getHour());
                model.setMinute(time.getMinute());
                model.setSecond(time.getSecond());
                cron = CronUtils.cronExpression(model);
            }
            schedule = CronScheduleBuilder.cronSchedule(cron);
        }
        TriggerBuilder<? extends Trigger> triggerBuilder = buildTrigger(schedule, params);
        if (isSingle) {
            triggerBuilder.startAt(d);
        }
        return triggerBuilder.build();
    }

    /**
     * 按月 或者周 周期执行
     * @param params 参数
     * @return 对象
     */
    private ScheduleBuilder<? extends Trigger> dailyBuilder (SchedulerCycleTimesParams params){
        ScheduleBuilder<? extends Trigger> schedule = null;
        // 非所有日期执行
        Integer[] days = null;
        TimeOfDay startingDailyAt = null;
        TimeOfDay endingDailyAt = null;
        LocalTime lt = params.getBeginTime();
        if (lt != null) {
            startingDailyAt = TimeOfDay.hourMinuteAndSecondOfDay(lt.getHour(), lt.getMinute(), lt.getSecond());
        }
        lt = params.getEndTime();
        if (lt != null) {
            endingDailyAt = TimeOfDay.hourMinuteAndSecondOfDay(lt.getHour(), lt.getMinute(), lt.getSecond());
        }
        DateBuilder.IntervalUnit unit = DateBuilder.IntervalUnit.HOUR;
        if (BaseConstants.STATUS_01.equals(params.getIntervalUnit())) {
            unit = DateBuilder.IntervalUnit.SECOND;
        } else if (BaseConstants.STATUS_02.equals(params.getIntervalUnit())) {
            unit = DateBuilder.IntervalUnit.MINUTE;
        }
        if (CollectionUtils.isNotEmpty(params.getRunDays()) && !params.getRunDays().contains(0)) {
            // 非每天
            days = PojoUtil.toArray(params.getRunDays(), Integer.class);
            if (BaseConstants.STATUS_02.equals(params.getJobType())) {
                schedule = DailyTimeIntervalMonthScheduleBuilder.dailyTimeIntervalSchedule()
                        .startingDailyAt(startingDailyAt).endingDailyAt(endingDailyAt)
                        .withInterval(params.getIntervalTime(), unit)
                        .onDaysOfTheMonth(days);
            }
        }
        if (schedule == null) {
            schedule = DailyTimeIntervalScheduleBuilder.dailyTimeIntervalSchedule()
                    .startingDailyAt(startingDailyAt).endingDailyAt(endingDailyAt)
                    .withInterval(params.getIntervalTime(), unit)
                    .onDaysOfTheWeek(days);
        }
        return schedule;
    }

    /**
     * 构建 Trigger
     * @param schedule schedule
     * @param params 参数
     * @return Trigger
     * @param <T> 类型
     */
    private <T extends Trigger> TriggerBuilder<T> buildTrigger(ScheduleBuilder<T> schedule, SchedulerBaseParams params){
        TriggerBuilder<T> builder = TriggerBuilder.newTrigger()
                .withIdentity(params.getId())
                .withSchedule(schedule)
                .usingJobData(JobKeyEnum.JOB_ARGS.name(), params.getArgs())
                .usingJobData(JobKeyEnum.JOB_RETRY_COUNT.name(), params.getRetryCount())
                .usingJobData(JobKeyEnum.JOB_RETRY_INTERVAL.name(), params.getRetryInterval());
        if (params.getEndDate() != null) {
            builder.endAt(new Date(DateUtil.epochMilli(params.getEndDate())));
        }
        if (params.getBeginDate() != null) {
            builder.startAt(new Date(DateUtil.epochMilli(params.getBeginDate())));
        }
        return builder;
    }


    /**
     * 验证定时任务是否开启
     */
    private void validateScheduler() {
        if (scheduler == null) {
            throw new CavException("BIZ000100044");
        }
    }

    private <T, U> void schedulerRun (T t, U u, JobOptEnum opt, BiConsumerException<T, U> consumer) {
        try {
            consumer.accept(t, u);
        } catch (Exception e) {
            throw new CavException("BIZ000100045", opt.getName(), e);
        }
    }

    private <T> void schedulerRun (T t, JobOptEnum opt, ConsumerException<T> consumer) {
        try {
            consumer.accept(t);
        } catch (Exception e) {
            throw new CavException("BIZ000100045", opt.getName(), e);
        }
    }
}
