/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.quartz.use.ext;

import org.quartz.DailyTimeIntervalTrigger;
import org.quartz.DateBuilder;
import org.quartz.Scheduler;
import org.quartz.TimeOfDay;
import org.quartz.Trigger;
import org.quartz.TriggerBuilder;

import java.util.Calendar;
import java.util.Set;

/**
 * 按月计算
 * @author xieyubin
 * @Description
 * @Copyright Copyright (c) 2024
 * @since 2024-01-15 14:26
 */
interface DailyTimeIntervalMonthTrigger extends Trigger {

    /**
     * <p>
     * Used to indicate the 'repeat count' of the trigger is indefinite. Or in
     * other words, the trigger should repeat continually until the trigger's
     * ending timestamp.
     * </p>
     */
    int REPEAT_INDEFINITELY = -1;

    /**
     * <p>
     * Instructs the <code>{@link Scheduler}</code> that upon a mis-fire
     * situation, the <code>{@link DailyTimeIntervalTrigger}</code> wants to be
     * fired now by <code>Scheduler</code>.
     * </p>
     */
    int MISFIRE_INSTRUCTION_FIRE_ONCE_NOW = 1;

    /**
     * <p>
     * Instructs the <code>{@link Scheduler}</code> that upon a mis-fire
     * situation, the <code>{@link DailyTimeIntervalTrigger}</code> wants to have it's
     * next-fire-time updated to the next time in the schedule after the
     * current time (taking into account any associated <code>{@link Calendar}</code>,
     * but it does not want to be fired now.
     * </p>
     */
    int MISFIRE_INSTRUCTION_DO_NOTHING = 2;

    /**
     * <p>Get the interval unit - the time unit on with the interval applies.</p>
     *
     * <p>The only intervals that are valid for this type of trigger are {@link DateBuilder.IntervalUnit#SECOND},
     * {@link DateBuilder.IntervalUnit#MINUTE}, and {@link DateBuilder.IntervalUnit#HOUR}.</p>
     * @return IntervalUnit
     */
    DateBuilder.IntervalUnit getRepeatIntervalUnit();

    /**
     * <p>
     * Get the the number of times for interval this trigger should
     * repeat, after which it will be automatically deleted.
     * </p>
     *
     * @see #REPEAT_INDEFINITELY
     * @return int
     */
    int getRepeatCount();

    /**
     * <p>
     * Get the the time interval that will be added to the <code>DateIntervalTrigger</code>'s
     * fire time (in the set repeat interval unit) in order to calculate the time of the
     * next trigger repeat.
     * @return repeatInterval
     */
    int getRepeatInterval();

    /**
     * The time of day to start firing at the given interval.
     * @return startTimeOfDay
     */
    TimeOfDay getStartTimeOfDay();

    /**
     * The time of day to complete firing at the given interval.
     * @return endTimeOfDay
     */
    TimeOfDay getEndTimeOfDay();

    /**
     * The days of the month upon which to fire.
     *
     * @return 1-31
     */
    Set<Integer> getDaysOfMonth();

    /**
     * <p>
     * Get the number of times the <code>DateIntervalTrigger</code> has already
     * fired.
     * </p>
     * @return timesTriggered
     */
    int getTimesTriggered();

    /**
     * triggerBuilder
     * @return triggerBuilder
     */
    @Override
    TriggerBuilder<DailyTimeIntervalMonthTrigger> getTriggerBuilder();
}
