package cn.sinozg.applet.opt.util;

import cn.sinozg.applet.common.annotation.OptTie;
import cn.sinozg.applet.common.utils.DateUtil;
import cn.sinozg.applet.common.utils.JsonUtil;
import cn.sinozg.applet.common.utils.PojoUtil;
import cn.sinozg.applet.common.utils.WebUtil;
import cn.sinozg.applet.opt.cache.OptLogCache;
import cn.sinozg.applet.opt.constant.OptLogConstant;
import cn.sinozg.applet.opt.enums.ModeEnum;
import cn.sinozg.applet.opt.module.AnalysisMiddleware;
import cn.sinozg.applet.opt.module.OptLogDiff;
import cn.sinozg.applet.opt.module.OptLogDiffPair;
import cn.sinozg.applet.opt.module.OptLogDiffTable;
import cn.sinozg.applet.opt.module.OptLogDiffTableChange;
import cn.sinozg.applet.opt.module.OptLogTableDetail;
import cn.sinozg.applet.opt.module.OptLogTableInfo;
import cn.sinozg.applet.opt.module.OptMapperColumn;
import cn.sinozg.applet.opt.module.OptTieCache;
import cn.sinozg.applet.opt.module.OptTieRecord;
import cn.sinozg.applet.opt.module.OptTieRelation;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.aspectj.lang.JoinPoint;
import org.springframework.http.HttpMethod;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.servlet.HandlerMapping;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentSkipListMap;

/**
 * 工具类
 * @author xieyubin
 * @Description
 * @Copyright Copyright (c) 2024
 * @since 2024-02-27 19:57
 */
@Slf4j
public class OptUtil {

    /** 最长的字符串 **/
    private static final int MAX_LENGTH = 2000;

    /**
     * 获取方法的第一个参数范型类型
     * @param method 方法
     * @param requestMethod 请求类型
     * @return 参数类型
     */
    public static Class<?> firstMethodParams(Method method, String requestMethod){
        if (HttpMethod.POST.name().equals(requestMethod)) {
            Type[] types = method.getGenericParameterTypes();
            Type type = types[0];
            if (type instanceof ParameterizedType) {
                ParameterizedType parameterizedType = (ParameterizedType) type;
                Type[] typeArguments = parameterizedType.getActualTypeArguments();
                Type gt = typeArguments[0];
                if (gt instanceof Class) {
                    Class<?> clazz = PojoUtil.cast(gt);
                    if (clazz != List.class && clazz != Map.class && clazz != Arrays.class && clazz != String.class && !clazz.isPrimitive()) {
                        return clazz;
                    }
                }
            }
        }
        return null;
    }

    /**
     * 获取到请求参数
     * @param requestMethod 请求类型
     * @param joinPoint joinPoint
     * @return 参数 json 或者字符串
     */
    public static String requestParamsToJson (String requestMethod, JoinPoint joinPoint){
        Object params;
        HttpServletRequest request = WebUtil.request();
        if (StringUtils.equalsAny(requestMethod, HttpMethod.PUT.name(), HttpMethod.POST.name())) {
            StringBuilder args = new StringBuilder();
            Object[] paramsArray =joinPoint.getArgs();
            if (paramsArray != null) {
                for (Object o : paramsArray) {
                    if (!isFilterObject(o)) {
                        String jsonObj = JsonUtil.toJson(o);
                        args.append(jsonObj).append(StringUtils.SPACE);
                    }
                }
            }
            params = args.toString();
        } else if (HttpMethod.GET.name().equals(requestMethod)) {
            params = request.getParameterMap();
        } else {
            params = request.getAttribute(HandlerMapping.URI_TEMPLATE_VARIABLES_ATTRIBUTE);
        }
        return substring(params);
    }

    /**
     * 排除全局表 或者字段
     * @param info 表名称或者字段名称
     * @param include 含有
     * @param exclude 排除
     * @return 是否满足
     */
    public static boolean exclude(String info, List<String> include, List<String> exclude){
        if (CollectionUtils.isNotEmpty(exclude)) {
            if (exclude.contains(info)) {
                return true;
            }
        }
        if (CollectionUtils.isNotEmpty(include)) {
            return !include.contains(info);
        }
        return false;
    }


    public static String substring (Object params){
        if (params == null) {
            return null;
        }
        String json;
        if (params instanceof String) {
            json = (String) params;
        } else {
            json = JsonUtil.toJson(params);
        }
        if (json != null && json.length() > MAX_LENGTH) {
            json = json.substring(0, MAX_LENGTH);
        }
        return json;
    }

    /**
     * 递归获取映射关系
     * @param clazz 类
     * @return 映射关系
     */
    public static <T> OptTieRelation<T> optTie(Class<T> clazz){
        Map<Class<?>, OptTieRecord> keyMap = new HashMap<>(16);
        keyMap.putIfAbsent(clazz, new OptTieRecord(OptLogConstant.DEFAULT_ID, null));
        List<OptTieCache<?>> list = optTieList(clazz, keyMap);
        OptTieCache.OptTieBuild<T> build = OptTieCache.build(clazz)
                .array(false)
                .subs(list);
        OptTieRelation<T> relation = new OptTieRelation<>();
        relation.setTie(build.build());
        relation.setKeyMap(keyMap);
        return relation;
    }

    /**
     * 所有的 对比 按照表显示
     * 不按照父子关系
     * @param mom 数据
     * @return 差异集合
     */
    public static String allDiffJson (AnalysisMiddleware mom){
        List<OptLogDiffTable> list = new ArrayList<>();
        Map<Class<?>, List<OptLogTableInfo>> map = mom.getMap();
        for (Map.Entry<Class<?>, List<OptLogTableInfo>> e : map.entrySet()) {
            OptLogDiffTable table = getOneTableDiff(e.getValue(), mom);
            list.add(table);
        }
        if (CollectionUtils.isNotEmpty(list)) {
            list.sort(Comparator.comparing(OptLogDiffTable::getDesc));
            return JsonUtil.toJson(list);
        }
        return null;
    }

    /**
     * 对比信息 按照依赖关系返回
     * @param mom 参数
     * @param tie 依赖关系结构
     */
    public static void diffWithDependency (AnalysisMiddleware mom, Class<?> tie){
        // 如果只有一层 直接写，不用依赖关系 最外层的 diff json
        if (mom.getMap().size() == 1) {
            List<OptLogTableInfo> tables = mom.getMap().values().iterator().next();
            // 初始化table
            OptLogDiffTable table = getOneTableDiff(tables, mom);
            mom.setDiff(table);
        } else {
            OptTieRelation<?> optTie = OptLogCache.optTie(tie);
            firstEntityClass(mom, optTie);
            itemTale(optTie.getTie(), mom, null,  null);
        }
    }



    /**
     * 对比一次sql 执行后的数据差异
     * @param newValues 新值
     * @param oldValues 旧数据
     * @param propMap 属性
     * @param tableDetail 表详情
     * @param modeEnum 修改类型
     * @return 具体信息
     */
    public static List<OptLogTableInfo> compare (List<Map<String, Object>> newValues, Map<String, Map<String, Object>> oldValues, Map<String, OptMapperColumn> propMap,
                                                 OptLogTableDetail tableDetail, ModeEnum modeEnum){
        /**
         * 新增 newValues != null  and oldValues == null
         * 删除 newValues == null and oldValues != null
         * 修改 newValues != null and oldValues != null
         */
        List<OptLogTableInfo> list = new ArrayList<>();
        if (ModeEnum.UPDATE == modeEnum) {
            // 批量
            boolean batch = newValues.size() > 1;
            for (Map.Entry<String, Map<String, Object>> e : oldValues.entrySet()) {
                // 如果是批量 就匹配
                Map<String, Object> nm = newValues.get(0);
                if (batch) {
                    String idName = tableDetail.getIdName();
                    for (Map<String, Object> nv : newValues) {
                        String id = MapUtils.getString(nv, idName);
                        if (e.getKey().equals(id)) {
                            nm = nv;
                            break;
                        }
                    }
                }
                OptLogTableInfo tableInfo = compareTable(nm, e.getValue(), propMap, tableDetail, modeEnum);
                if (tableInfo != null) {
                    list.add(tableInfo);
                }
            }
        } else if (ModeEnum.DELETE == modeEnum) {
            for (Map.Entry<String, Map<String, Object>> e : oldValues.entrySet()) {
                OptLogTableInfo tableInfo = compareTable(null, e.getValue(), propMap, tableDetail, modeEnum);
                if (tableInfo != null) {
                    list.add(tableInfo);
                }
            }
            // insert
        } else {
            for (Map<String, Object> nv : newValues) {
                OptLogTableInfo tableInfo = compareTable(nv, null, propMap, tableDetail, modeEnum);
                if (tableInfo != null) {
                    list.add(tableInfo);
                }
            }
        }
        return list;
    }

    /**
     * 对比表一条记录的数据差异
     * @param newValues 新值
     * @param oldValues 旧数据
     * @param propMap 属性
     * @param tableDetail 表详情
     * @param modeEnum 修改类型
     * @return 具体信息
     */
    private static OptLogTableInfo compareTable (Map<String, Object> newValues, Map<String, Object> oldValues, Map<String, OptMapperColumn> propMap,
                                                 OptLogTableDetail tableDetail, ModeEnum modeEnum) {
        OptLogDiff diff;
        OptMapperColumn c;
        Object nv, ov;
        String key;
        String idName = tableDetail.getIdName();
        String foreignIdName = tableDetail.getForeignIdName();
        String id;
        String foreignId = null;
        // 以数据库的为准
        if (oldValues != null) {
            if (StringUtils.isNotBlank(foreignIdName)) {
                foreignId = MapUtils.getString(oldValues, foreignIdName);
            }
            id = MapUtils.getString(oldValues, idName);
        } else {
            if (StringUtils.isNotBlank(foreignIdName)) {
                foreignId = MapUtils.getString(newValues, foreignIdName);
            }
            id = MapUtils.getString(newValues, idName);
        }
        List<OptLogDiff> fieldDiff = new ArrayList<>();
        for (Map.Entry<String, OptMapperColumn> e : propMap.entrySet()) {
            key = e.getKey();
            // 新值 不存在 说明可能是外键
            if (newValues != null && !newValues.containsKey(key)) {
                continue;
            }
            nv = MapUtils.getObject(newValues, key);
            ov = MapUtils.getObject(oldValues, key);
            Pair<String, String> pair = valueIsEquals(ov, nv);
            if (pair == null) {
                continue;
            }
            c = e.getValue();
            diff = new OptLogDiff();
            diff.setDesc(c.getColumnDesc());
            diff.setNm(c.getPropertyName());
            diff.setNv(pair.getValue());
            diff.setOv(pair.getKey());
            fieldDiff.add(diff);
        }
        if (CollectionUtils.isEmpty(fieldDiff)) {
            return null;
        }
        OptLogTableInfo tableInfo = new OptLogTableInfo();
        tableInfo.setIdValue(id);
        tableInfo.setForeignId(foreignId);
        tableInfo.setDiff(fieldDiff);
        tableInfo.setMode(modeEnum);
        tableInfo.setTableDetail(tableDetail);
        return tableInfo;
    }

    /**
     * 判断对象是否相等
     * @param oldValue oldValue
     * @param newValue newValue
     * @return 是否相等
     */
    private static Pair<String, String> valueIsEquals(Object oldValue, Object newValue) {
        // 判断是否相等
        if (oldValue == null && newValue == null) {
            return null;
        }
        String ov = formatColumnValue(oldValue);
        String nv = formatColumnValue(newValue);
        if (oldValue == null || newValue == null) {
            return Pair.of(ov, nv);
        }
        // 数字类型
        if (oldValue instanceof Long || oldValue instanceof Integer || oldValue instanceof Double || oldValue instanceof Float || oldValue instanceof BigDecimal
                || oldValue instanceof Short || oldValue instanceof Byte) {
            BigDecimal o = new BigDecimal(oldValue.toString());
            BigDecimal n = new BigDecimal(newValue.toString());
            return o.compareTo(n) == 0 ? null : Pair.of(ov, nv);
        }
        // 日期 布尔 字符串
        return StringUtils.equals(ov, nv) ? null : Pair.of(ov, nv);
    }

    /**
     * 不同类型的数据格式化
     *
     * @param value 格式化值
     * @return 格式化
     */
    public static String formatColumnValue(Object value) {
        if (value == null) {
            return null;
        }
        try {
            if (value instanceof Date) {
                LocalDateTime ldt = DateUtil.ldt(((Date) value).getTime());
                // 如果时分秒都为空，则使用年月日格式化
                if (ldt.getHour() == 0 && ldt.getMinute() == 0 && ldt.getSecond() == 0) {
                    return DateUtil.formatDate(ldt.toLocalDate(), null);
                } else {
                    // 使用年月日时分秒格式化
                    return DateUtil.formatDateTime(ldt, null);
                }
            } else if (value instanceof LocalDate) {
                LocalDate localDate = (LocalDate) value;
                return DateUtil.formatDate(localDate, null);
            } else if (value instanceof LocalDateTime) {
                LocalDateTime localDateTime = (LocalDateTime) value;
                return DateUtil.formatDateTime(localDateTime, null);
            }
        } catch (Exception e) {
            log.error("格式化字符串异常，跳过", e);
        }
        return value.toString();
    }



    /**
     * 递归获取映射关系
     * @param clazz 类
     * @param keyMap map
     * @return 映射关系
     */
    private static List<OptTieCache<?>> optTieList(Class<?> clazz, Map<Class<?>, OptTieRecord> keyMap){
        if (clazz == null) {
            return null;
        }
        List<Field> fields = PojoUtil.allField(clazz);
        List<OptTieCache<?>> list = new ArrayList<>();
        for (Field field : fields) {
            OptTie optTie = field.getAnnotation(OptTie.class);
            if (optTie != null ) {
                boolean array = field.getType() == List.class;
                Class<?> c = optTie.clazz();
                keyMap.putIfAbsent(c, new OptTieRecord(optTie.idName(), optTie.foreignName()));
                OptTieCache.OptTieBuild<?> sub = OptTieCache.build(c)
                        .array(array).key(optTie.foreignName())
                        .desc(optTie.desc())
                        .subs(optTieList(c, keyMap));
                list.add(sub.build());
            }
        }
        return list;
    }

    /**
     * 迭代处理数据
     * @param tie 依赖
     * @param mom 参数
     * @param diffTable 父级diff 信息
     * @param pId 父id
     */
    private static void itemTale(OptTieCache<?> tie, AnalysisMiddleware mom, OptLogDiffTable diffTable, String pId){
        if (StringUtils.isBlank(pId) && tie == null) {
            return;
        }
        boolean isFirst = diffTable == null && StringUtils.isBlank(pId);
        Class<?> currClazz = tie.getClazz();
        if (isFirst) {
            currClazz = mom.getMainClazz();
        }
        List<OptLogTableInfo> list = null;
        if (currClazz == null) {
            log.error("未找到主表类型！！！");
        } else {
            list = MapUtils.getObject(mom.getMap(), currClazz);
        }
        if (CollectionUtils.isEmpty(list)) {
            return;
        }
        // 当前表
        OptLogDiffTable currTable = initDiffTable(list);
        // 本表的修改记录
        Map<ModeEnum, OptLogDiffTableChange> subChange = new HashMap<>(16);
        for (OptLogTableInfo t : list) {
            if (isFirst || pId.equals(t.getForeignId())) {
                setTableDiffByMode(subChange, t, mom);
                // 子表
                List<OptTieCache<?>> subClazz = tie.getSubs();
                if (CollectionUtils.isNotEmpty(subClazz)) {
                    for (OptTieCache<?> ti : subClazz) {
                        itemTale(ti, mom, currTable, t.getIdValue());
                    }
                }
            }
        }
        setChangeList(subChange, currTable);
        // 设置字表信息 非root级别
        if (diffTable != null) {
            PojoUtil.setBeanList(diffTable, currTable, OptLogDiffTable::getTs, OptLogDiffTable::setTs);
        } else {
            mom.setDiff(currTable);
        }
    }


    /**
     * 根据类型 设置表的修改信息
     * @param map map
     * @param t 表数据
     * @param mom 原始数据
     */
    private static void setTableDiffByMode (Map<ModeEnum, OptLogDiffTableChange> map, OptLogTableInfo t, AnalysisMiddleware mom){
        Map<String, OptLogDiffPair> diff = new ConcurrentSkipListMap<>();
        List<OptLogDiff> ds = t.getDiff();
        for (OptLogDiff d : ds) {
            diff.put(d.getNm(), new OptLogDiffPair(d.getOv(), d.getNv()));
        }
        OptLogDiffTableChange change = map.get(t.getMode());
        if (change == null) {
            change = new OptLogDiffTableChange();
            change.setMode(t.getMode().getCode());
        }
        PojoUtil.setBeanList(change, diff, OptLogDiffTableChange::getLs, OptLogDiffTableChange::setLs);
        map.put(t.getMode(), change);
        // 设置实体对象
        mom.addTable(t);
    }

    /**
     * 设置一个类型的表的数据
     * @param tables 数据
     * @param mom 参数
     * @return 表信息
     */
    private static OptLogDiffTable getOneTableDiff (List<OptLogTableInfo> tables, AnalysisMiddleware mom){
        // 初始化table
        OptLogDiffTable table = initDiffTable(tables);
        Map<ModeEnum, OptLogDiffTableChange> changeMap = new HashMap<>(16);
        for (OptLogTableInfo t : tables) {
            setTableDiffByMode(changeMap, t, mom);
        }
        setChangeList(changeMap, table);
        return table;
    }

    /**
     * 设置 对比结果到一个表的集合里面去
     * @param changes 对比结果
     * @param table 表
     */
    private static void setChangeList (Map<ModeEnum, OptLogDiffTableChange> changes, OptLogDiffTable table){
        if (MapUtils.isNotEmpty(changes)) {
            List<OptLogDiffTableChange> temp = new ArrayList<>(changes.values());
            table.setCs(temp);
        }
    }

    /**
     * 获取主表的类型
     * @param mom 已经处理后的表数据
     * @param optTie 依赖
     */
    private static void firstEntityClass (AnalysisMiddleware mom, OptTieRelation<?> optTie){
        OptTieCache<?> tie = optTie.getTie();
        Class<?> clazz = tie.getClazz();
        // 如果设置的依赖有主表则返回
        Map<Class<?>, List<OptLogTableInfo>> entityMap = mom.getMap();
        Class<?> mainClazz = null;
        if (entityMap.containsKey(clazz)) {
            mainClazz = clazz;
        } else {
            Map<Class<?>, OptTieRecord> keyMap = optTie.getKeyMap();
            List<Class<?>> list = new ArrayList<>();
            for (Map.Entry<Class<?>, List<OptLogTableInfo>> e : entityMap.entrySet()) {
                if (!keyMap.containsKey(e.getKey())) {
                    list.add(e.getKey());
                }
            }
            if (list.size() == 1) {
                mainClazz = list.get(0);

            }
        }
        mom.setMainClazz(mainClazz);
    }

    /**
     * 初始化设置表的diff信息
     * @param list 数据集合
     * @return 表信息
     */
    private static OptLogDiffTable initDiffTable (List<OptLogTableInfo> list){
        OptLogTableInfo tableInfo = list.get(0);
        OptLogDiffTable table = new OptLogDiffTable();
        table.setDesc(tableInfo.getTableDetail().getTableDesc());
        List<OptLogDiff> ds = tableInfo.getDiff();
        Map<String, String> fs = new ConcurrentSkipListMap<>();
        for (OptLogDiff d : ds) {
            fs.put(d.getNm(), d.getDesc());
        }
        table.setFs(fs);
        return table;
    }

    /**
     * 判断是否需要过滤的对象。
     *
     * @param o 对象信息。
     * @return 如果是需要过滤的对象，则返回true；否则返回false。
     */
    private static boolean isFilterObject(final Object o) {
        return o instanceof MultipartFile || o instanceof HttpServletRequest || o instanceof HttpServletResponse;
    }
}
