package cn.sinozg.applet.common.aspectj;

import cn.sinozg.applet.common.holder.MdcContextHolder;
import cn.sinozg.applet.common.service.FrameworkAuthService;
import cn.sinozg.applet.common.utils.OsUtil;
import cn.sinozg.applet.common.utils.SnowFlake;
import cn.sinozg.applet.common.utils.UserUtil;
import cn.sinozg.applet.common.utils.WebUtil;
import cn.sinozg.applet.opt.cache.OptLogCache;
import cn.sinozg.applet.opt.cache.OptLogThreadCache;
import cn.sinozg.applet.opt.config.OptLogProperties;
import cn.sinozg.applet.opt.module.OptLogCycleInfo;
import cn.sinozg.applet.opt.module.OptLogMain;
import cn.sinozg.applet.opt.service.OptLogService;
import cn.sinozg.applet.opt.util.OptUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.AfterReturning;
import org.aspectj.lang.annotation.AfterThrowing;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.annotation.Before;
import org.aspectj.lang.reflect.MethodSignature;

import javax.servlet.http.HttpServletRequest;
import java.lang.reflect.Method;
import java.util.ArrayList;

/**
 * @author xieyubin
 * @Description
 * @Copyright Copyright (c) 2024
 * @since 2024-02-27 20:52
 */
@Slf4j
@Aspect
public class OptLogDataAspect {

    private final OptLogProperties recordProperties;

    private final OptLogService optLogService;

    private final FrameworkAuthService authService;

    /** 返回值对象名称 */
    private static final String RESULT = "result";

    /** 异常对象参数名称 */
    private static final String EXCEPTION = "exception";

    /** 切点 */
    private static final String POINTCUT = "@annotation(cn.sinozg.applet.common.annotation.OptLog)";

    public OptLogDataAspect(OptLogProperties properties, OptLogService optLogService, FrameworkAuthService authService){
        this.recordProperties = properties;
        this.optLogService = optLogService;
        this.authService = authService;
    }

    /**
     * 方法执行之前
     *
     * @param joinPoint joinPoint
     */
    @Before(POINTCUT)
    public void doBefore(JoinPoint joinPoint) {
        try {
            OptLogCycleInfo result = new OptLogCycleInfo();
            String traceId = MdcContextHolder.get();
            if (StringUtils.isBlank(traceId)) {
                traceId = SnowFlake.genId();
            }
            result.setTraceId(traceId);
            result.setThreadName(Thread.currentThread().getName());
            // 获取请求信息 获取当前的HttpServletRequest对象
            HttpServletRequest request = WebUtil.request();
            Signature signature = joinPoint.getSignature();
            MethodSignature methodSignature = (MethodSignature) signature;
            Method method = methodSignature.getMethod();
            Class<?> controller = joinPoint.getTarget().getClass();
            OptLogMain main = OptLogCache.apiBaseInfo(controller, method, recordProperties.isDataLog(), request);
            if (main.getTie() != null) {
                result.setItems(new ArrayList<>());
            }
            if (main.isSaveParams()) {
                String params = OptUtil.requestParamsToJson(request.getMethod(), joinPoint);
                main.setJsonParam(params);
            }
            String ip = OsUtil.getIpAddr(request);
            main.setOptIp(ip);
            result.setMain(main);
            // 设置到线程变量中
            OptLogThreadCache.set(result);
        } catch (Exception e) {
            log.error("获取请求信息错误", e);
        }
    }

    /**
     * 方法正常执行并成功返回
     *
     * @param joinPoint joinPoint
     * @param result result
     */
    @AfterReturning(value = POINTCUT, returning = RESULT)
    public void doAfterReturning(JoinPoint joinPoint, Object result) {
        try {
            // 从线程变量中获取
            OptLogCycleInfo cycleInfo = OptLogThreadCache.get();
            String token = authService.token();
            optLogService.saveLog(token, UserUtil.user(), cycleInfo, null, result);
        } catch (Exception e) {
            log.error("记录日志错误！", e);
        } finally {
            // 释放资源
            OptLogThreadCache.remove();
        }

    }

    /**
     * 异常处理
     *
     * @param exception 异常
     */
    @AfterThrowing(value = POINTCUT, throwing = EXCEPTION)
    protected void doAfterThrowing(Exception exception) {
        try {
            // 从线程变量中获取
            OptLogCycleInfo cycleInfo = OptLogThreadCache.get();
            String token = authService.token();
            optLogService.saveLog(token, UserUtil.user(), cycleInfo, exception, null);
        } catch (Exception e) {
            log.error("日志记录异常！", e);
        } finally {
            // 释放资源
            OptLogThreadCache.remove();
        }
    }
}
