/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.biz.opt.service.impl;

import cn.sinozg.applet.biz.opt.entity.OptLogData;
import cn.sinozg.applet.biz.opt.entity.OptLogTable;
import cn.sinozg.applet.biz.opt.entity.OptLogText;
import cn.sinozg.applet.biz.opt.service.OptLogDataService;
import cn.sinozg.applet.biz.opt.service.OptLogTableService;
import cn.sinozg.applet.biz.opt.service.OptLogTextService;
import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.core.model.LoginUserVo;
import cn.sinozg.applet.common.holder.UserContextHolder;
import cn.sinozg.applet.common.utils.JsonUtil;
import cn.sinozg.applet.common.utils.OsUtil;
import cn.sinozg.applet.common.utils.PojoUtil;
import cn.sinozg.applet.opt.module.AnalysisMiddleware;
import cn.sinozg.applet.opt.module.OptLogCycleInfo;
import cn.sinozg.applet.opt.module.OptLogMain;
import cn.sinozg.applet.opt.module.OptLogTableDetail;
import cn.sinozg.applet.opt.module.OptLogTableInfo;
import cn.sinozg.applet.opt.service.OptLogService;
import cn.sinozg.applet.opt.util.OptUtil;
import jakarta.annotation.Resource;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Map;

/**
 * @author xieyubin
 * @Description
 * @Copyright Copyright (c) 2024
 * @since 2024-02-28 18:12
 */
@Slf4j
@Service
public class OptLogServiceImpl implements OptLogService {

    @Resource
    private OptLogTextService textService;

    @Resource
    private OptLogDataService logDataService;

    @Resource
    private OptLogTableService logTableService;

    @Async
    @Override
    @Transactional(rollbackFor = Throwable.class)
    public void saveLog(String token, LoginUserVo user, OptLogCycleInfo cycleInfo, Exception e, Object jsonResult) {
        // 日志记录 和数据
        OptLogMain main = cycleInfo.getMain();
        OptLogData logData = PojoUtil.copyBean(main, OptLogData.class);
        logData.setTraceId(cycleInfo.getTraceId());

        logData.setOptLocation(OsUtil.realAddressByIp(logData.getOptIp()));
        OptLogText logText = new OptLogText();
        if (e != null) {
            logData.setDataStatus(BaseConstants.STATUS_STOP);
            logText.setErrorMsg(OptUtil.substring(e.getMessage()));
        } else {
            logData.setDataStatus(BaseConstants.STATUS_COMMON);
        }
        if (main.isSaveParams()) {
            logText.setJsonParam(main.getJsonParam());
            logText.setJsonResult(OptUtil.substring(jsonResult));
        }
        // 读取详情 记录数据日志
        List<OptLogTableInfo> list = cycleInfo.getItems();
        List<OptLogTable> tables = null;
        // 异常不记录变更日志
        if (e == null && CollectionUtils.isNotEmpty(list)) {
            // 根据结构 转为对应的map数据 递归 先分类
            Map<Class<?>, List<OptLogTableInfo>> map = PojoUtil.groupMap(list, c -> c.getTableDetail().getClazz());
            AnalysisMiddleware mom = new AnalysisMiddleware(map);
            String logId = mom.getLogId();
            logData.setId(logId);
            logText.setLogId(logId);
            String jsonAllDiff = OptUtil.allDiffJson(mom);
            logText.setJsonAllDiff(jsonAllDiff);

            mom.setFun(t -> this.initTableInfo(logId, t));
            OptUtil.diffWithDependency(mom, main.getTie());
            if (mom.getDiff() != null) {
                logText.setJsonDiff(JsonUtil.toJson(mom.getDiff()));
            }
            tables = mom.getTables();
        }
        if (user != null) {
            UserContextHolder.setInfo(user.getId(), token, user.getTenantId());
            logData.setOptName(user.getNickName());
            logData.setDeptName(user.getDeptName());
        }
        if (CollectionUtils.isNotEmpty(tables)) {
            logTableService.saveBatch(tables);
        }
        // save
        logDataService.save(logData);
        if (!StringUtils.isAllBlank(logText.getErrorMsg(), logText.getJsonParam(), logText.getJsonAllDiff())) {
            logText.setLogId(logData.getId());
            textService.save(logText);
        }
        if (user != null) {
            UserContextHolder.clear();
        }
    }

    /**
     * 初始化表实体信息
     * @param logId 日志id
     * @param t 表变更信息
     * @return 实体信息
     */
    private OptLogTable initTableInfo (String logId, OptLogTableInfo t){
        OptLogTableDetail detail = t.getTableDetail();
        // entity
        OptLogTable logTable = new OptLogTable();
        logTable.setIdValue(t.getIdValue());
        logTable.setLogId(logId);
        logTable.setIdName(detail.getIdName());
        logTable.setForeignKey(t.getForeignId());
        logTable.setForeignKeyName(detail.getForeignIdName());
        logTable.setTabName(detail.getTableName());
        logTable.setMethodId(detail.getMethodId());
        logTable.setTableDesc(detail.getTableDesc());
        logTable.setDiffMode(t.getMode().getCode());
        logTable.setJsonDiff(JsonUtil.toJson(t.getDiff()));
        return logTable;
    }
}
