/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.opt.cache;

import cn.sinozg.applet.common.annotation.OptIgnore;
import cn.sinozg.applet.common.annotation.OptLog;
import cn.sinozg.applet.common.annotation.OptLogField;
import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.constant.BaseRedisKeys;
import cn.sinozg.applet.common.utils.PojoUtil;
import cn.sinozg.applet.common.utils.RedisUtil;
import cn.sinozg.applet.opt.config.OptLogProperties;
import cn.sinozg.applet.opt.constant.OptLogConstant;
import cn.sinozg.applet.opt.module.OptLogMain;
import cn.sinozg.applet.opt.module.OptMapperColumn;
import cn.sinozg.applet.opt.module.OptMapperTable;
import cn.sinozg.applet.opt.module.OptTieRelation;
import cn.sinozg.applet.opt.util.OptUtil;
import com.baomidou.mybatisplus.annotation.TableField;
import com.baomidou.mybatisplus.annotation.TableId;
import com.baomidou.mybatisplus.annotation.TableName;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.media.Schema;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.servlet.http.HttpServletRequest;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentSkipListMap;

/**
 * 缓存信息
 * @Description
 * @Copyright Copyright (c) 2024
 * @author xieyubin
 * @since 2024-02-27 21:37:43
 */
@Slf4j
public class OptLogCache {
    /** 依赖缓存 */
    private static final Map<Class<?>, OptTieRelation<?>> TIE_MAP = new ConcurrentHashMap<>();
    /**
     * 删除所有的缓存
     */
    public static boolean clearApiBaseInfo (){
       return RedisUtil.deleteKeys(BaseRedisKeys.OPT_LOG);
    }

    /**
     * 获取api请求的基本信息缓存
     * @param controller 控制器
     * @param method 方法
     * @param isDataLog 配置是否记录日志
     * @param request request
     * @return 基本信息
     */
    public static OptLogMain apiBaseInfo (Class<?> controller, Method method, boolean isDataLog, HttpServletRequest request) {
        String className = controller.getName();
        String methodName = method.getName();
        String key = className + BaseConstants.SPOT + methodName;
        String redisKey = String.format(BaseRedisKeys.OPT_LOG_API, key);
        OptLogMain main = RedisUtil.getCacheObject(redisKey);
        if (main != null) {
            return main;
        }
        main = new OptLogMain();
        String title = null;
        String moduleName = null;
        // 一定存在
        OptLog optLog = method.getAnnotation(OptLog.class);
        if (StringUtils.isNotBlank(optLog.module())){
            moduleName = optLog.module();
        } else {
            Tag tag = controller.getAnnotation(Tag.class);
            if (tag != null) {
                moduleName = tag.description();
            }
        }
        if (StringUtils.isNotBlank(optLog.title())){
            title = optLog.title();
        } else {
            Operation operation = method.getAnnotation(Operation.class);
            if (operation != null) {
                title = operation.summary();
            }
        }
        Class<?> tie = null;
        String requestMethod = request.getMethod();
        if (isDataLog && optLog.dataLog()) {
            tie = optLog.tie();
            // 默认取第一个
            if (tie == Void.class) {
                tie = OptUtil.firstMethodParams(method, requestMethod);
            }
            if (tie == null) {
                log.error("不记录修改日志详情，请设置日志参数的关联关系！{}.{}", className, methodName);
            }
        }
        // 模块名称
        main.setModuleName(moduleName);
        // 业务名称
        main.setTitle(title);
        // 设置方法名称
        main.setMethodName(key + "()");
        // 设置action动作
        main.setBusinessType(optLog.bizType().getCode());
        // 设置操作人类别
        main.setOptType(optLog.optType().getCode());
        // 请求信息
        String url = request.getRequestURI();
        // 设置请求全路径/IP地址/请求方式/设置token/线程名称
        main.setOptUrl(url);
        main.setRequestMethod(requestMethod);
        // 是否需要保存request，参数和值
        if (optLog.saveParams()) {
            // 获取参数的信息，传入到数据库中。
            main.setSaveParams(true);
        }
        main.setTie(tie);
        RedisUtil.setCacheObject(redisKey, main);
        return main;
    }

    /**
     * 从缓存获取到 实体信息
     * @param clazz 实体类型
     * @param properties 配置
     * @return 对象
     */
    public static OptMapperTable getCacheTableInfo(Class<?> clazz, OptLogProperties properties) {
        // 如果缓存中存在，则返回
        String redisKey = String.format(BaseRedisKeys.OPT_LOG_TABLE, clazz.getName());
        OptMapperTable tableInfo = RedisUtil.getCacheObject(redisKey);
        if (tableInfo != null) {
            return tableInfo;
        }
        tableInfo = generatorTableInfo(clazz, properties);
        RedisUtil.setCacheObject(redisKey, tableInfo);
        return tableInfo;
    }


    /**
     * 生成依赖映射关系
     * @param clazz 请求类
     * @return 映射关系
     * @param <T> 类
     */
    public static <T> OptTieRelation<T> optTie(Class<T> clazz){
        if (clazz == null) {
            return null;
        }
        if (TIE_MAP.containsKey(clazz)) {
            return PojoUtil.cast(TIE_MAP.get(clazz));
        }
        // 迭代查询关系
        OptTieRelation<T> relation = OptUtil.optTie(clazz);
        TIE_MAP.put(clazz, relation);
        return relation;
    }


    /**
     * 判定方法是否被忽略
     * @param clazz 类
     * @param methodName 方法
     * @return 是否忽略
     */
    public static boolean ignoreMethod (Class<?> clazz, String methodName){
        String redisKey = String.format(BaseRedisKeys.OPT_LOG_METHOD_IGNORE, clazz.getName());
        List<String> list = RedisUtil.getCacheObject(redisKey);
        if (list == null) {
            Method[] methods = clazz.getDeclaredMethods();
            list = new ArrayList<>();
            if (ArrayUtils.isNotEmpty(methods)) {
                for (Method method : methods) {
                    OptIgnore optIgnore = method.getAnnotation(OptIgnore.class);
                    if (optIgnore != null) {
                        list.add(method.getName());
                    }
                }
            }
            RedisUtil.setCacheObject(redisKey, list);
        }
        return CollectionUtils.containsAny(list, methodName);
    }


    /**
     * 根据实体配置，生成数据日志记录信息
     * @param clazz 实体类型
     * @param properties 配置
     * @return 对象
     */
    private static OptMapperTable generatorTableInfo(Class<?> clazz, OptLogProperties properties) {
        OptMapperTable tableInfo = tableBaseInfo(clazz, properties);
        if (!tableInfo.isRecord()) {
            return tableInfo;
        }
        // 获取列
        List<Field> fields = PojoUtil.allField(clazz);
        if (CollectionUtils.isEmpty(fields)) {
            return tableInfo;
        }
        List<OptMapperColumn> columns = new ArrayList<>();
        int index = 0;
        for (Field field : fields) {
            OptMapperColumn column = column(tableInfo, field, properties);
            if (column == null) {
                continue;
            }
            column.setSort(Math.min(index, column.getSort()));
            columns.add(column);
            index ++;
        }
        if (CollectionUtils.isNotEmpty(columns)) {
            columns.sort(Comparator.comparing(OptMapperColumn::getSort));
            // 属性名对象map
            Map<String, OptMapperColumn> propertyMap = new ConcurrentSkipListMap<>();
            for (OptMapperColumn column : columns) {
                propertyMap.put(column.getPropertyName(), column);
            }
            // 设置map集合
            tableInfo.setPropertyMap(propertyMap);
        } else {
            // 是否有字段需要记录
            tableInfo.setRecord(false);
        }
        return tableInfo;
    }

    /**
     * 获取字段信息
     * @param table 表
     * @param field 字段
     * @param properties 配置
     * @return 字段
     */
    private static OptMapperColumn column (OptMapperTable table, Field field, OptLogProperties properties){
        field.setAccessible(true);
        // 属性名称
        String propertyName = field.getName();
        // 排除字段
        if (OptLogConstant.DEFAULT_EXCLUDE_FIELDS.contains(propertyName)) {
            return null;
        }
        // 列名称
        String columnName = null;
        // 获取列描述
        String columnDesc = null;
        // 列对象
        OptMapperColumn column = new OptMapperColumn();
        column.setPropertyName(propertyName);
        column.setTypeClass(field.getType());
        // 查找id主键
        TableId tableId = field.getAnnotation(TableId.class);
        if (tableId != null) {
            columnName = tableId.value();
            table.setIdColumnName(tableId.value());
            table.setIdPropertyName(propertyName);
        }
        int sort = Integer.MAX_VALUE;
        OptLogField recordField = field.getAnnotation(OptLogField.class);
        if (recordField != null && recordField.ignore()) {
            return null;
        }
        // 获取列信息
        Schema schema = field.getAnnotation(Schema.class);
        if (schema != null) {
            columnDesc = schema.description();
        }
        if (recordField != null) {
            columnDesc = recordField.desc();
            sort = recordField.sort();
        }
        if (StringUtils.isBlank(columnName)) {
            TableField tableField = field.getAnnotation(TableField.class);
            if (tableField != null) {
                columnName = tableField.value();

            }
        }
        if (StringUtils.isBlank(columnDesc)) {
            columnDesc = columnName;
        }
        column.setColumnName(columnName);
        column.setColumnDesc(columnDesc);
        if (OptUtil.exclude(columnName, properties.getIncludeColumns(), properties.getExcludeColumns())) {
            return null;
        }
        // 添加到map集合
        column.setSort(sort);
        return column;
    }

    /**
     * 基础信息
     * @param clazz 实体类型
     * @param properties 配置
     * @return 对象
     */
    private static OptMapperTable tableBaseInfo (Class<?> clazz, OptLogProperties properties){
        // 获取TableInfo，并保存到map缓存中
        OptMapperTable tableInfo = new OptMapperTable();
        tableInfo.setType(clazz);
        String entityName = clazz.getSimpleName();
        tableInfo.setEntityName(entityName);
        // 获取实体类注解信息
        Schema classSchema = clazz.getAnnotation(Schema.class);
        TableName tn = clazz.getAnnotation(TableName.class);
        String tableName = null;
        String tableDesc = null;
        if (classSchema != null) {
            tableDesc = classSchema.description();
        }
        if (tn != null) {
            tableName = tn.value();
        }
        tableInfo.setTableName(tableName);
        tableInfo.setTableDesc(tableDesc);
        // 全局判断是否排除
        assert tableName != null;
        if (CollectionUtils.containsAny(OptLogConstant.DEFAULT_EXCLUDE_TABLE, tableName.toLowerCase()) || OptUtil.exclude(tableName, properties.getIncludeTables(), properties.getExcludeTables())) {
            tableInfo.setRecord(false);
            if (CollectionUtils.containsAny(OptLogConstant.DEFAULT_EXCLUDE_TABLE, tableName.toLowerCase())) {
                tableInfo.setLogTable(true);
            }
        }
        return tableInfo;
    }
}
