/*
 * Copyright (C) 2020-2025, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.oss.service.impl;

import cn.sinozg.applet.common.properties.OssValue;
import com.aliyun.oss.HttpMethod;
import com.aliyun.oss.OSS;
import com.aliyun.oss.model.DeleteObjectsRequest;
import com.aliyun.oss.model.DeleteObjectsResult;
import com.aliyun.oss.model.GeneratePresignedUrlRequest;
import com.aliyun.oss.model.GetObjectRequest;
import com.aliyun.oss.model.PutObjectRequest;
import com.aliyun.oss.model.PutObjectResult;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.net.URL;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.Collection;
import java.util.Date;
import java.util.List;

/**
 * @author xieyubin
 * @Description
 * @Copyright Copyright (c) 2025
 * @since 2025-04-25 18:07
 */
public class FileStoreAliServiceImpl extends FileStoreBaseService<OSS> {

    private final OSS aliClient;

    private static final Logger log = LoggerFactory.getLogger(FileStoreAliServiceImpl.class);

    public FileStoreAliServiceImpl(OssValue oss, OSS client) {
        super(oss);
        this.aliClient = client;
        setClient(client);
    }

    @Override
    public String upload(byte[] bs, String bucketName, String key, String contentType) throws Exception {
        try (ByteArrayInputStream bis = new ByteArrayInputStream(bs)){
            PutObjectRequest request = new PutObjectRequest(bucketName, key, bis);
            PutObjectResult result = aliClient.putObject(request);
            return StringUtils.lowerCase(result.getETag());
        }
    }

    @Override
    public String preSignedUrl(String key, boolean upload) throws Exception {
        GeneratePresignedUrlRequest request = new GeneratePresignedUrlRequest(largeBucket(), key);
        LocalDateTime newTime = LocalDateTime.now().plusMinutes(signedExp());
        request.setMethod(upload ? HttpMethod.PUT : HttpMethod.GET);
        request.setExpiration(Date.from(newTime.atZone(ZoneId.systemDefault()).toInstant()));
        URL url = aliClient.generatePresignedUrl(request);
        if (url != null) {
            return url.toExternalForm();
        }
        return null;
    }

    @Override
    public Collection<?> deletes(String bucketName, List<String> list) throws Exception {
        DeleteObjectsRequest params = new DeleteObjectsRequest(bucketName).withKeys(list).withEncodingType("url");
        DeleteObjectsResult result = aliClient.deleteObjects(params);
        return result.getDeletedObjects();
    }

    @Override
    public InputStream download(String bucketName, String key) throws Exception {
        GetObjectRequest request = new GetObjectRequest(bucketName, key);
        return aliClient.getObject(request).getObjectContent();
    }
}
