/*
 * Copyright (C) 2020-2025, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.oss.config;

import cn.sinozg.applet.common.properties.AppValue;
import cn.sinozg.applet.common.properties.OssValue;
import cn.sinozg.applet.oss.OssConstants;
import cn.sinozg.applet.oss.service.FileStoreService;
import cn.sinozg.applet.oss.service.impl.FileStoreS3ServiceImpl;
import jakarta.annotation.Resource;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Lazy;
import software.amazon.awssdk.auth.credentials.AwsBasicCredentials;
import software.amazon.awssdk.auth.credentials.AwsCredentialsProvider;
import software.amazon.awssdk.auth.credentials.AwsCredentialsProviderChain;
import software.amazon.awssdk.http.apache.ApacheHttpClient;
import software.amazon.awssdk.http.apache.ProxyConfiguration;
import software.amazon.awssdk.regions.Region;
import software.amazon.awssdk.services.s3.S3Client;
import software.amazon.awssdk.services.s3.S3ClientBuilder;
import software.amazon.awssdk.services.s3.presigner.S3Presigner;

import java.net.URI;

/**
 * @author xieyubin
 * @Description
 * @Copyright Copyright (c) 2025
 * @since 2025-07-23 10:48
 */
@Configuration
@ConditionalOnProperty(name = OssConstants.PREFIX, havingValue = OssConstants.S3)
public class OssAmazonConfig {

    @Resource
    private AppValue appValue;

    private static final Logger log = LoggerFactory.getLogger(OssAmazonConfig.class);

    @Lazy
    @Bean(name = OssConstants.S3 + OssConstants.BN_CLIENT, destroyMethod = OssConstants.DM_CLOSE)
    public S3Client s3Client(){
        return s3Client(appValue.getOss());
    }

    @Lazy
    @Bean(destroyMethod = OssConstants.DM_CLOSE)
    public S3Presigner s3Presigner(){
        return s3Presigner(appValue.getOss());
    }

    @Lazy
    @Bean(name = OssConstants.S3 + OssConstants.BN_SERVICE)
    public FileStoreService<S3Client> amazonStoreService(S3Client s3Client, S3Presigner s3Presigner){
        return new FileStoreS3ServiceImpl(appValue.getOss(), s3Client, s3Presigner);
    }

    /**
     * 亚马逊S3
     * @param oss 配置
     * @return 亚马逊S3
     */
    public static S3Client s3Client (OssValue oss){
        S3ClientBuilder builder = S3Client.builder().credentialsProvider(
                AwsCredentialsProviderChain.builder()
                        .addCredentialsProvider(credentialsProvider(oss))
                        .build()
        ).region(Region.of(oss.getRegion()));
        if (StringUtils.isNotBlank(oss.getProxyUrl())) {
            proxy(builder, oss);
        }
        return builder.build();
    }
    /**
     * 亚马逊S3预签名
     * @param oss 配置
     * @return 亚马逊S3预签名
     */
    public static S3Presigner s3Presigner (OssValue oss){
        return S3Presigner.builder()
                .region(Region.of(oss.getRegion()))
                .credentialsProvider(credentialsProvider(oss))
                .build();
    }

    /**
     * 亚马逊证书
     * @param oss 配置
     * @return 亚马逊证书
     */
    private static AwsCredentialsProvider credentialsProvider(OssValue oss){
        return () -> AwsBasicCredentials.create(oss.getAccessKey(), oss.getSecretKey());
    }

    /**
     * 代理
     * @param builder 对象
     * @param oss 配置
     */
    private static void proxy(S3ClientBuilder builder, OssValue oss){
        log.error("user Proxy ..... ");
        builder.httpClientBuilder(ApacheHttpClient.builder()
                .proxyConfiguration(ProxyConfiguration.builder().endpoint(URI.create(oss.getProxyUrl()))
                        .build()));
    }
}
