/*
 * Copyright (C) 2020-2025, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */
package cn.sinozg.applet.oss.service.impl;

import cn.sinozg.applet.common.properties.OssValue;
import org.apache.commons.lang3.RegExUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.awscore.presigner.PresignedRequest;
import software.amazon.awssdk.core.sync.RequestBody;
import software.amazon.awssdk.core.sync.ResponseTransformer;
import software.amazon.awssdk.services.s3.S3Client;
import software.amazon.awssdk.services.s3.model.Delete;
import software.amazon.awssdk.services.s3.model.DeleteObjectsRequest;
import software.amazon.awssdk.services.s3.model.DeleteObjectsResponse;
import software.amazon.awssdk.services.s3.model.GetObjectRequest;
import software.amazon.awssdk.services.s3.model.ObjectIdentifier;
import software.amazon.awssdk.services.s3.model.PutObjectRequest;
import software.amazon.awssdk.services.s3.model.PutObjectResponse;
import software.amazon.awssdk.services.s3.presigner.S3Presigner;
import software.amazon.awssdk.services.s3.presigner.model.GetObjectPresignRequest;
import software.amazon.awssdk.services.s3.presigner.model.PutObjectPresignRequest;

import java.io.InputStream;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * @author xieyubin
 * @Description
 * @Copyright Copyright (c) 2025
 * @since 2025-04-25 18:07
 */
public class FileStoreS3ServiceImpl extends FileStoreBaseService<S3Client> {

    private static final Logger log = LoggerFactory.getLogger(FileStoreS3ServiceImpl.class);

    private final S3Client s3Client;

    private final S3Presigner s3Presigner;
    public FileStoreS3ServiceImpl(OssValue oss, S3Client s3Client, S3Presigner s3Presigner) {
        super(oss);
        this.s3Client = s3Client;
        this.s3Presigner = s3Presigner;
        setClient(s3Client);
    }

    @Override
    public String upload(byte[] bs, String bucketName, String key, String contentType) throws Exception {
        PutObjectRequest request = PutObjectRequest.builder()
                .bucket(bucketName)
                .contentType(contentType)
                .key(key)
                .build();
        PutObjectResponse response = s3Client.putObject(request, RequestBody.fromBytes(bs));
        String md5 = response.eTag();
        return RegExUtils.replaceAll(md5, "\"", StringUtils.EMPTY);
    }

    @Override
    public String preSignedUrl(String key, boolean upload) throws Exception {
        PresignedRequest request;
        if (upload) {
            request = putUrl(key);
        } else {
            request = getUrl(key);
        }
        if (request != null) {
            return request.url().toExternalForm();
        }
        return null;
    }

    @Override
    public Collection<?> deletes(String bucketName, List<String> list) throws Exception {
        ArrayList<ObjectIdentifier> keys = new ArrayList<>();
        for (String key : list) {
            keys.add(ObjectIdentifier.builder().key(key).build());
        }
        DeleteObjectsRequest request = DeleteObjectsRequest.builder()
                .bucket(bucketName)
                .delete(Delete.builder().objects(keys).build())
                .build();
        DeleteObjectsResponse response = s3Client.deleteObjects(request);
        return response.deleted();
    }
    @Override
    public InputStream download(String bucketName, String key) throws Exception {
        GetObjectRequest request = GetObjectRequest.builder()
                .bucket(bucketName)
                .key(key)
                .build();
        return s3Client.getObject(request, ResponseTransformer.toInputStream());
    }

    private PresignedRequest getUrl (String key) {
        GetObjectRequest gutRequest = GetObjectRequest.builder()
                .bucket(largeBucket())
                .key(key)
                .build();
        GetObjectPresignRequest request = GetObjectPresignRequest.builder()
                .signatureDuration(Duration.ofMinutes(signedExp()))
                .getObjectRequest(gutRequest)
                .build();
        return s3Presigner.presignGetObject(request);
    }

    private PresignedRequest putUrl (String key) {
        PutObjectRequest putRequest = PutObjectRequest.builder()
                .bucket(largeBucket())
                .key(key)
                .build();
        PutObjectPresignRequest request = PutObjectPresignRequest.builder()
                .signatureDuration(Duration.ofMinutes(signedExp()))
                .putObjectRequest(putRequest)
                .build();
        return s3Presigner.presignPutObject(request);
    }
}
