/*
 * Copyright (C) 2020-2025, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.oss.clients;

import cn.sinozg.applet.common.properties.OssValue;
import com.qiniu.http.Client;
import com.qiniu.storage.BucketManager;
import com.qiniu.storage.Configuration;
import com.qiniu.storage.Region;
import com.qiniu.storage.UploadManager;
import com.qiniu.util.Auth;

import java.io.Closeable;
import java.io.IOException;
import java.util.Objects;

/**
 * 七牛 客户度
 * @author xieyubin
 * @Description
 * @Copyright Copyright (c) 2025
 * @since 2025-07-22 23:28
 */
public class QnClient implements Closeable {

    private final OssValue oss;

    private volatile Auth auth;
    private volatile Client client;
    private volatile Configuration configuration;
    private volatile BucketManager bucketManager;
    private volatile UploadManager uploadManager;

    private final Object lock = new Object();
    private volatile boolean closed = false;

    public QnClient(OssValue oss) {
        this.oss = Objects.requireNonNull(oss);
    }

    public Auth getAuth() {
        if (auth == null) {
            synchronized (lock) {
                if (auth == null) {
                    auth = Auth.create(oss.getAccessKey(), oss.getSecretKey());
                }
            }
        }
        return auth;
    }

    public Client getClient() {
        if (client == null) {
            synchronized (lock) {
                if (client == null) {
                    client = new Client(getConfiguration());
                }
            }
        }
        return client;
    }

    public Configuration getConfiguration() {
        if (configuration == null) {
            synchronized (lock) {
                if (configuration == null) {
                    configuration = Configuration.create(Region.createWithRegionId(oss.getRegion()));
                }
            }
        }
        return configuration;
    }

    public BucketManager getBucketManager() {
        if (bucketManager == null) {
            synchronized (lock) {
                if (bucketManager == null) {
                    bucketManager = new BucketManager(getAuth(), getConfiguration(), getClient());
                }
            }
        }
        return bucketManager;
    }

    public UploadManager getUploadManager() {
        if (uploadManager == null) {
            synchronized (lock) {
                if (uploadManager == null) {
                    uploadManager = new UploadManager(getConfiguration());
                }
            }
        }
        return uploadManager;
    }

    @Override
    public void close() throws IOException {
        synchronized (lock) {
            if (closed) {
                return;
            }
            closed = true;
            // 清空引用
            auth = null;
            configuration = null;
            bucketManager = null;
            uploadManager = null;
            client = null;
        }
    }
}
