/*
 * Copyright (C) 2020-2025, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.oss.service.impl;

import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.exception.CavException;
import cn.sinozg.applet.common.properties.OssValue;
import cn.sinozg.applet.common.utils.FileUtil;
import cn.sinozg.applet.common.utils.SnowFlake;
import cn.sinozg.applet.oss.OssConstants;
import cn.sinozg.applet.oss.model.FileUploadParams;
import cn.sinozg.applet.oss.model.FileUploadResult;
import cn.sinozg.applet.oss.service.FileStoreService;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Strings;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.multipart.MultipartFile;

import java.io.ByteArrayOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.MessageDigest;
import java.util.Collection;
import java.util.List;

/**
 * @Description 
 * @Copyright Copyright (c) 2025
 * @author xieyubin
 * @since 2025-07-20 15:19:24
 */
public abstract class FileStoreBaseService<T> implements FileStoreService<T> {

    private final OssValue oss;

    private T client;

    private static final Logger log = LoggerFactory.getLogger(FileStoreBaseService.class);

    public FileStoreBaseService(OssValue oss) {
        this.oss = oss;
    }

    public abstract String upload (byte[] bs, String bucketName, String key, String contentType) throws Exception;

    public abstract Collection<?> deletes (String bucketName, List<String> list) throws Exception;

    public abstract String preSignedUrl (String key, boolean upload) throws Exception;

    public abstract InputStream download(String bucketName, String key) throws Exception;

    @Override
    public FileUploadResult uploadOss(FileUploadParams params) {
        MultipartFile file = params.getFile();
        String orgName = file.getOriginalFilename();
        // 扩展名称
        String extName = checkFileInfo(file, orgName, params.getMaxSize(), params.getAllowedExtension());
        String fileId = SnowFlake.genId();
        String ossKey = fileId;
        if (!params.isKeyEqId()) {
            ossKey = SnowFlake.genId();
        }
        String fileName = fileId + BaseConstants.SPOT + extName.toLowerCase();
        String bucketName = params.getBucketName();
        if (StringUtils.isBlank(bucketName)) {
            bucketName = oss.getBucketName();
        }
        if (params.isShow()) {
            ossKey = ossKey + BaseConstants.SPOT + extName.toLowerCase();
        }
        if (StringUtils.isNotBlank(params.getKeyPrefix())) {
            ossKey = params.getKeyPrefix() + ossKey;
        }
        String contentType = FileUtil.fileType(file.getOriginalFilename());
        String sha256 = null;
        String md5;
        try (InputStream is = file.getInputStream();
             ByteArrayOutputStream bos = new ByteArrayOutputStream()){
            MessageDigest digest = MessageDigest.getInstance("SHA-256");
            int bytesRead;
            int cacheLength = 8192;
            byte[] buffer = new byte[cacheLength];
            while ((bytesRead = is.read(buffer))!= -1) {
                bos.write(buffer, 0, bytesRead);
                if (params.isSha()) {
                    digest.update(buffer, 0, bytesRead);
                }
            }
            md5 = upload(bos.toByteArray(), bucketName, ossKey, contentType);
            if (params.isSha()) {
                sha256 = Hex.encodeHexString(digest.digest());
            }
            if (StringUtils.isBlank(md5)) {
                throw new CavException("BIZ000100104");
            }
        } catch (Exception e) {
            log.error("上传文件至OSS错误", e);
            throw new CavException("BIZ000100104");
        }
        FileUploadResult result = new FileUploadResult(bucketName, ossKey, md5);
        result.setSha256(sha256);
        result.setSize(file.getSize());
        result.setId(fileId);
        result.setExtName(extName);
        result.setOrgName(orgName);
        result.setFileName(fileName);
        return result;
    }

    @Override
    public String largeBucket() {
        String bucketName = oss.getLargeBucket();
        if (StringUtils.isBlank(bucketName)) {
            bucketName = oss.getBucketName();
        }
        return bucketName;
    }

    @Override
    public Pair<String, String> createPreSignedUrl(String key, boolean upload) {
        key = upload ? SnowFlake.genId() : key;
        if (StringUtils.isBlank(key)) {
            throw new CavException("BIZ000100108");
        }
        try {
            String url = preSignedUrl(key, upload);
            if (StringUtils.isBlank(url)) {
                throw new CavException("BIZ000100105");
            }
            return Pair.of(key, url);
        } catch (Exception e) {
            log.error("创建预签名失败！", e);
            throw new CavException("BIZ000100105");
        }
    }

    @Override
    public void deleteFiles(String bucketName, List<String> list) {
        try {
            Collection<?> result = deletes(bucketName, list);
            if (CollectionUtils.isNotEmpty(result)) {
                log.info("删除成功数量:{}", result.size());
            }
        } catch (Exception e) {
            log.error("删除文件错误！", e);
            throw new CavException("BIZ000100106", e);
        }
    }

    @Override
    public void download(OutputStream os, String bucketName, String key) {
        try (InputStream fis = download(bucketName, key)){
            fis.transferTo(os);
        } catch (Exception e) {
            log.error("下载文件失败！", e);
            throw new CavException("BIZ000100107");
        }
    }

    /**
     * 获取文件的扩展名
     * @param orgName 原始名称
     * @param contentType contentType
     * @param allowedExtension 支持的扩展名称
     * @return 文件扩展名
     */
    @Override
    public String fileExtension(String orgName, String contentType, String[] allowedExtension) {
        if (StringUtils.isBlank(orgName) || orgName.length() > OssConstants.DEFAULT_FILE_NAME_LENGTH) {
            throw new CavException("BIZ000100101", OssConstants.DEFAULT_FILE_NAME_LENGTH);
        }
        String extension = FilenameUtils.getExtension(orgName);
        if (StringUtils.isEmpty(extension)) {
            if (StringUtils.isNotBlank(contentType)) {
                extension = StringUtils.substringAfterLast(contentType, "/");
            }
        }
        if (allowedExtension != null && !Strings.CI.equalsAny(extension, allowedExtension)) {
            throw new CavException("BIZ000100102", extension);
        }
        return StringUtils.lowerCase(extension);
    }

    @Override
    public T client() {
        return client;
    }

    protected void setClient (T client){
        this.client = client;
    }

    protected int signedExp(){
        Integer exp = oss.getSignedExp();
        if (exp == null) {
            exp = 30;
        }
        return exp;
    }

    protected String preSignedBaseUrl(){
        String baseUrl = oss.getLargeUrl();
        if (StringUtils.isBlank(baseUrl)) {
            baseUrl = oss.getUrl();
        }
        return baseUrl;
    }

    /**
     * 检查文件的大小
     * @param file 文件
     * @param orgName 原始名称
     * @param maxSize 文件最大大小
     * @param allowedExtension 支持的扩展名称
     * @return 文件扩展名
     */
    private String checkFileInfo(MultipartFile file, String orgName, long maxSize, String[] allowedExtension) {
        long size = file.getSize();
        if (size > maxSize) {
            throw new CavException("BIZ000100103", maxSize / 1024 / 1024);
        }
        return fileExtension(orgName, file.getContentType(), allowedExtension);
    }
}
