/*
 * Copyright (C) 2020-2025, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.oss.service.impl;

import cn.sinozg.applet.common.properties.OssValue;
import cn.sinozg.applet.common.utils.PojoUtil;
import io.minio.GetObjectArgs;
import io.minio.GetPresignedObjectUrlArgs;
import io.minio.MinioClient;
import io.minio.ObjectWriteResponse;
import io.minio.PutObjectArgs;
import io.minio.RemoveObjectsArgs;
import io.minio.Result;
import io.minio.http.Method;
import io.minio.messages.DeleteError;
import io.minio.messages.DeleteObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.ByteArrayInputStream;
import java.io.FilterInputStream;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * @author xieyubin
 * @Description
 * @Copyright Copyright (c) 2025
 * @since 2025-07-20 15:45
 */
public class FileStoreMinioServiceImpl extends FileStoreBaseService<MinioClient> {

    private static final Logger log = LoggerFactory.getLogger(FileStoreMinioServiceImpl.class);

    private final MinioClient minioClient;

    public FileStoreMinioServiceImpl(OssValue oss, MinioClient minioClient) {
        super(oss);
        this.minioClient = minioClient;
        setClient(minioClient);
    }

    @Override
    public String upload(byte[] bs, String bucketName, String key, String contentType) throws Exception {
        try (ByteArrayInputStream bis = new ByteArrayInputStream(bs)){
            ObjectWriteResponse response = minioClient.putObject(PutObjectArgs.builder()
                    .bucket(bucketName)
                    .object(key)
                    .contentType(contentType)
                    .stream(bis, bis.available(), -1)
                    .build());
            return response.etag();
        }
    }

    @Override
    public String preSignedUrl(String key, boolean upload) throws Exception {
        GetPresignedObjectUrlArgs params = GetPresignedObjectUrlArgs.builder()
                .bucket(largeBucket())
                .expiry(signedExp(), TimeUnit.MINUTES)
                .method(upload ? Method.PUT : Method.GET)
                .object(key)
                .build();
        return minioClient.getPresignedObjectUrl(params);
    }

    @Override
    public Collection<?> deletes(String bucketName, List<String> list) throws Exception {
        List<DeleteObject> dos = PojoUtil.toList(list, DeleteObject::new);
        Iterable<Result<DeleteError>> results = minioClient.removeObjects(RemoveObjectsArgs
                .builder()
                .bucket(bucketName)
                .objects(dos)
                .build());
        List<Boolean> result = new ArrayList<>();
        for (Result<DeleteError> r : results) {
            try {
                r.get();
                result.add(true);
            } catch (Exception e) {
                DeleteError de = r.get();
                log.error("minio 文件删除失败{}", de.message());
            }
        }
        return result;
    }

    @Override
    public FilterInputStream download(String bucketName, String key) throws Exception{
        GetObjectArgs params = GetObjectArgs.builder()
                .bucket(bucketName)
                .object(key)
                .build();
        return minioClient.getObject(params);
    }
}
