package cn.sinozg.applet.common.utils;

import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.core.model.ExportData;
import cn.sinozg.applet.common.core.model.ImportExcelResult;
import cn.sinozg.applet.common.excel.EasyExcelReadListener;
import cn.sinozg.applet.common.exception.CavException;
import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.ExcelWriter;
import com.alibaba.excel.write.builder.ExcelWriterBuilder;
import com.alibaba.excel.write.builder.ExcelWriterSheetBuilder;
import com.alibaba.excel.write.metadata.WriteSheet;
import com.alibaba.excel.write.metadata.fill.FillConfig;
import com.alibaba.excel.write.metadata.style.WriteCellStyle;
import com.alibaba.excel.write.metadata.style.WriteFont;
import com.alibaba.excel.write.style.HorizontalCellStyleStrategy;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.poi.ss.usermodel.HorizontalAlignment;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.ClassPathResource;
import org.springframework.web.multipart.MultipartFile;

import jakarta.servlet.http.HttpServletResponse;
import java.io.BufferedOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.function.Consumer;
import java.util.function.Function;

/**
 * 导入导出封装 依赖 EasyExcel
 * 简单的封装 复杂的需要自己去实现
 *
 * @Author: xyb
 * @Description:
 * @Date: 2023-05-01 上午 12:50
 **/
public class ExcelUtil {

    private static final Logger log = LoggerFactory.getLogger(ExcelUtil.class);

    /**
     * 通过实体对象，直接读取导入的excel
     * <pre> {@code
     * List<UserImportDto> list = ExcelUtil.importExcel(multipartFile, UserImportVo.class);
     * }</pre>
     *
     * @param file  上传的文件
     * @param clazz 导入实体类型
     * @param <T>   导入实体类型
     * @return 获取到的结果
     */
    public static <T> List<T> importExcel(MultipartFile file, Class<T> clazz) {
        return importExcel(file, f -> f, clazz).getData();
    }

    /**
     * 导入excel<p>
     * 导入的实体bean 需要转化成db存储对象
     * <pre> {@code
     *         ImportExcelResult<UserImportDto> result = ExcelUtil.importExcel(multipartFile, f -> {
     *             UserImportDto dto = PojoUtil.copyBean(f, UserImportDto.class);
     *             dto.setNonce(SnowFlake.genId());
     *             return dto;
     *         } , UserImportVo.class);
     * }</pre>
     *
     * @param file     上传的文件
     * @param function 导入的实体bean 转为db的entity
     * @param clazz    导入实体类型
     * @param <T>      导入实体类型
     * @param <U>      db实体类型
     * @return 处理后的集合
     */
    public static <T, U> ImportExcelResult<U> importExcel(MultipartFile file, Function<T, U> function, Class<T> clazz) {
        ImportExcelResult<U> result = new ImportExcelResult<>();
        // 此处将前端传过来的流变为文件 不然会导致识别错误
        if (!FileUtil.judgeFile(file, BaseConstants.XLS, BaseConstants.XLSX)) {
            throw new CavException("BIZ000100027");
        }
        EasyExcelReadListener<T, U> listener = new EasyExcelReadListener<>(function, result);
        try (InputStream is = file.getInputStream()) {
            EasyExcel.read(is, listener).head(clazz).sheet().doRead();
        } catch (Exception e) {
            log.error("读取excel错误，{}， {}", file.getOriginalFilename(), e.getMessage());
            throw new CavException("BIZ000100026");
        }
        return result;
    }

    /**
     * 用模板导出复制的 excel
     * <pre> {@code
     * ExcelUtil.exportWithTemp(response, "temp/air_port.xlsx", list, other);
     * }</pre>
     *
     * @param response   HttpServletResponse
     * @param tempPath   模板名称
     * @param exportData 导出的数据集合
     * @param otherInfo  要导出的单个对象数据 可以为null
     * @param <T>        数据集合
     */
    public static <T> void exportWithTemp(HttpServletResponse response, String tempPath, List<T> exportData, Object otherInfo) {
        String fileName = StringUtils.substringAfterLast(tempPath, "/");
        setResponse(response, fileName);
        ClassPathResource classPathResource = new ClassPathResource(tempPath);
        try (InputStream inputStream = classPathResource.getInputStream();
             OutputStream os = response.getOutputStream();
             BufferedOutputStream bos = new BufferedOutputStream(os);
             ExcelWriter excelWriter = EasyExcel.write(bos).withTemplate(inputStream).build()
        ) {
            // 设置输出流和模板信息
            WriteSheet writeSheet = EasyExcel.writerSheet().build();
            // 开启自动换行,自动换行表示每次写入一条list数据是都会重新生成一行空行,此选项默认是关闭的,需要提前设置为true
            FillConfig fillConfig = FillConfig.builder().forceNewRow(Boolean.TRUE).build();
            excelWriter.fill(exportData, fillConfig, writeSheet);
            if (otherInfo != null) {
                // 写其他的单行记录
                excelWriter.fill(otherInfo, writeSheet);
            }
            excelWriter.finish();
        } catch (Exception e) {
            log.error("根据模板excel文件导出excel失败:", e);
            throw new CavException("BIZ000100024");
        }
    }

    /**
     * 导出excel 下载
     * 使用bean的方式直接下载导出
     * <pre> {@code
     * ExcelUtil.exportSingleFile(response, null, list, "sheet_1");
     * }</pre>
     *
     * @param response   response
     * @param fileName   文件名称
     * @param exportData 导出数据
     * @param sheetName  sheet 名称
     * @param <T>        实体类型
     */
    public static <T> void exportSingleFile(HttpServletResponse response, String fileName, List<T> exportData, String sheetName) {
        if (CollectionUtils.isEmpty(exportData)) {
            throw new CavException("BIZ000100025");
        }
        exportFile(fileName, PojoUtil.arrayToList(exportData), response, sheetName);
    }

    /**
     * 导出excel 下载
     * 使用bean的方式直接下载导出 自定义样式 和表头
     * @param response   response
     * @param fileName   文件名称
     * @param exportData 导出数据
     * @param writeConsumer 样式等定义
     */
    public static <T> void exportSingleFile(HttpServletResponse response, String fileName, ExportData<T> exportData, Consumer<ExcelWriterBuilder> writeConsumer) {
        if (CollectionUtils.isEmpty(exportData.getSheetData())) {
            throw new CavException("BIZ000100025");
        }
        List<ExportData<T>> list = Collections.singletonList(exportData);
        exportFile(fileName, list, writeConsumer, response);
    }

    /**
     * excel文件导出(可以包含多个sheet页)，固定表头(通过实体指定属性的方式)
     *
     * @param fileName   导出文件名
     * @param exportData 需要导出数据
     * @param response   response
     * @param sheetNames sheet页的名称，为空则默认以:sheet + 数字规则命名
     * @param <T>        类型
     */
    public static <T> void exportFile(String fileName, List<List<T>> exportData, HttpServletResponse response, String... sheetNames) {
        List<ExportData<T>> list = new ArrayList<>();
        for (int itemIndex = 0; itemIndex < exportData.size(); itemIndex++) {
            ExportData<T> exportDataItem = new ExportData<>();
            if (ArrayUtils.isNotEmpty(sheetNames) && sheetNames.length >= itemIndex) {
                exportDataItem.setSheetName(sheetNames[itemIndex]);
            }
            exportDataItem.setSheetData(exportData.get(itemIndex));
            list.add(exportDataItem);
        }
        exportFile(fileName, list, null, response);
    }

    /**
     * excel文件导出(可以包含多个sheet页)，可以指定表头，样式等自定义方式
     *
     * @param fileName   导出文件名
     * @param exportData 需要导出数据
     * @param response   response
     * @param <T>        类型
     */
    public static <T> void exportFile(String fileName, List<ExportData<T>> exportData, Consumer<ExcelWriterBuilder> writeConsumer, HttpServletResponse response) {
        if (Objects.isNull(response) || CollectionUtils.isEmpty(exportData)) {
            log.error("ExcelUtil exportFile required param can't be empty");
            throw new CavException("BIZ000100025");
        }
        setResponse(response, fileName);
        try (OutputStream os = response.getOutputStream();
             BufferedOutputStream bos = new BufferedOutputStream(os)) {
            ExcelWriterBuilder writerBuild = EasyExcel.write(bos);
            if (writeConsumer != null) {
                writeConsumer.accept(writerBuild);
            } else {
                writerBuild.registerWriteHandler(getExportDefaultStyle());
            }
            try (ExcelWriter writer = writerBuild.build()) {
                // 设置导出的表格样式
                ExportData<T> ed;
                for (int itemIndex = 0; itemIndex < exportData.size(); itemIndex++) {
                    // sheet页的数据
                    ed = exportData.get(itemIndex);
                    List<T> list = ed.getSheetData();
                    String sheetName = "sheet" + (itemIndex + 1);
                    if (StringUtils.isNotBlank(ed.getSheetName())) {
                        sheetName = ed.getSheetName();
                    }
                    ExcelWriterSheetBuilder sheet = EasyExcel.writerSheet(itemIndex, sheetName);
                    Consumer<ExcelWriterSheetBuilder> sheetConsumer = ed.getSheetConsumer();
                    if (sheetConsumer != null) {
                        sheetConsumer.accept(sheet);
                    } else {
                        // 表头数据
                        Class<?> clazz = list.get(0).getClass();
                        sheet.head(clazz);
                    }
                    writer.write(list, sheet.build());
                }
            }
        } catch (Exception e) {
            log.error("ExcelUtil exportFile in error:", e);
            throw new CavException("BIZ000100024");
        }
    }

    /**
     * 下载excel 模板
     *
     * @param response HttpServletResponse
     * @param tempName 模板路径
     */
    public static void downloadExcelTemp(HttpServletResponse response, String tempName) {
        ClassPathResource classPathResource = new ClassPathResource(tempName);
        try (InputStream inputStream = classPathResource.getInputStream()) {
            DowUtil.download(response, tempName, IOUtils.toByteArray(inputStream), null);
        } catch (Exception e) {
            log.error("下载excel 模版错误！", e);
        }
    }

    /**
     * 设置 response
     *
     * @param response response
     * @param fileName 文件名
     */
    public static void setResponse(HttpServletResponse response, String fileName) {
        if (StringUtils.isBlank(fileName)) {
            fileName = SnowFlake.genId() + BaseConstants.XLSX;
        } else {
            String[] arrays = StringUtils.split(fileName, BaseConstants.SPOT);
            fileName = String.format("%s(%s).%s", arrays[0], DateUtil.formatDateTime(LocalDateTime.now(), "MMddHHmmssSSS"), arrays[1]);
        }
        DowUtil.setResponse(response, fileName, null);
    }

    /**
     * 配置默认的excel表格样式对象
     *
     * @return HorizontalCellStyleStrategy
     */
    private static HorizontalCellStyleStrategy getExportDefaultStyle() {
        WriteCellStyle headWriteCellStyle = new WriteCellStyle();
        // 设置头字体
        WriteFont headWriteFont = new WriteFont();
        headWriteFont.setBold(true);
        headWriteCellStyle.setWriteFont(headWriteFont);
        // 设置头居中
        headWriteCellStyle.setHorizontalAlignment(HorizontalAlignment.CENTER);
        // 内容策略
        WriteCellStyle contentWriteCellStyle = new WriteCellStyle();
        // 设置 水平居中
        contentWriteCellStyle.setHorizontalAlignment(HorizontalAlignment.CENTER);
        return new HorizontalCellStyleStrategy(headWriteCellStyle, contentWriteCellStyle);
    }
}