/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.excel;

import ai.chat2db.excel.converters.Converter;
import ai.chat2db.excel.enums.CellDataTypeEnum;
import ai.chat2db.excel.metadata.GlobalConfiguration;
import ai.chat2db.excel.metadata.data.WriteCellData;
import ai.chat2db.excel.metadata.property.ExcelContentProperty;
import cn.sinozg.applet.common.annotation.DictTrans;
import cn.sinozg.applet.common.annotation.PicUrl;
import cn.sinozg.applet.common.utils.DictTransUtil;
import cn.sinozg.applet.common.utils.StringUtil;

import java.lang.reflect.Field;

/**
* 导出转移 使用方式为
 * <pre> {@code
 * @ExcelProperty(converter = ExcelWriteConverter.class)
 * }</pre>
* @Author: xyb
* @Description:
* @Date: 2023-04-30 上午 01:18
**/
public class ExcelWriteConverter implements Converter<String> {

    @Override
    public Class<?> supportJavaTypeKey() {
        return String.class;
    }

    @Override
    public CellDataTypeEnum supportExcelTypeKey() {
        return CellDataTypeEnum.STRING;
    }


    /**
     * 读取转化
     * @param value               Java Data.NotNull.
     * @param contentProperty     Content property.NotNull
     * @param globalConfiguration Global configuration.NotNull.
     * @return 要写入excel的值
     */
    @Override
    public WriteCellData<?> convertToExcelData(String value, ExcelContentProperty contentProperty, GlobalConfiguration globalConfiguration) {
        String targetValue = value;
        Field field = contentProperty.getField();
        // 标准数据转换
        DictTrans dict = field.getAnnotation(DictTrans.class);
        if (dict != null) {
            targetValue = DictTransUtil.dictTrans(value, dict);
        }
        // 图片
        PicUrl pic = field.getAnnotation(PicUrl.class);
        if (pic != null) {
            targetValue = StringUtil.showUrl(value);
        }
        assert targetValue != null;
        return new WriteCellData<>(targetValue);
    }
}
