/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.common.excel;

import ai.chat2db.excel.context.AnalysisContext;
import ai.chat2db.excel.event.AnalysisEventListener;
import ai.chat2db.excel.exception.ExcelDataConvertException;
import cn.sinozg.applet.common.core.model.ImportExcelResult;
import cn.sinozg.applet.common.utils.PojoUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.function.Function;

/**
 * 读取excel  监听<p/>
 * 流式处理，减少内存占用
 * @Author: xyb
 * @Description:
 * @Date: 2023-04-30 下午 01:36
 **/
public class EasyExcelReadListener<T, U> extends AnalysisEventListener<T> {

    /** 数据转化 可以用来处理空白行 */
    private final Function<T, U> function;

    /** 处理结果 */
    private final ImportExcelResult<U> result;

    private final Logger log = LoggerFactory.getLogger(EasyExcelReadListener.class);

    public EasyExcelReadListener(Function<T, U> function, ImportExcelResult<U> result){
        this.function = function;
        this.result = result;
    }

    @Override
    public void onException(Exception exception, AnalysisContext context) throws Exception {
        if (exception instanceof ExcelDataConvertException e) {
            String detail = String.format("第%d行，第%d列解析异常，数据为：%s", e.getRowIndex(), e.getColumnIndex(), e.getCellData().getStringValue());
            log.error(detail, e);
            PojoUtil.setBeanList(result, detail, ImportExcelResult::getErrorInfo, ImportExcelResult::setErrorInfo);
        } else {
            log.error("导入excel 错误！", exception);
        }
    }

    @Override
    public void invoke(T data, AnalysisContext context) {
        U u = function.apply(data);
        if (u != null) {
            PojoUtil.setBeanList(result, function.apply(data), ImportExcelResult::getData, ImportExcelResult::setData);
        }
    }

    @Override
    public void doAfterAllAnalysed(AnalysisContext context) {
    }
}