/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.sa.plugin;

import cn.dev33.satoken.dao.SaTokenDao;
import cn.dev33.satoken.dao.auto.SaTokenDaoBySessionFollowObject;
import cn.dev33.satoken.util.SaFoxUtil;
import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.utils.RedisUtil;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.TimeUnit;
import org.apache.commons.lang3.StringUtils;

/**
 * Sa-Token 持久层实现 [Redis存储、Jackson序列化]
 * 不实现SaTokenDaoByObjectFollowString有点多此一举，直接实现 SaTokenDaoBySessionFollowObject即可
 * @Author: xyb
 * @Description: 
 * @Date: 2022-11-14 下午 10:19
 **/
class SaRedisJackson implements SaTokenDaoBySessionFollowObject {

    /**
     * 获取Value，如无返空
     * @param key key值
     * @return 结果
     */
    @Override
    public String get(String key) {
        return RedisUtil.getCacheObject(key);
    }

    /**
     * 写入Value，并设定存活时间 (单位: 秒)
     * @param key key值
     * @param value value值
     * @param timeout 过期时间
     */
    @Override
    public void set(String key, String value, long timeout) {
        this.setObject(key, value, timeout);
    }

    /**
     * 修修改指定key-value键值对 (过期时间不变)
     * @param key key值
     * @param value value值
     */
    @Override
    public void update(String key, String value) {
        this.updateObject(key, value);
    }

    /**
     * 删除Value
     * @param key key值
     */
    @Override
    public void delete(String key) {
        RedisUtil.deleteObject(key);
    }

    /**
     * 获取Value的剩余存活时间 (单位: 秒)
     * @param key key值
     * @return 过期时间
     */
    @Override
    public long getTimeout(String key) {
        return RedisUtil.objExpireTime(key);
    }

    /**
     * 修改Value的剩余存活时间 (单位: 秒)
     * @param key key值
     * @param timeout 过期时间
     */
    @Override
    public void updateTimeout(String key, long timeout) {
        this.updateObjectTimeout(key, timeout);
    }

    /**
     * 获取Object，如无返空
     * @param key key值
     * @return 结果
     */
    @Override
    public Object getObject(String key) {
        return RedisUtil.getCacheObject(key);
    }

    @Override
    public <T> T getObject(String key, Class<T> classType) {
        return RedisUtil.getCacheObject(key);
    }

    /**
     * 写入Object，并设定存活时间 (单位: 秒)
     * @param key key值
     * @param object 对象的值
     * @param timeout 过期时间
     */
    @Override
    public void setObject(String key, Object object, long timeout) {
        if (timeout == 0 || timeout <= SaTokenDao.NOT_VALUE_EXPIRE) {
            return;
        }
        RedisUtil.setCacheObject(key, object, timeout);
    }

    /**
     * 更新Object (过期时间不变)
     * @param key key值
     * @param object 对象的值
     */
    @Override
    public void updateObject(String key, Object object) {
        long expire = getObjectTimeout(key);
        // -2 = 无此键
        if (expire == SaTokenDao.NOT_VALUE_EXPIRE) {
            return;
        }
        this.setObject(key, object, expire);
    }

    /**
     * 删除Object
     * @param key key值
     */
    @Override
    public void deleteObject(String key) {
        RedisUtil.deleteObject(key);
    }


    /**
     * 获取Object的剩余存活时间 (单位: 秒)
     * @param key key值
     * @return 存活时间
     */
    @Override
    public long getObjectTimeout(String key) {
        return RedisUtil.objExpireTime(key);
    }

    /**
     * 修改Object的剩余存活时间 (单位: 秒)
     * @param key key值
     * @param timeout 过期时间
     */
    @Override
    public void updateObjectTimeout(String key, long timeout) {
        // 判断是否想要设置为永久
        if (timeout == SaTokenDao.NEVER_EXPIRE) {
            long expire = getTimeout(key);
            // 如果其已经被设置为永久，则不作任何处理
            // 如果尚未被设置为永久，那么再次set一次
            if (expire != SaTokenDao.NEVER_EXPIRE) {
                this.setObject(key, this.getObject(key), timeout);
            }
            return;
        }
        RedisUtil.expire(key, timeout, TimeUnit.SECONDS);
    }

    /**
     * 搜索数据
     * @param prefix 前缀
     * @param keyword 关键词
     * @param start 开始
     * @param size 数量
     * @param sortType 排序类型
     * @return 搜索结果
     */
    @Override
    public List<String> searchData(String prefix, String keyword, int start, int size, boolean sortType) {
        String pattern = prefix + BaseConstants.ALL;
        if (StringUtils.isNotEmpty(keyword)) {
            pattern += keyword + BaseConstants.ALL;
        }
        Collection<String> keys = RedisUtil.keys(pattern);
        List<String> list = new ArrayList<>(keys);
        return SaFoxUtil.searchList(list, start, size, sortType);
    }
}
