/*
 * Copyright (C) 2020-2025, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.sa;

import cn.dev33.satoken.exception.StopMatchException;
import cn.dev33.satoken.stp.StpUtil;
import cn.dev33.satoken.strategy.SaAnnotationStrategy;
import cn.sinozg.applet.common.utils.PojoWebUtil;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import java.lang.reflect.Method;
import org.jetbrains.annotations.NotNull;
import org.springframework.web.method.HandlerMethod;
import org.springframework.web.servlet.HandlerInterceptor;

/**
 * Sa-Token 综合拦截器，提供注解鉴权和路由拦截鉴权能力<br/>
 * 重写，保证可以拿到具体异常 由全局异常处理器去处理所有的异常
 * @Description
 * @Copyright Copyright (c) 2025
 * @author xyb
 * @since 2025-03-29 14:29:50
*/
public class SaInterceptor implements HandlerInterceptor {

	/**
	 * 是否打开注解鉴权，配置为 true 时注解鉴权才会生效，配置为 false 时，即使写了注解也不会进行鉴权
	 */
	public boolean isAnnotation = true;


	/**
	 * 创建一个 Sa-Token 综合拦截器，默认带有注解鉴权能力 
	 */
	public SaInterceptor() {
	}


	/**
	 * 设置是否打开注解鉴权：配置为 true 时注解鉴权才会生效，配置为 false 时，即使写了注解也不会进行鉴权
	 * @param isAnnotation /
	 * @return 对象自身
	 */
	public SaInterceptor isAnnotation(boolean isAnnotation) {
		this.isAnnotation = isAnnotation;
		return this;
	}

	/**
	 * 每次请求之前触发的方法 
	 */
	@Override
	public boolean preHandle(@NotNull HttpServletRequest request, @NotNull HttpServletResponse response, @NotNull Object handler) throws Exception {
		try {
			// 这里必须确保 handler 是 HandlerMethod 类型时，才能进行注解鉴权
			if(isAnnotation && handler instanceof HandlerMethod hm) {
				// 获取此请求对应的 Method 处理函数 
				Method method = hm.getMethod();
				// 执行注解鉴权
				SaAnnotationStrategy.instance.checkMethodAnnotation.accept(method);
			}
			// Auth 路由拦截鉴权校验
			StpUtil.checkLogin();
		} catch (StopMatchException ignored) {
		} catch (Exception e) {
			PojoWebUtil.resolveException(request, response, e);
			return false;
		}
		// 通过验证 
		return true;
	}

}
