/*
 * Copyright (C) 2020-2024, Xie YuBin
 * The GNU Free Documentation License covers this file. The original version
 * of this license can be found at http://www.gnu.org/licenses/gfdl.html.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Free Documentation License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Free Documentation License for more details.
 *
 * You should have received a copy of the GNU Free Documentation License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

package cn.sinozg.applet.sa.service.impl;

import cn.sinozg.applet.common.config.RedisConfig;
import cn.sinozg.applet.common.properties.AppValue;
import cn.sinozg.applet.common.service.AuthStorageService;
import cn.sinozg.applet.common.utils.PojoUtil;
import jakarta.annotation.Resource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.data.redis.connection.RedisConfiguration;
import org.springframework.data.redis.connection.RedisSentinelConfiguration;
import org.springframework.data.redis.connection.RedisStandaloneConfiguration;
import org.springframework.data.redis.connection.lettuce.LettuceConnectionFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;

import java.util.Collection;
import java.util.concurrent.TimeUnit;

/**
 * @author xieyubin
 * @Description
 * @Copyright Copyright (c) 2025
 * @since 2025-07-18 00:14
 */
@Service
@ConditionalOnProperty(name = "app.login.token-database")
public class AuthStorageCustomizeServiceImpl implements AuthStorageService {

    @Resource
    private AppValue appValue;

    @Resource
    private LettuceConnectionFactory defaultFactory;

    private RedisTemplate<String, Object> redisTemplate;

    private static final Logger log = LoggerFactory.getLogger(AuthStorageCustomizeServiceImpl.class);

    @Override
    public <T> T getCacheObject(String key) {
        Object o = redis().opsForValue().get(key);
        return PojoUtil.cast(o);
    }

    @Override
    public void setCacheObject(String key, Object value, long timeout) {
        if (timeout > 0) {
            redis().opsForValue().set(key, value, timeout, TimeUnit.SECONDS);
        } else {
            redis().opsForValue().set(key, value);
        }
    }

    @Override
    public void deleteObject(String key) {
        redis().delete(key);
    }

    @Override
    public long objExpireTime(String key) {
        Long l = redis().getExpire(key);
        return l == null ? 0 : l;
    }

    @Override
    public void expire(String key, long timeout, TimeUnit unit) {
        redis().expire(key, timeout, unit);
    }

    @Override
    public Collection<String> keys(String pattern) {
        return redis().keys(pattern);
    }

    private RedisTemplate<String, Object> redis(){
        if(redisTemplate == null){
            redisTemplate = createTemplate();
        }
        return redisTemplate;
    }

    private RedisTemplate<String, Object> createTemplate() {
        RedisSentinelConfiguration sentinelConfiguration = defaultFactory.getSentinelConfiguration();
        RedisConfiguration sc;
        int database = appValue.getLogin().getTokenDatabase();
        if (sentinelConfiguration != null) {
            RedisSentinelConfiguration sentinelConfig = new RedisSentinelConfiguration();
            sentinelConfig.setDatabase(database);
            sentinelConfig.setUsername(sentinelConfiguration.getUsername());
            sentinelConfig.setPassword(sentinelConfiguration.getPassword());
            sentinelConfig.setSentinels(sentinelConfiguration.getSentinels());
            sc = sentinelConfig;
        } else {
            RedisStandaloneConfiguration standaloneConfiguration = defaultFactory.getStandaloneConfiguration();
            RedisStandaloneConfiguration standaloneConfig = new RedisStandaloneConfiguration();
            standaloneConfig.setHostName(standaloneConfiguration.getHostName());
            standaloneConfig.setPort(standaloneConfiguration.getPort());
            standaloneConfig.setPassword(standaloneConfiguration.getPassword());
            standaloneConfig.setUsername(standaloneConfiguration.getUsername());
            standaloneConfig.setDatabase(database);
            sc = standaloneConfig;
        }
        LettuceConnectionFactory factory = new LettuceConnectionFactory(sc, defaultFactory.getClientConfiguration());
        factory.afterPropertiesSet();
        return RedisConfig.setRedisConfig(factory);
    }
}
