package cn.sinozg.applet.common.utils;

import cn.sinozg.applet.common.config.SystemConfig;
import cn.sinozg.applet.common.constant.HeaderConstants;
import cn.sinozg.applet.common.core.model.AesRsaEncrypt;
import cn.sinozg.applet.common.exception.CavException;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.crypto.bcrypt.BCryptPasswordEncoder;

import javax.crypto.Cipher;
import javax.crypto.KeyGenerator;
import javax.crypto.SecretKey;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import javax.servlet.http.HttpServletRequest;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.security.AlgorithmParameters;
import java.security.Key;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyPairGenerator;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.Arrays;

/**
 * 加密工具类，获取随机数。<br/>
 *
 * @Author: xyb
 * @Description:
 * @Date: 2023-04-24 下午 09:52
 **/
public class CypherUtil {

    public static final String AES_ALGORITHM = "AES";

    public static final String RSA_ALGORITHM = "RSA";
    private static final String PKCS7 = "AES/CBC/PKCS7Padding";
    private static final BCryptPasswordEncoder ENCODER = new BCryptPasswordEncoder();

    private static final Logger log = LoggerFactory.getLogger(CypherUtil.class);


    private CypherUtil() {
    }

    /**
     * 签名判断 参数排序拼接后 签名再与原始的签名比较
     *
     * @param sha2      是否为 sha2
     * @param signature 签名字符串
     * @param params    要签名的数据
     * @return 是否与签名一致
     */
    public static boolean signature(boolean sha2, String signature, String... params) {
        if (StringUtils.isBlank(signature)) {
            log.error("非法请求参数，有部分参数为空 : {}", signature);
            return false;
        }
        String source = signMessage(sha2, params);
        return signature.equals(source);
    }

    /**
     * 签名
     * @param sha2 sha2      是否为 sha2
     * @param params 要签名的数据
     * @return 签名值
     */
    public static String signMessage(boolean sha2, String... params) {
        if (StringUtils.isAnyEmpty(params)) {
            log.error("非法请求参数，有部分参数为空 : {}", Arrays.toString(params));
            return null;
        }
        Arrays.sort(params);
        String source = StringUtils.join(params);
        source = sha2 ? DigestUtils.sha256Hex(source) : DigestUtils.sha1Hex(source);
        return source;
    }

    /**
     * 解密微信用户信息
     *
     * @param encryptedData 加密数据
     * @param sessionKey    解密秘钥
     * @param iv            初始向量
     * @return 解密以后的数据
     */
    public static String wxDecrypt(String encryptedData, String sessionKey, String iv) {
        byte[] eds = Base64.decodeBase64(encryptedData);
        byte[] sks = Base64.decodeBase64(sessionKey);
        byte[] ivs = Base64.decodeBase64(iv);
        int base = 16;
        boolean flag = sks.length % base != 0;
        if (flag) {
            int groups = sks.length / base + 1;
            byte[] temp = new byte[groups * base];
            Arrays.fill(temp, (byte) 0);
            System.arraycopy(sks, 0, temp, 0, sks.length);
            sks = temp;
        }
        byte[] original = doFinalAes(PKCS7, true, eds, sks, ivs);
        String json = "";
        if (ArrayUtils.isNotEmpty(original)) {
            json = new String(original, StandardCharsets.UTF_8);
        }
        return json;
    }



    /**
     * 加密信息
     *
     * @param rawPassword 原始密码
     * @return 加密后的密码
     */
    public static String encoder(String rawPassword) {
        return ENCODER.encode(rawPassword);
    }

    /**
     * 密码匹配
     *
     * @param rawPassword     原始密码
     * @param encodedPassword 加密后的密码
     * @return 是否匹配
     */
    public static boolean matches(CharSequence rawPassword, String encodedPassword) {
        return ENCODER.matches(rawPassword, encodedPassword);
    }

    /**
     * md5 加密返回大写
     *
     * @param input 要加密的数据
     * @return md5
     */
    public static String md5Upper(String input) {
        return md5(input).toUpperCase();
    }

    /**
     * md5 加密
     *
     * @param input 要加密的数据
     * @return md5
     */
    public static String md5(String input) {
        if (StringUtils.isBlank(input)) {
            return StringUtils.EMPTY;
        }
        return DigestUtils.md5Hex(input);
    }

    /**
     * 解密前端传过来的加密数据
     * <p>先用RSA私钥（java）解密前端传过来的aes加密信息，得到前端生成的随机aes key
     * <p>再用 aes key解密得到数据
     *
     * @param content       加密数据
     * @param rsaPrivateKey RSA私钥
     * @param aesKey        aes 加密后的数据
     * @return 解密后的数据
     */
    public static byte[] decryptJson(String content, String rsaPrivateKey, String aesKey) throws Exception {
        // 得到RSA私钥 对象
        Key privateKey = rsaKey(rsaPrivateKey, false);
        // 用私钥解密，得到aesKey
        byte[] aesKeyBytes = decrypt(RSA_ALGORITHM, privateKey, aesKey);
        // 将Base64编码后的AES秘钥转换成 SecretKey对象
        SecretKey key = new SecretKeySpec(aesKeyBytes, AES_ALGORITHM);
        // 用AES秘钥解密实际的内容
        return decrypt(AES_ALGORITHM, key, content);
    }

    /**
     * 加密数据返回到前端
     *
     * @param request request
     * @param data    请求参数
     * @return 请求参数或者加密后的数据
     */
    public static Object encrypt(HttpServletRequest request, Object data) throws IOException {
        if (SystemConfig.APP.getSign().isRsaEnable()) {
            String jsPublicKey = StringUtils.trimToEmpty(request.getHeader(HeaderConstants.X_PUB_KEY));
            String json = JsonUtil.toJson(data);
            if (json == null) {
                throw new RuntimeException("出参转json错误！");
            }
            try {
                return encrypt(json, jsPublicKey);
            } catch (Exception e) {
                throw new IOException(e);
            }
        }
        return data;
    }

    /**
     * 随机生成16位的 aes私钥
     * <p>用aes 私钥加密json得到加密信息
     * <p>用前端传过来的 RSA 公钥加密 aes私钥 得到aes私钥加密信息
     *
     * @param content      加密信息
     * @param rsaPublicKey 前端传过来的RSA公钥
     * @return 加密信息
     */
    private static AesRsaEncrypt encrypt(String content, String rsaPublicKey) throws Exception {
        // 随机生成16位的 aes私钥
        String aesKey = getAesKey();
        // 用AES秘钥加密实际的json内容
        String json = encrypt(AES_ALGORITHM, new SecretKeySpec(aesKey.getBytes(), AES_ALGORITHM), content);
        // 加密aes私钥 base64格式的key字符串转Key对象
        Key publicKey = rsaKey(rsaPublicKey, true);
        String encryptAesKey = encrypt(RSA_ALGORITHM, publicKey, aesKey);
        AesRsaEncrypt encrypt = new AesRsaEncrypt();
        encrypt.setData(json);
        encrypt.setAesKey(encryptAesKey);
        return encrypt;
    }

    /**
     * 通过字符串等到 key对象
     *
     * @param key      key的字符串
     * @param isPublic 是否为公钥
     * @return 对象
     * @throws Exception 异常
     */
    public static Key rsaKey(String key, boolean isPublic) throws Exception {
        Key rsaKey;
        byte[] bytes = Base64.decodeBase64(key);
        KeyFactory factory = KeyFactory.getInstance(RSA_ALGORITHM);
        if (isPublic) {
            rsaKey = factory.generatePublic(new X509EncodedKeySpec(bytes));
        } else {
            rsaKey = factory.generatePrivate(new PKCS8EncodedKeySpec(bytes));
        }
        return rsaKey;
    }

    /**
     * 解密数据
     *
     * @param transformation 加密方式
     * @param key            密钥
     * @param input          数据
     * @return 返回操作后的结果
     */
    public static byte[] decrypt(String transformation, Key key, String input) {
        byte[] params = Base64.decodeBase64(input);
        if (AES_ALGORITHM.equals(transformation)) {
            return doFinalAes(transformation, true, params, key);
        } else {
            return doFinalRsa(transformation, Cipher.DECRYPT_MODE, key, params);
        }
    }

    /**
     * 加密数据
     *
     * @param transformation 加密方式
     * @param key            密钥
     * @param input          数据
     * @return 返回操作后的结果
     */
    public static String encrypt(String transformation, Key key, String input) {
        byte[] params = input.getBytes();
        byte[] encryptBytes;
        if (AES_ALGORITHM.equals(transformation)) {
            encryptBytes =  doFinalAes(transformation, false, params, key);
        } else {
            encryptBytes = doFinalRsa(transformation, Cipher.ENCRYPT_MODE, key, params);
        }
        return Base64.encodeBase64String(encryptBytes);
    }

    /**
     * rsa 加密或者解密数据 分段
     *
     * @param transformation 加密方式
     * @param opMode         加密、解密
     * @param key            密钥
     * @param input          数据
     * @return 返回操作后的结果
     */
    private static byte[] doFinalRsa(String transformation, int opMode, Key key, byte[] input) {
        int inputLen = input.length;
        int offSet = 0;
        byte[] cache;
        int i = 0;
        // RSA最大加密明文大小 117 RSA最大解密密文大小 128
        int maxLength = Cipher.ENCRYPT_MODE == opMode ? 117 : 128;
        try (ByteArrayOutputStream out = new ByteArrayOutputStream()) {
            Cipher cipher = Cipher.getInstance(transformation);
            cipher.init(opMode, key);
            while (inputLen - offSet > 0) {
                if (inputLen - offSet > maxLength) {
                    cache = cipher.doFinal(input, offSet, maxLength);
                } else {
                    cache = cipher.doFinal(input, offSet, inputLen - offSet);
                }
                out.write(cache, 0, cache.length);
                i++;
                offSet = i * maxLength;
            }
            return out.toByteArray();
        } catch (Exception e) {
            log.error("RSA 签名错误！", e);
            throw new CavException("BIZ000100031");
        }
    }

    /**
     * aes 签名
     * @param transformation 签名方法
     * @param decrypt 加密或者解密
     * @param input 文本信息
     * @param aesKey 密钥
     * @return 加解密后的字节信息
     */
    public static byte[] doFinalAes(String transformation, boolean decrypt, byte[] input, Key aesKey){
        return doFinalAes(transformation, decrypt, input, null, null, aesKey);
    }

    /**
     * aes 签名
     * @param transformation 签名方法
     * @param decrypt 加密或者解密
     * @param input 文本信息
     * @param aesKey 密钥
     * @param ivKey 密钥偏移量
     * @return 加解密后的字节信息
     */
    public static byte[] doFinalAes(String transformation, boolean decrypt, byte[] input, byte[] aesKey, byte[] ivKey){
        return doFinalAes(transformation, decrypt, input, aesKey, ivKey, null);
    }

    /**
     * aes 签名
     * @param transformation 签名方法
     * @param decrypt 加密或者解密
     * @param input 文本信息
     * @param aesKey 密钥
     * @param ivKey 偏移量
     * @param aesPriKey 密钥
     * @return 加解密后的字节信息
     */
    private static byte[] doFinalAes(String transformation, boolean decrypt, byte[] input, byte[] aesKey, byte[] ivKey, Key aesPriKey){
        boolean flag = input == null || (aesKey == null && aesPriKey == null);
        if (flag) {
            throw new CavException("BIZ000100030");
        }
        try {
            Cipher cipher = Cipher.getInstance(transformation);
            if (aesPriKey != null) {
                cipher.init(decrypt ? Cipher.DECRYPT_MODE : Cipher.ENCRYPT_MODE, aesPriKey);
            } else {
                SecretKeySpec keySpec = new SecretKeySpec(aesKey, AES_ALGORITHM);
                if (ivKey == null) {
                    ivKey = Arrays.copyOfRange(aesKey, 0, 16);
                }
                AlgorithmParameters params = AlgorithmParameters.getInstance(AES_ALGORITHM);
                params.init(new IvParameterSpec(ivKey));
                cipher.init(decrypt ? Cipher.DECRYPT_MODE : Cipher.ENCRYPT_MODE, keySpec);
            }
            return cipher.doFinal(input);
        } catch (Exception e) {
            log.error("AES 签名错误！", e);
            throw new CavException("BIZ000100029");
        }
    }

    /**
     * 获取随机的 aes密钥 必须是 8的倍数
     *
     * @return 密钥
     */
    private static String getAesKey() throws Exception {
        KeyGenerator kg = KeyGenerator.getInstance(AES_ALGORITHM);
        kg.init(128);
        SecretKey key = kg.generateKey();
        return Base64.encodeBase64String(key.getEncoded());
    }


    /**
     * 生成秘钥对
     *
     * @return rsa密钥对
     * @throws Exception 异常
     */
    public static KeyPair getKeyPair() throws Exception {
        KeyPairGenerator keyPairGenerator = KeyPairGenerator.getInstance(RSA_ALGORITHM);
        keyPairGenerator.initialize(1024);
        return keyPairGenerator.generateKeyPair();
    }
}
