package cn.sinozg.applet.common.utils;

import org.apache.commons.lang3.ArrayUtils;
import org.jetbrains.annotations.NotNull;
import org.springframework.aop.framework.AopContext;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.BeanFactoryPostProcessor;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.stereotype.Component;

import java.util.Map;

/**
* spring工具类 方便在非spring管理环境中获取bean
* @Author: xyb
* @Description: 
* @Date: 2022-11-14 下午 10:05
**/
@Component
public final class SpringUtil implements BeanFactoryPostProcessor, ApplicationContextAware {
    /**
     * Spring应用上下文环境
     */
    private static ConfigurableListableBeanFactory beanFactory;

    private static ApplicationContext context;

    @Override
    public void postProcessBeanFactory(@NotNull ConfigurableListableBeanFactory beanFactory) throws BeansException {
        SpringUtil.beanFactory = beanFactory;
    }

    @Override
    public void setApplicationContext(@NotNull ApplicationContext applicationContext) throws BeansException {
        SpringUtil.context = applicationContext;
    }

    /**
     * 获取对象
     *
     * @param name bean名称
     * @return Object 一个以所给名字注册的bean的实例
     * @throws BeansException Exception
     */
    public static <T> T getBean(String name) throws BeansException {
        if (beanFactory.containsBean(name)) {
            return PojoUtil.cast(beanFactory.getBean(name));
        }
        return null;
    }

    /**
     * 获取类型为requiredType的对象
     *
     * @param clazz bean类型
     * @return 对象
     * @throws BeansException Exception
     */
    public static <T> T getBean(Class<T> clazz) throws BeansException {
        return beanFactory.getBean(clazz);
    }

    /**
     * 如果BeanFactory包含一个与所给名称匹配的bean定义，则返回true
     *
     * @param name bean名称
     * @return boolean
     */
    public static boolean containsBean(String name) {
        return beanFactory.containsBean(name);
    }

    /**
     * 判断以给定名字注册的bean定义是一个singleton还是一个prototype。 如果与给定名字相应的bean定义没有被找到，将会抛出一个异常（NoSuchBeanDefinitionException）
     *
     * @param name bean名称
     * @return boolean
     * @throws NoSuchBeanDefinitionException Exception
     */
    public static boolean isSingleton(String name) throws NoSuchBeanDefinitionException {
        return beanFactory.isSingleton(name);
    }

    /**
     * @param name bean名称
     * @return Class 注册对象的类型
     * @throws NoSuchBeanDefinitionException Exception
     */
    public static Class<?> getType(String name) throws NoSuchBeanDefinitionException {
        return beanFactory.getType(name);
    }

    /**
     * 如果给定的bean名字在bean定义中有别名，则返回这些别名
     *
     * @param name bean名称
     * @return 结果
     * @throws NoSuchBeanDefinitionException Exception
     */
    public static String[] getAliases(String name) throws NoSuchBeanDefinitionException {
        return beanFactory.getAliases(name);
    }

    /**
     * 获取aop代理对象
     *
     * @param invoker invoker
     * @return 对象
     */
    public static <T> T getAopProxy(T invoker) {
        return PojoUtil.cast(AopContext.currentProxy());
    }

    /**
     * 获取当前的环境配置，无配置返回null
     *
     * @return 当前的环境配置
     */
    public static String[] getActiveProfiles() {
        return context.getEnvironment().getActiveProfiles();
    }

    /**
     * 获取当前的环境配置，当有多个环境配置时，只获取第一个
     *
     * @return 当前的环境配置
     */
    public static String getActiveProfile() {
        final String[] activeProfiles = getActiveProfiles();
        return ArrayUtils.isNotEmpty(activeProfiles) ? activeProfiles[0] : null;
    }

    /**
     * 获取bean
     * @param clazz 类型
     * @return 实例
     * @param <T> 类型
     */
    public static <T> Map<String, T> beansOfType (Class<T> clazz){
        return context.getBeansOfType(clazz);
    }

    /**
     * 获取bean
     * @param name 名称
     * @param clazz 类型
     * @return 实例
     * @param <T> 类型
     */
    public static <T> T beanOfNmTp(String name, Class<T> clazz){
        return context.getBean(name, clazz);
    }

    /**
     * 动态注册bean
     * @param beanName bean的名称
     * @param clazz 对象类型
     * @param args 参数
     * @return bean
     * @param <T> 对象
     */
    public static <T> T registerBean(String beanName, Class<T> clazz, Object... args) {
        if (containsBean(beanName)) {
            T t = getBean(beanName);
            if (t.getClass().isAssignableFrom(clazz)) {
                return t;
            }
            throw new RuntimeException("重复的bean " + beanName);
        }
        BeanDefinitionBuilder builder = BeanDefinitionBuilder.genericBeanDefinition(clazz);
        if (ArrayUtils.isNotEmpty(args)) {
            for (Object arg : args) {
                builder.addConstructorArgValue(arg);
            }
        }
        BeanDefinition definition = builder.getRawBeanDefinition();
        ConfigurableApplicationContext applicationContext = (ConfigurableApplicationContext) context;
        BeanDefinitionRegistry registry = (BeanDefinitionRegistry) applicationContext.getBeanFactory();
        registry.registerBeanDefinition(beanName, definition);
        return getBean(beanName);
    }
}
