package cn.sinozg.applet.common.utils;

import cn.sinozg.applet.common.config.SystemConfig;
import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.constant.BaseRedisKeys;
import cn.sinozg.applet.common.core.model.DictDataVo;
import cn.sinozg.applet.common.enums.DictType;
import cn.sinozg.applet.common.holder.UserContextHolder;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;

import java.time.Duration;
import java.util.Collection;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.function.Function;

/**
* 字典工具类
* @Author: xyb
* @Description:
* @Date: 2022-11-14 下午 10:03
**/
public class DictUtil {

    private DictUtil(){
    }
    /**
     * 设置字典缓存
     *
     * @param data       参数键 字典数据列表
     */
    public static void setDictCache(Map<String, List<DictDataVo>> data) {
        RedisUtil.setCacheObjectBatch(data, dictKeyFun(), Duration.ofMinutes(30));
    }

    /**
     * 根据字典类型和字典值获取字典标签
     *
     * @param dictType  字典类型
     * @param dictValue 字典值
     * @return 字典标签
     */
    public static String getDictLabel(DictType dictType, String dictValue) {
        return getDictLabel(dictType, dictValue, BaseConstants.COMMA);
    }

    /**
     * 根据字典类型和字典标签获取字典值
     *
     * @param dictType  字典类型
     * @param dictLabel 字典标签
     * @return 字典值
     */
    public static String getDictValue(DictType dictType, String dictLabel) {
        return getDictValue(dictType, dictLabel, BaseConstants.COMMA);
    }

    /**
     * 根据字典类型和字典值获取字典标签
     *
     * @param dictType  字典类型
     * @param dictValue 字典值
     * @param separator 分隔符
     * @return 字典标签
     */
    public static String getDictLabel(DictType dictType, String dictValue, String separator) {
        return tran(dictType, dictValue, separator, true);
    }

    /**
     * 根据字典类型和字典标签获取字典值
     *
     * @param dictType  字典类型
     * @param dictLabel 字典标签
     * @param separator 分隔符
     * @return 字典值
     */
    public static String getDictValue(DictType dictType, String dictLabel, String separator) {
        return tran(dictType, dictLabel, separator, false);
    }

    /**
     * 数据字典转义
     * @param dictType 类型
     * @param text 文本
     * @param separator 分隔符
     * @param value 是否为值
     * @return 转义值
     */
    private static String tran (DictType dictType, String text, String separator, boolean value){
        separator = StringUtils.trimToEmpty(separator);
        List<DictDataVo> list = getDictCache(dictType);
        String[] texts = StringUtils.split(text, separator);
        if (CollectionUtils.isEmpty(list) || ArrayUtils.isEmpty(texts)) {
            return null;
        }
        Locale locale = SystemConfig.APP.getLogin().isI18n() ? UserContextHolder.getLocale() : null;
        // key和label 的键值对
        Map<String, String> map = PojoUtil.toMap(list, d -> value ? d.getDictValue() : dictLabel(d, locale), d -> !value ? d.getDictValue() : dictLabel(d, locale));
        StringBuilder names = new StringBuilder();
        for (String t : texts) {
            String v = map.get(t);
            if (StringUtils.isNotBlank(v)) {
                names.append(v).append(separator);
            }
        }
        // 截取掉末尾
        return StringUtils.stripEnd(names.toString(), separator);
    }
    /**
     * 清空字典缓存
     */
    public static void clearDictCache() {
        Collection<String> keys = RedisUtil.keys(getCacheKey(BaseConstants.ALL));
        RedisUtil.deleteObject(keys);
    }

    /**
     * 获取到label
     * @param dictData 数据
     * @return label
     */
    private static String dictLabel (DictDataVo dictData, Locale locale){
        if (locale == null) {
            return dictData.getDictLabel();
        }
        return I18nUtil.getDictName(dictData, locale);
    }

    /**
     * 获取字典缓存
     *
     * @param key 参数键
     * @return 字典数据列表
     */
    public static List<DictDataVo> getDictCache(DictType key) {
        Object cacheObj = RedisUtil.getCacheObject(getCacheKey(key.getCode()));
        return PojoUtil.cast(cacheObj);
    }
    /**
     * 设置cache key
     *
     * @param dictType 参数键
     * @return 缓存键key
     */
    public static String getCacheKey(String dictType) {
        return dictKeyFun().apply(dictType);
    }

    /**
     * redis key
     * @return key的function
     */
    public static Function<String, String> dictKeyFun (){
        return k -> BaseRedisKeys.SYS_DICT_KEY + k;
    }

}
