package cn.sinozg.applet.common.utils;


import cn.sinozg.applet.common.config.SystemConfig;
import cn.sinozg.applet.common.constant.BaseConstants;
import cn.sinozg.applet.common.constant.BaseRedisKeys;
import cn.sinozg.applet.common.core.model.UserAgentInfo;
import com.blueconic.browscap.Capabilities;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpHeaders;

import javax.servlet.http.HttpServletRequest;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.charset.Charset;
import java.time.Duration;
import java.util.Map;

/**
* 操作系统相关判断
* @Author: xyb
* @Description: 
* @Date: 2022-11-14 下午 10:04
**/
public class OsUtil {
    private static final Logger log = LoggerFactory.getLogger(OsUtil.class);
    /** IP地址查询 */
    public static final String IP_URL = "http://whois.pconline.com.cn/ipJson.jsp?ip=%s&json=true";

    /** 未知地址 */
    private static final String UNKNOWN = Capabilities.UNKNOWN_BROWSCAP_VALUE;

    /** 本地ip */
    private static final String LOCAL_HOST = "127.0.0.1";

    private OsUtil(){
    }
    /**
     * 判断操作系统
     * @return 是否为windows
     */
    public static boolean isWindows() {
        String osName = System.getProperties().getProperty("os.name");
        return osName.toUpperCase().contains("WINDOWS");
    }

    /**
     * 通过网络方式获取到ip的地址
     * <p>先从缓存获取，再从网络获取 缓存7天
     * @param ip ip地址
     * @return 实际物理地址
     */
    public static String realAddressByIp (String ip) {
        String ipKey = String.format(BaseRedisKeys.IP_LOCATION, StringUtils.replace(ip, BaseConstants.SPOT, BaseConstants.UNDERLINE));
        String location = RedisUtil.getCacheObject(ipKey);
        if (StringUtils.isBlank(location)) {
            location = getRealAddressByIp(ip);
            RedisUtil.setCacheObject(ipKey, location, Duration.ofDays(7));
        }
        return location;
    }

    /**
     * 通过网络方式获取到ip的地址
     * @param ip ip地址
     * @return 实际物理地址
     */
    public static String getRealAddressByIp(String ip) {
        // 内网不查询
        if (internalIp(ip)) {
            return "内网IP";
        }
        try {
            byte[] bs = HttpUtil.getBytes(String.format(IP_URL, ip));
            if (bs == null) {
                log.error("获取地理位置异常 {}", ip);
                return UNKNOWN;
            }
            String rspStr = new String(bs, Charset.forName(BaseConstants.GBK));
            Map<String, String> map = PojoUtil.cast(JsonUtil.toPojo(rspStr, Map.class));
            String region = MapUtils.getString(map, "pro");
            String city =  MapUtils.getString(map, "city");
            String location;
            if (StringUtils.isAllBlank(region, city)) {
                location = MapUtils.getString(map, "addr");
            } else {
                location = String.format("%s %s", region, city);
            }
            return StringUtils.trim(location);
        } catch (Exception e) {
            log.error("获取地理位置异常 {}", ip);
        }
        return UNKNOWN;
    }

    /**
     * 获取ip地址
     * @param request http request
     * @return ip地址
     */
    public static String getIpAddr(HttpServletRequest request) {
        String unknown = Capabilities.UNKNOWN_BROWSCAP_VALUE;
        if (request == null) {
            return unknown;
        }
        String[] headers = { "x-forwarded-for", "Proxy-Client-IP", "WL-Proxy-Client-IP", "HTTP_Client_IP", "HTTP_X_FORWARDED_FOR" };
        String ip = null;
        int max = headers.length + 1;
        for (int i = 0; i < max; i++) {
            if (StringUtils.isEmpty(ip) || unknown.equalsIgnoreCase(ip)) {
                if (i == headers.length) {
                    ip = request.getRemoteAddr();
                } else {
                    ip = request.getHeader(headers[i]);
                }
            } else {
                break;
            }
        }
        // 使用代理，则获取第一个IP地址
        int maxLen = 15;
        if (StringUtils.isNotEmpty(ip) && ip.length() > maxLen) {
            ip = StringUtils.substringBefore(ip, BaseConstants.COMMA);
        }
        String[] ips = {LOCAL_HOST, "0:0:0:0:0:0:0:1"};
        if (StringUtils.equalsAny(ip, ips)) {
            ip = getHostIp();
        }
        return ip;
    }

    /**
     * 判断是否为内网ip
     * @param ip ip地址
     * @return  是否为内网ip
     */
    public static boolean internalIp(String ip) {
        byte[] addr = textToNumericFormatV4(ip);
        return internalIp(addr) || LOCAL_HOST.equals(ip);
    }

    /**
     * 判断是否为内网ip
     * <p>A类  10.0.0.0-10.255.255.255
     * <p>B类  172.16.0.0-172.31.255.255
     * <p>C类  192.168.0.0-192.168.255.255
     * @param addr 地址
     * @return 是否为内网ip
     */
    private static boolean internalIp(byte[] addr) {
        int internalLength = 2;
        if (ArrayUtils.isEmpty(addr) || addr.length < internalLength) {
            return true;
        }
        final byte b0 = addr[0];
        final byte b1 = addr[1];
        // 10.x.x.x/8
        final byte section01 = 0x0A;
        // 172.16.x.x/12
        final byte section02 = (byte) 0xAC;
        final byte section03 = (byte) 0x10;
        final byte section04 = (byte) 0x1F;
        // 192.168.x.x/16
        final byte section05 = (byte) 0xC0;
        final byte section06 = (byte) 0xA8;
        switch (b0) {
            case section01:
                return true;
            case section02:
                if (b1 >= section03 && b1 <= section04) {
                    return true;
                }
            case section05:
                if (b1 == section06) {
                    return true;
                }
            default:
                return false;
        }
    }


    /**
     * 获取ip地址
     * @return ip地址
     */
    public static String getHostIp() {
        try {
            return InetAddress.getLocalHost().getHostAddress();
        } catch (UnknownHostException e) {
            log.error("获取ip地址失败:{}", e.getMessage());
        }
        return LOCAL_HOST;
    }

    /**
     * 获取客户端名字
     * @return 客户端名称
     */
    public static String getHostName() {
        try {
            return InetAddress.getLocalHost().getHostName();
        } catch (UnknownHostException e) {
            log.error("获取host name失败:{}", e.getMessage());
        }
        return Capabilities.UNKNOWN_BROWSCAP_VALUE;
    }

    /**
     * 获取到浏览器和操作系统信息
     * @param request request
     * @return 浏览器和操作系统信息
     */
    public static UserAgentInfo userAgent (HttpServletRequest request){
        UserAgentInfo agent = new UserAgentInfo();
        String userAgent = request.getHeader(HttpHeaders.USER_AGENT);
        Capabilities capabilities = SystemConfig.PARSER.parse(userAgent);
        agent.setBrowser(formatUserAgent(capabilities.getBrowser(), capabilities.getBrowserMajorVersion()));
        agent.setPlatform(formatUserAgent(capabilities.getPlatform(), capabilities.getPlatformVersion()));
        return agent;
    }

    /**
     * 格式化 浏览器或者操作系统的信息
     * @param name 名称
     * @param version 版本
     * @return 格式化后信息
     */
    private static String formatUserAgent(String name, String version){
        String[] searchStrings = {"0", Capabilities.UNKNOWN_BROWSCAP_VALUE};
        if (StringUtils.equalsAny(version, searchStrings)) {
            return name;
        }
        return String.format("%s(%s)", name, version);
    }

    /**
     * Converts IPv4 address in its textual presentation form
     * into its numeric binary form.
     * @param src a String representing an IPv4 address in standard format
     * @return ip a byte array representing the IPv4 numeric address
     */
    public static byte[] textToNumericFormatV4(String src) {
        byte[] res = new byte[4];
        long tmpValue = 0;
        int currByte = 0;
        boolean newOctet = true;
        int len = src.length();
        int ipLen = 15;
        if (len == 0 || len > ipLen) {
            return null;
        }
        for (int i = 0; i < len; i++) {
            char c = src.charAt(i);
            if (c == '.') {
                if (newOctet || tmpValue < 0 || tmpValue > 0xff || currByte == 3) {
                    return null;
                }
                res[currByte++] = (byte) (tmpValue & 0xff);
                tmpValue = 0;
                newOctet = true;
            } else {
                int digit = Character.digit(c, 10);
                if (digit < 0) {
                    return null;
                }
                tmpValue *= 10;
                tmpValue += digit;
                newOctet = false;
            }
        }
        int min = 4, max = 8;
        if (newOctet || tmpValue < 0 || tmpValue >= (1L << ((min - currByte) * max))) {
            return null;
        }
        switch (currByte) {
            case 0:
                res[0] = (byte) ((tmpValue >> 24) & 0xff);
                break;
            case 1:
                res[1] = (byte) ((tmpValue >> 16) & 0xff);
                break;
            case 2:
                res[2] = (byte) ((tmpValue >>  8) & 0xff);
                break;
            case 3:
                res[3] = (byte) ((tmpValue) & 0xff);
                break;
            default:
        }
        return res;
    }
}
