package cn.sinozg.applet.common.utils;

import cn.sinozg.applet.common.constant.BaseConstants;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.connection.RedisStringCommands;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.ZSetOperations;
import org.springframework.data.redis.core.types.Expiration;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.data.redis.support.atomic.RedisAtomicLong;

import java.time.Duration;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.function.Supplier;

/**
* redis 工具类
* @Author: xyb
* @Description:
* @Date: 2022-11-14 下午 10:05
**/
public class RedisUtil {
    protected static final RedisTemplate<String, Object> REDIS_TEMPLATE;

    protected static final RedisTemplate<String, String> STRING_REDIS_TEMPLATE;

    static {
        REDIS_TEMPLATE = SpringUtil.getBean("objRedisTemplate");
        STRING_REDIS_TEMPLATE = SpringUtil.getBean("stringRedisTemplate");
    }
    /**
     * 缓存基本的对象，Integer、String、实体类等
     *
     * @param key   缓存的键值
     * @param value 缓存的值
     */
    public static <T> void setCacheObject(final String key, final T value) {
        REDIS_TEMPLATE.opsForValue().set(key, value);
    }

    /**
     * 缓存基本的对象，Integer、String、实体类等
     *
     * @param key      缓存的键值
     * @param value    缓存的值
     * @param timeout  时间
     * @param timeUnit 时间颗粒度
     */
    public static <T> void setCacheObject(final String key, final T value, final long timeout, final TimeUnit timeUnit) {
        if (timeout > 0) {
            REDIS_TEMPLATE.opsForValue().set(key, value, timeout, timeUnit);
        } else {
            REDIS_TEMPLATE.opsForValue().set(key, value);
        }
    }

    /**
     * 设置缓存
     * @param key key
     * @param value 值
     * @param timeout 有效时间
     * @param <T> value的类型
     */
    public static <T> void setCacheObject(final String key, final T value, final long timeout) {
        setCacheObject(key, value, timeout, TimeUnit.SECONDS);
    }

    /**
     *
     * @param key key
     * @param value 值
     * @param duration 有效时间
     * @param <T> value的类型
     */
    public static <T> void setCacheObject(final String key, final T value, final Duration duration) {
        setCacheObject(key, value, duration.getSeconds(), TimeUnit.SECONDS);
    }

    /**
     * 通过管道模式写入数据
     * @param data 数据
     * @param fun 处理key值
     * @param duration 过期时间
     * @param <T> 数据类型
     */
    public static <T> void setCacheObjectBatch(final Map<String, T> data, Function<String, String> fun, final Duration duration){
        RedisSerializer<String> keySerializer = REDIS_TEMPLATE.getStringSerializer();
        RedisSerializer<T> serializer = PojoUtil.cast (REDIS_TEMPLATE.getValueSerializer());
        REDIS_TEMPLATE.executePipelined((RedisCallback<T>) connection -> {
            data.forEach((k, v) -> connection.set(Objects.requireNonNull(keySerializer.serialize(fun == null ? k : fun.apply(k))), Objects.requireNonNull(serializer.serialize(v)), Expiration.from(duration), RedisStringCommands.SetOption.UPSERT));
            return null;
        }, serializer);
    }
    /**
     * 设置值 如果不存在就设置
     * <p>并且key存在返回false，key不存在就返回ture
     * @param key key
     * @param value value
     * @param duration 有效期
     * @return key是否存在
     * @param <T> 类型
     */
    public static <T> boolean setIfAbsentObject (final String key, final T value, final Duration duration){
        return Boolean.TRUE.equals(REDIS_TEMPLATE.opsForValue().setIfAbsent(key, value, duration.getSeconds(), TimeUnit.SECONDS));
    }


    /**
     * 设置有效时间
     *
     * @param key     Redis键
     * @param timeout 超时时间
     * @return true=设置成功；false=设置失败
     */
    public static boolean expire(final String key, final long timeout) {
        return expire(key, timeout, TimeUnit.SECONDS);
    }


    /**
     * 设置有效时间
     *
     * @param key     Redis键
     * @param timeout 超时时间
     * @param unit    时间单位
     * @return true=设置成功；false=设置失败
     */
    public static boolean expire(final String key, final long timeout, final TimeUnit unit) {
        return Boolean.TRUE.equals(REDIS_TEMPLATE.expire(key, timeout, unit));
    }

    /**
     * 判断是否有效
     * @param key redis的key
     * @return 有效期
     */
    public static Long objExpireTime(final String key) {
        return REDIS_TEMPLATE.getExpire(key);
    }


    /**
     * 获得缓存的基本对象。
     *
     * @param key 缓存键值
     * @return 缓存键值对应的数据
     */

    public static <T> T getCacheObject(final String key) {
        Object o =  REDIS_TEMPLATE.opsForValue().get(key);
        return PojoUtil.cast(o);
    }

    /**
     * 管道模式获取到数据
     * @param keyList key集合
     * @return 所有的value
     * @param <T> 返回类型
     */
    public static <T> List<T> cacheByPipelined(List<String> keyList) {
        RedisSerializer<String> keySerializer = REDIS_TEMPLATE.getStringSerializer();
        RedisSerializer<T> serializer = PojoUtil.cast(REDIS_TEMPLATE.getValueSerializer());
        List<Object> list = REDIS_TEMPLATE.executePipelined((RedisConnection connection) -> {
            keyList.forEach(k -> connection.get(Objects.requireNonNull(keySerializer.serialize(k))));
            return null;
        }, serializer);
        return PojoUtil.cast(list);
    }

    /**
     * 删除单个对象
     *
     * @param key redis的key
     */
    public static boolean deleteObject(final String key) {
        return Boolean.TRUE.equals(REDIS_TEMPLATE.delete(key));
    }

    /**
     * 根据key 批量删除
     * @param key key
     * @return 是否成功
     */
    public static boolean deleteKeys (final String key){
        String pattern = key;
        if (!StringUtils.endsWith(key, BaseConstants.ALL)) {
            pattern = key + BaseConstants.ALL;
        }
        Collection<String> keys = REDIS_TEMPLATE.keys(pattern);
        if (CollectionUtils.isNotEmpty(keys)) {
            return deleteObject(keys) > 0;
        }
        return true;
    }

    /**
     * 删除集合对象
     *
     * @param collection 多个对象
     * @return 删除是否成功
     */
    public static long deleteObject(final Collection<String> collection) {
        Long l = REDIS_TEMPLATE.delete(collection);
        if (l != null) {
            return l;
        }
        return 0;
    }

    /**
     * 获取自增长的数值
     * @param key key值
     * @param duration 过期时间 秒为单位
     * @return 返回当前的值
     */
    public static long getIncr (String key, final Duration duration) {
        return getIncr(key, duration, () -> 0L);
    }
    /**
     * 获取自增长的数值
     * @param key key值
     * @param duration 过期时间 秒为单位
     * @param supplier 初始化值
     * @return 返回当前的值
     */
    public static long getIncr (String key, final Duration duration, Supplier<Long> supplier){
        synchronized (key.intern()) {
            RedisAtomicLong ai = new RedisAtomicLong(key, Objects.requireNonNull(REDIS_TEMPLATE.getConnectionFactory()));
            long count = ai.getAndIncrement();
            // 重新开始初始化
            if (duration != null) {
                ai.expire(duration.getSeconds(), TimeUnit.SECONDS);
            }
            // 重新开始初始化
            if (count == 0) {
                ai.set(supplier.get() + 1);
                count = ai.getAndIncrement();
            }
            return count;
        }
    }

    /**
     * 设置自增长的值
     * @param key redis key
     * @param value 值
     * @return 返回当前值
     */
    public static long setIncr(String key, long value){
        synchronized (key.intern()) {
            RedisAtomicLong ai = new RedisAtomicLong(key, Objects.requireNonNull(REDIS_TEMPLATE.getConnectionFactory()));
            ai.set(value);
            return ai.getAndIncrement();
        }
    }

    /**
     * 缓存Map
     *
     * @param key key值
     * @param dataMap 数据map
     */
    public static <T> void setCacheMap(final String key, final Map<String, T> dataMap) {
        if (dataMap != null) {
            REDIS_TEMPLATE.opsForHash().putAll(key, dataMap);
        }
    }

    /**
     * 获得缓存的Map
     *
     * @param key key值
     * @return 对应的map
     */
    public static <T> Map<String, T> getCacheMap(final String key) {
        Map<Object, Object> map = REDIS_TEMPLATE.opsForHash().entries(key);
        return PojoUtil.cast(map);
    }

    /**
     * 获得缓存的基本对象列表
     *
     * @param pattern 字符串前缀
     * @return 对象列表
     */
    public static Collection<String> keys(final String pattern) {
        return REDIS_TEMPLATE.keys(pattern);
    }


    /**
     * 添加一个元素, zset与set最大的区别就是每个元素都有一个score，因此有个排序的辅助功能;
     *
     * @param key key值
     * @param value value值
     * @param score 权重
     */
    public static Boolean setCacheSet(String key, String value, double score) {
        return REDIS_TEMPLATE.opsForZSet().add(key, value, score);
    }

    /**
     * 删除元素 zrem
     *
     * @param key key值
     * @param start 开始
     * @param end 结束
     */
    public static Long removeRangeCacheSet(String key, long start, long end) {
        return REDIS_TEMPLATE.opsForZSet().removeRange(key, start, end);
    }

    /**
     * 删除元素 zrem
     *
     * @param key key值
     * @param value value值
     */
    public static Long removeCacheSet(String key, String value) {
        return REDIS_TEMPLATE.opsForZSet().remove(key, value);
    }

    /**
     * 查询集合中指定顺序的值和score，0, -1 表示获取全部的集合内容
     *
     * @param key key值
     * @param start 开始
     * @param end 结束
     * @return 结果集
     */
    public static Set<ZSetOperations.TypedTuple<String>> rangeWithScore(String key, int start, int end) {
        return STRING_REDIS_TEMPLATE.opsForZSet().rangeWithScores(key, start, end);
    }


    /**
     * 根据score的值，来获取满足条件的集合
     *
     * @param key key值
     * @param min 最小值
     * @param max 最大值
     * @return 结果集
     */
    public static Set<ZSetOperations.TypedTuple<String>> sortRangeWithScoresCacheSet(String key, double min, double max) {
        return STRING_REDIS_TEMPLATE.opsForZSet().rangeByScoreWithScores(key, min, max);
    }
}
